#include <QFormLayout>
#include <QLineEdit>
#include <QPushButton>
#include <QScrollArea>
#include <QLabel>
#include <QImageReader>
#include <QStringBuilder>
#include <QMatrix>
#include <QFile>
#include <QFileInfo>

#include "viewmessagecontents.h"

ViewMessageContents::ViewMessageContents(const QUrl &url, QWidget *parent)
    : QScrollArea(parent)
    , m_parent(parent)
    , m_photoUrl(url)
{
    setAttribute(Qt::WA_Maemo5StackedWindow);
    setWindowFlags(windowFlags() | Qt::Window);

    m_message = new QLineEdit(this);
    m_message->setMinimumWidth(625);
    m_caption = new QLineEdit(this);

    m_channelOptions = new QComboBox(this);
    m_channelList = Mouho::instance().channels();
    QString defaultChannel = Mouho::instance().config().defaultChannel();
    for (int i = 0; i < m_channelList->count(); i++) {
        m_channelOptions->addItem(" " % m_channelList->at(i)->name);
        if (m_channelList->at(i)->name == defaultChannel) {
            m_channelOptions->setCurrentIndex(i);
        }
    }

    m_sizeOptions = new QComboBox(this);

    QPushButton *submit = new QPushButton(this);
    connect(submit, SIGNAL(clicked()), this, SLOT(sendMessage()));
    submit->setText("Submit");

    QLabel *example = new QLabel(this);
    example->setText("10:25 < mouhotin> <you> here is message 1\n10:25 < mouhotin> <you> message 2 follows -> [link]");

    QFormLayout *formLayout = new QFormLayout(this);
    formLayout->setContentsMargins(0, 10, 0, 0);
    formLayout->setSizeConstraint(QLayout::SetMinimumSize);
    formLayout->addRow("Channel", m_channelOptions);
    formLayout->addRow("Image size", m_sizeOptions);
    formLayout->addRow("Message 1", m_message);
    formLayout->addRow("Message 2", m_caption);
    formLayout->addRow("", submit);
    formLayout->addRow("Example", example);

    QWidget *scrollWidget = new QWidget(this);
    scrollWidget->setContentsMargins(25, 5, 20, 25);
    scrollWidget->setLayout(formLayout);
    setWidget(scrollWidget);
}

ViewMessageContents::~ViewMessageContents()
{
}

/**
 * Set rotation, calculate image proportions
 */
void ViewMessageContents::setRotation(int rotation)
{
    m_rotation = rotation;
    bool swap = (rotation % 180 == 90);

    QImageReader reader(m_photoUrl.toLocalFile());
    if (reader.canRead()) {
        m_sizeOptions->clear();
        QSize size = reader.size();

        QSize temp = size;
        scaleSize(320, 320, temp);
        m_sizeOptions->addItem(" Small " % getSizeLiteral(temp, swap), temp);

        temp = size;
        scaleSize(640, 640, temp);
        m_sizeOptions->addItem(" Medium " % getSizeLiteral(temp, swap), temp);

        temp = size;
        scaleSize(1280, 1280, temp);
        m_sizeOptions->addItem(" Large " % getSizeLiteral(temp, swap), temp);

        m_sizeOptions->addItem(" Original " % getSizeLiteral(size, swap), size);

        // Set the default size
        int defaultIndex = Mouho::instance().config().defaultSize();
        if (defaultIndex >= 0 && m_sizeOptions->count() > defaultIndex) {
            m_sizeOptions->setCurrentIndex(defaultIndex);
        }
    } else {
        Mouho::instance().showBanner("Can't read image " % reader.fileName());
    }
}

/**
 * Submit!
 */
void ViewMessageContents::sendMessage()
{
    QSharedPointer<Image> image = QSharedPointer<Image>(new Image());
    image->channel = m_channelList->at(m_channelOptions->currentIndex());
    image->imageSummary = encodeXML(m_caption->text().toUtf8()); // Summary -> link
    image->postTitle = encodeXML(m_message->text().toUtf8()); // Title line of post
    image->postContent = ""; // Used for image rotation angle, disregard

    Mouho::instance().config().setDefaultSize(m_sizeOptions->currentIndex());
    Mouho::instance().config().setDefaultChannel(image->channel->name);

    QSize postSize = m_sizeOptions->itemData(m_sizeOptions->currentIndex()).toSize();
    QImageReader reader(m_photoUrl.toLocalFile());
    QFileInfo nameExtracor(reader.fileName());
    image->imageTitle = nameExtracor.fileName();
    QByteArray *base64image = 0;

    if (!reader.canRead()) {
        Mouho::instance().showBanner("Can't read image " % reader.fileName());
        return;
    } else if (reader.size() == postSize && m_rotation == 0) {
        QFile file(m_photoUrl.toLocalFile());
        if (file.open(QFile::ReadOnly)) {
            base64image = new QByteArray(file.readAll().toBase64());
        }
    } else {
        reader.setScaledSize(postSize);
        QImage image = reader.read();
        if (m_rotation != 0) {
            QMatrix m;
            m.rotate(m_rotation);
            image = image.transformed(m);
        }
        QByteArray tempImage;
        QBuffer buffer(&tempImage);
        buffer.open(QIODevice::WriteOnly);
        image.save(&buffer, "JPG", 90);
        base64image = new QByteArray(tempImage.toBase64());
    }

    Q_ASSERT(base64image != 0);
    image->imageData = QSharedPointer<QByteArray>(base64image);
    Mouho::instance().post(image);

    hide();
    if (m_parent) {
        m_parent->hide();
    }
}

/**
 * Replace XML special chars from input
 */
QString& ViewMessageContents::encodeXML(QString source)
{
    // Small input, excessive QString creation probably will not hurt
    return source.replace("&", "&amp;")
            .replace("\\", "&apos")
            .replace("\"", "&quot;")
            .replace("<", "&lt;")
            .replace(">", "&gt;");
}

/**
 * String version of the size for display
 */
QString ViewMessageContents::getSizeLiteral(const QSize &size, bool swap) const
{
    return "(" % QString::number(swap ? size.height() : size.width()) % "x" %
            QString::number(swap ? size.width() : size.height()) % ")";
}

/**
 * Scale the size. Round 1 down to get rid of ugly sizes such as 1280x961.
 */
void ViewMessageContents::scaleSize(int width, int height, QSize &size)
{
    size.scale(width, height, Qt::KeepAspectRatio);
    if (size.width() % 10 == 1) {
        size.setWidth(size.width() - 1);
    }
    if (size.height() % 10 == 1) {
        size.setHeight(size.height() - 1);
    }
}
