#include <QDeclarativeEngine>
#include <QDeclarativeContext>
#include <QPushButton>
#include <QFormLayout>
#include <QLabel>
#include <QFont>

#include "viewlogin.h"
#include "mouho.h"

ViewLogin::ViewLogin(QObject *parent)
    : QObject(parent)
    , m_loginview(0)
    , m_photolist(0)
    , m_details(0)
{
    // Main login widget
    m_loginview = new QWidget();

    // Connections
    connect(&Mouho::instance(), SIGNAL(channelListReceived(bool)), this, SLOT(channelListReceived(bool)));
    connect(&Mouho::instance(), SIGNAL(postStarted()), this, SLOT(postStarted()));
    connect(&Mouho::instance(), SIGNAL(postFinished()), this, SLOT(postFinished()));

    // Logo
    QLabel *logo = new QLabel(m_loginview);
    logo->setPixmap(QPixmap(":/logo.png"));
    logo->setMinimumHeight(150);
    logo->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    // Login form
    m_username = new QLineEdit(m_loginview);
    m_username->setText(Mouho::instance().config().username());
    m_username->setInputMethodHints(m_username->inputMethodHints() | Qt::ImhPreferLowercase);
    m_password = new QLineEdit(m_loginview);
    m_password->setEchoMode(QLineEdit::Password);
    m_password->setInputMethodHints(m_password->inputMethodHints() | Qt::ImhPreferLowercase);
    m_password->setText(Mouho::instance().config().password());

    QPushButton *submit = new QPushButton(m_loginview);
    submit->setText("Log in");
    connect(submit, SIGNAL(clicked()), this, SLOT(login()));

    QFormLayout *formLayout = new QFormLayout(m_loginview);
    formLayout->setContentsMargins(0, 10, 0, 0);
    formLayout->setSizeConstraint(QLayout::SetMinimumSize);
    formLayout->addRow("Username", m_username);
    formLayout->addRow("Password", m_password);
    formLayout->addRow("", submit);

    m_formWidget = new QWidget(m_loginview);
    m_formWidget->setLayout(formLayout);
    m_formWidget->setVisible(true);

    // Wait indicator
    QFont font = QFont();
    font.setPixelSize(30);
    QLabel *loading = new QLabel(m_loginview);
    loading->setText("Loading channels...");
    loading->setFont(font);

    QVBoxLayout *loadingLayout = new QVBoxLayout(m_loginview);
    loadingLayout->addStretch(1);
    loadingLayout->addWidget(loading);
    loadingLayout->addStretch(1);

    m_waitWidget = new QWidget(m_loginview);
    m_waitWidget->setLayout(loadingLayout);
    m_waitWidget->setVisible(false);

    // Main layout
    QVBoxLayout *layout = new QVBoxLayout(m_loginview);
    layout->addWidget(logo);
    layout->setAlignment(logo, Qt::AlignHCenter);
    layout->addWidget(m_waitWidget);
    layout->setAlignment(m_waitWidget, Qt::AlignHCenter);
    layout->addWidget(m_formWidget);

    m_loginview->setLayout(layout);
}

ViewLogin::~ViewLogin()
{
    delete m_loginview;
    delete m_photolist;
    delete m_details;
}

void ViewLogin::show() const
{
    m_loginview->showMaximized();
}

void ViewLogin::login(bool setCredentials)
{
    m_formWidget->setVisible(false);
    m_waitWidget->setVisible(true);
    m_loginview->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);

    if (setCredentials)
    {
        Mouho::instance().config().setUsername(m_username->text());
        Mouho::instance().config().setPassword(m_password->text());
    }

    Mouho::instance().initChannels();
}

void ViewLogin::channelListReceived(bool success)
{
    m_loginview->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    if (success)
    {
        delete m_photolist;
        m_photolist = new QDeclarativeView();
        m_photolist->engine()->addImportPath(QLatin1String("/opt/qtm12/imports"));
        m_photolist->rootContext()->setContextProperty("controller", this);
        m_photolist->setResizeMode(QDeclarativeView::SizeRootObjectToView);
        m_photolist->setSource(QUrl("qrc:/qml/PhotoList.qml"));
        m_photolist->setAttribute(Qt::WA_Maemo5StackedWindow);
        m_photolist->setWindowFlags(m_photolist->windowFlags() | Qt::Window);

        m_photolist->showMaximized();
        m_loginview->hide();
    }
    else
    {
        m_waitWidget->setVisible(false);
        m_formWidget->setVisible(true);
    }
}

void ViewLogin::selectPhoto(QString url)
{
    delete m_details;
    m_details = new ViewPhotoDetails(QUrl(url), m_photolist);
    m_details->showMaximized();
}

void ViewLogin::postStarted()
{
    m_photolist->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
}

void ViewLogin::postFinished()
{
    m_photolist->setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}
