#include <QNetworkRequest>
#include <QNetworkReply>
#include <QDomDocument>
#include <QDomNodeList>
#include <QDateTime>
#include <QCryptographicHash>

#include "networkhandler.h"
#include "mouho.h"

static const QString textTemplateXml =
    "<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
    "<entry xmlns=\"http://purl.org/atom/ns#\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\">\n"
    "<title type=\"text\">%1</title>\n"
    "<created>%2</created>\n"
    "<content type=\"application/xhtml+xml\" mode=\"xml\">%3</content>\n"
    "<link rel=\"related\" type=\"image/jpeg\" href=\"%4\"/>\n"
    "<generator>%5</generator>\n"
    "</entry>";

static const QString imageTemplateXmlPart1 =
    "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n"
    "<entry xmlns=\"http://purl.org/atom/ns#\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\">\r\n"
    "<title type=\"text\">%1</title>\r\n"
    "<issued>%2</issued>\r\n"
    "<standalone xmlns=\"http://sixapart.com/atom/typepad#\">1</standalone>\r\n"
    "<content type=\"image/jpeg\" mode=\"base64\">";

static const QString imageTemplateXmlPart2 =
    "</content>\r\n"
    "<summary>%1</summary>\r\n"
    "<generator>%2</generator>\r\n"
    "</entry>";

NetworkHandler::NetworkHandler(QUrl baseUrl, QObject *parent)
    : QObject(parent)
    , m_manager()
    , m_baseUrl(baseUrl)
{
}

NetworkHandler::~NetworkHandler()
{
}

/**
 * List all the user's channels
 */
void NetworkHandler::listChannels()
{
    QNetworkRequest request;
    request.setUrl(m_baseUrl);
    setWSSEHeadersToRequest(request);

    QNetworkReply *reply = m_manager.get(request);
    connect(reply, SIGNAL(finished()), this, SLOT(handleChannelListReply()));
}

/**
 * Well, post an image
 */
void NetworkHandler::postImage(QSharedPointer<Image> image)
{
    // A smallish string, simple QString->QByteArray conversion
    QByteArray *xml = new QByteArray(imageTemplateXmlPart1
                                       .arg(image->imageTitle)
                                       .arg(isoDate()).toLatin1()
                                     );

    // Insert contents *without* creating a QString out of it
    xml->insert(xml->length(), *image->imageData);

    // Add rest of the XML as QString again
    xml->insert(xml->length(), imageTemplateXmlPart2
                               .arg(image->imageSummary)
                               .arg(Mouho::version()));

    QBuffer *buffer = new QBuffer();
    buffer->setBuffer(xml);

    QNetworkRequest request;
    request.setUrl(image->channel->address);
    setWSSEHeadersToRequest(request);

    QNetworkReply *reply = m_manager.post(request, buffer->data());
    connect(reply, SIGNAL(finished()), this, SLOT(handlePostImageReply()));

    // Image is needed later; buffer has to remain
    // open until finished() signal
    m_images.insert(reply, image);
    m_buffers.insert(reply, QSharedPointer<QBuffer>(buffer));
    m_datas.insert(reply, QSharedPointer<QByteArray>(xml));
}

/**
 * Post the text data accompanying an image
 */
void NetworkHandler::postImageText(QSharedPointer<Image> image)
{
    QByteArray xml = textTemplateXml.arg(image->postTitle)
                       .arg(isoDate())
                       .arg(image->postContent)
                       .arg(image->postId)
                       .arg(Mouho::version())
                       .toLatin1();

    QNetworkRequest request;
    request.setUrl(image->channel->address);
    setWSSEHeadersToRequest(request);

    QNetworkReply *reply = m_manager.post(request, xml);
    connect(reply, SIGNAL(finished()), this, SLOT(handlePostTextReply()));
    m_images.insert(reply, image);
}

/**
 * Handles a network reply hopefully containing a list of channels
 */
void NetworkHandler::handleChannelListReply()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error() == QNetworkReply::NoError) {
        QDomDocument document;
        document.setContent(reply->readAll());

        // Construct a list of channels
        ChannelList channelList(new QList<QSharedPointer<Channel> >());
        QDomNodeList list = document.firstChildElement("feed").childNodes();
        for (int i = 0; i < list.count(); i++) {
            QDomElement node = list.at(i).toElement();
            if (node.nodeName() == "link" && node.hasAttribute("rel") && node.attribute("rel") == "service.post") {
                QSharedPointer<Channel> channel = QSharedPointer<Channel>(new Channel());
                channel->name = node.attribute("title");
                channel->address = node.attribute("href");
                channelList->append(channel);
            }
        }
        emit channelsReceived(channelList);
    } else {
        emit channelListRequestFailed(QString::number(reply->error()));
    }
    reply->deleteLater();
}

/**
 * Get stuff from post image reply
 */
void NetworkHandler::handlePostImageReply()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error() == QNetworkReply::NoError) {
        QDomDocument document;
        document.setContent(reply->readAll());
        QString postId = document.firstChildElement("entry").firstChildElement("id").text();
        if (!postId.isEmpty()) {
            QSharedPointer<Image> image = m_images.value(reply);
            if (image) {
                // Successful image posting, create an accompanying text post
                image->postId = postId;
                postImageText(image);
            } else {
                emit postImageFailed("Can't find the image associated with given image part request");
            }
        } else {
            emit postImageFailed("Failed to receive a valid post identifier for image part");
        }
    } else {
        emit postImageFailed(QString::number(reply->error()));
    }

    // Remove any possible images associated with this reply
    m_images.remove(reply);
    m_buffers.remove(reply);
    m_datas.remove(reply);
    reply->deleteLater();
}

/**
 * Check that the answer is valid -> success! Profit!
 */
void NetworkHandler::handlePostTextReply()
{
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());
    if (reply->error() == QNetworkReply::NoError) {
        QDomDocument document;
        document.setContent(reply->readAll());
        QString postId = document.firstChildElement("entry").firstChildElement("id").text();
        if (!postId.isEmpty()) {
            QSharedPointer<Image> image = m_images.value(reply);
            if (image) {
                emit postImageCompleted(image);
            } else {
                emit postImageFailed("Can't find the image associated with given text part request");
            }
        } else {
            emit postImageFailed("Failed to receive a valid post identifier for text part");
        }
    } else {
        emit postImageFailed(QString::number(reply->error()));
    }
    m_images.remove(reply);
    reply->deleteLater();
}

/**
 * Returns a list of WSSE headers to be inserted to request
 */
void NetworkHandler::setWSSEHeadersToRequest(QNetworkRequest &request)
{
    static const QString headerTemplate = "UsernameToken Username=\"%1\", PasswordDigest=\"%2\", Nonce=\"%3\", Created=\"%4\"";

    qsrand(QDateTime::currentDateTime().toTime_t());
    QCryptographicHash hash(QCryptographicHash::Sha1);
    hash.addData(QByteArray::number(qrand()));
    QByteArray nonce = hash.result().toHex();

    QString datetime = isoDate();
    hash.reset();
    hash.addData(nonce);
    hash.addData(QByteArray(datetime.toLatin1()));
    hash.addData(QByteArray(Mouho::instance().config().password().toLatin1()));

    request.setRawHeader("Authorization", "WSSE profile=\"UsernameToken\"");
    request.setRawHeader("X-WSSE", QByteArray(headerTemplate.arg(Mouho::instance().config().username())
                                                .arg(hash.result().toBase64().constData())
                                                .arg(nonce.constData())
                                                .arg(datetime)
                                                .toLatin1()
                                              ));
}

/**
 * Returns a compliant datetime string
 */
QString NetworkHandler::isoDate()
{
    return QDateTime::currentDateTime().toString(Qt::ISODate) + "Z";
}

/**
 * Returns the length of XML data
 */
int NetworkHandler::xmlTemplateSize() const
{
    return textTemplateXml.length() + imageTemplateXmlPart1.length() + imageTemplateXmlPart2.length();
}
