/*
 *  Microfeed - Backend for accessing feed-based services (utils)
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <gtk/gtk.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-subscriber/microfeedsubscriber.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedmisc.h>
#include <string.h>

#include "configuration.h"

typedef struct {
	int wizard_mode;
	GtkWidget* dialog;
	GtkWidget* ok_button;
} NewDialog;

static void publisher_created(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data) {
	GtkWidget* dialog;

	if (error_name) {
		dialog = gtk_message_dialog_new(*((GtkWindow**)user_data), GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, "Could not create %s: %s", publisher, (error_message ? error_message : error_name));
		gtk_window_set_title(GTK_WINDOW(dialog), "Error");
		gtk_dialog_run(GTK_DIALOG(dialog));
		gtk_widget_destroy(dialog);
		if (*((GtkDialog**)user_data)) {
			gtk_dialog_response(*((GtkDialog**)user_data), GTK_RESPONSE_REJECT);
		}
	} else {
		if (*((GtkDialog**)user_data)) {
			gtk_dialog_response(*((GtkDialog**)user_data), GTK_RESPONSE_ACCEPT);
		}
	}
}

static gboolean create_new_publisher(MicrofeedSubscriber* subscriber, NewDialog* new_dialog, const char* publisher) {
	gboolean success = FALSE;
	GtkWidget* dialog = NULL;

	if (microfeed_subscriber_create_publisher(subscriber, publisher, publisher_created, &dialog)) {
		dialog = gtk_message_dialog_new(GTK_WINDOW(new_dialog->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_CANCEL, "Creating publisher... Wait!");
		gtk_window_set_title(GTK_WINDOW(dialog), "Wait");
		if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
			success = TRUE;
		}
		gtk_widget_destroy(dialog);
		dialog = NULL;
	}

	return success;
}

void on_entry_changed(GtkEditable* editable, gpointer user_data) {
	NewDialog* new_dialog;
	gboolean selected;
	
	new_dialog = (NewDialog*)user_data;
	selected = strlen(gtk_entry_get_text(GTK_ENTRY(editable))) > 0;
	gtk_widget_set_sensitive(new_dialog->ok_button, selected);
}

#ifdef MAEMO_CHANGES

static void on_changed(HildonTouchSelector* touch_selector, gint column, gpointer user_data) {
	NewDialog* new_dialog;
	gboolean selected;
	
	new_dialog = (NewDialog*)user_data;
	selected = (hildon_touch_selector_get_active(touch_selector, 0) == -1 ? FALSE : TRUE);
	gtk_widget_set_sensitive(new_dialog->ok_button, selected);
}

#else

static void on_row_activated(GtkTreeView* tree_view, GtkTreePath* path, GtkTreeViewColumn* column, gpointer user_data) {
	NewDialog* new_dialog;
	
	new_dialog = (NewDialog*)user_data;
	gtk_dialog_response(GTK_DIALOG(new_dialog->dialog), GTK_RESPONSE_ACCEPT);
}

static void on_selection_changed(GtkTreeSelection* selection, gpointer user_data) {
	NewDialog* new_dialog;
	gboolean selected;

	new_dialog = (NewDialog*)user_data;
	selected = gtk_tree_selection_get_selected(selection, NULL, NULL);
	gtk_widget_set_sensitive(new_dialog->ok_button, selected);
}

#endif

char* show_new_dialog(MicrofeedSubscriber* subscriber, GtkWindow* parent) {
	char* new_publisher = NULL;
	MicrofeedConfiguration* configuration;
	const char** providers;
	const char** providers_list;
	NewDialog* new_dialog;
	GtkWidget* button;
	GtkListStore* list_store;
	const char** wizards;
	const char* separator;
	char* publisher_part;
	GtkTreeIter iter;
#ifdef MAEMO_CHANGES
	GtkWidget* wizard_touch_selector;
	HildonTouchSelectorColumn* column;
#else
	GtkWidget* tree_view;
	GtkWidget* scrolled_window;
	GtkTreeViewColumn* column;
	GtkTreeSelection* selection;
#endif
	GtkCellRenderer* renderer;
	GtkWidget* table;
	GtkWidget* label;
	GtkWidget* entry;
#ifdef MAEMO_CHANGES
	GtkWidget* touch_selector;
	GtkWidget* picker_button;
#else
	GtkWidget* combo_box;
#endif
	char* text;
	const char* publisher;
	const char* provider;
	int index;
	int response;
	GtkWidget* mode_button;
	GtkLabel* mode_label;
	GValue value = { 0 };

	configuration = microfeed_configuration_new();
	
	new_dialog = microfeed_memory_allocate(NewDialog);
	new_dialog->wizard_mode = 1;
	new_dialog->dialog = gtk_dialog_new_with_buttons("New Microfeed Publisher", NULL, 0, NULL);
	mode_button = gtk_button_new_with_label("Advanced");
	gtk_dialog_add_action_widget(GTK_DIALOG(new_dialog->dialog), mode_button, 1);
	button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
	gtk_dialog_add_action_widget(GTK_DIALOG(new_dialog->dialog), button, GTK_RESPONSE_REJECT);

#ifdef MAEMO_CHANGES
	new_dialog->ok_button = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_VERTICAL, "OK", NULL);;
#else
	new_dialog->ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
#endif
	gtk_widget_set_sensitive(new_dialog->ok_button, FALSE);
	gtk_dialog_add_action_widget(GTK_DIALOG(new_dialog->dialog), new_dialog->ok_button, GTK_RESPONSE_ACCEPT);

	list_store = gtk_list_store_new(3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	for (wizards = microfeed_configuration_get_wizards(configuration); wizards && *wizards; wizards++) {
		if (!microfeed_configuration_get_publisher_directory(configuration, *wizards)) {			
			if ((separator = strchr(*wizards, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR))) {
				publisher_part = g_strndup(*wizards, separator - *wizards);
			} else {
				publisher_part = g_strdup(*wizards);
			}
			gtk_list_store_append(list_store, &iter);
			gtk_list_store_set(list_store, &iter, 0, publisher_part, 1, microfeed_configuration_get_wizard_name(configuration, *wizards), 2, *wizards, -1);
			g_free(publisher_part);
		}
	}
#ifdef MAEMO_CHANGES
	wizard_touch_selector = hildon_touch_selector_new();
	gtk_widget_set_size_request(wizard_touch_selector, -1, 250);
	g_signal_connect(wizard_touch_selector, "changed", G_CALLBACK(on_changed), new_dialog);
	column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR(wizard_touch_selector), GTK_TREE_MODEL(list_store), NULL, NULL);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "xpad", 4, NULL);
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, FALSE);
	gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(column), renderer, "text", 0);

	renderer = gtk_cell_renderer_text_new();
	g_object_set(renderer, "xpad", 4, NULL);
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
	gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(column), renderer, "text", 1);

 	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(new_dialog->dialog)->vbox), wizard_touch_selector);
#else
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(list_store));
	gtk_widget_set_size_request(tree_view, 300, 100);
	g_signal_connect(tree_view, "row-activated", G_CALLBACK(on_row_activated), new_dialog);
	g_object_unref(G_OBJECT(list_store));

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new_with_attributes ("Publisher", renderer, "text", 0, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW(tree_view), column);

	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Information", renderer, "text", 1, NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW(tree_view), column);

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled_window), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(scrolled_window), tree_view);
 	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(new_dialog->dialog)->vbox), scrolled_window);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree_view));
	g_signal_connect(selection, "changed", G_CALLBACK(on_selection_changed), new_dialog);
#endif

	gtk_widget_show_all(new_dialog->dialog);

	table = gtk_table_new(2, 2, FALSE);
 	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(new_dialog->dialog)->vbox), table);
	label = gtk_label_new("Name:");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, 0, 3, 3);
	entry = gtk_entry_new();
	g_signal_connect(entry, "changed", G_CALLBACK(on_entry_changed), new_dialog);
	gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
	label = gtk_label_new("Provider:");
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, 0, 3, 3);
#ifdef MAEMO_CHANGES
	touch_selector = hildon_touch_selector_new_text();
	for (providers = providers_list = microfeed_configuration_get_providers(configuration); providers && *providers; providers++) {
		text = g_strdup_printf("%s (%s)", microfeed_configuration_get_provider_name(configuration, *providers), *providers);
		hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(touch_selector), text);
		g_free(text);
	}
	hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(touch_selector), 0, 0);
	picker_button = hildon_picker_button_new(HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
	hildon_picker_button_set_selector(HILDON_PICKER_BUTTON(picker_button), HILDON_TOUCH_SELECTOR(touch_selector));
	gtk_table_attach(GTK_TABLE(table), picker_button, 1, 2, 1, 2, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
#else
	combo_box = gtk_combo_box_new_text();
	for (providers = providers_list = microfeed_configuration_get_providers(configuration); providers && *providers; providers++) {
		text = g_strdup_printf("%s (%s)", microfeed_configuration_get_provider_name(configuration, *providers), *providers);
		gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box), text);
		g_free(text);
	}
	gtk_combo_box_set_active(GTK_COMBO_BOX(combo_box), 0);
	gtk_table_attach(GTK_TABLE(table), combo_box, 1, 2, 1, 2, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 3, 3);
#endif

	while ((response = gtk_dialog_run(GTK_DIALOG(new_dialog->dialog))) == 1) {
		mode_label = GTK_LABEL(gtk_bin_get_child(GTK_BIN(mode_button)));
		if (new_dialog->wizard_mode) {
			new_dialog->wizard_mode = 0;
			gtk_label_set_text(mode_label, "Simple");
			gtk_entry_set_text(GTK_ENTRY(entry), "");
			gtk_widget_set_sensitive(new_dialog->ok_button, FALSE);
#ifdef MAEMO_CHANGES
			gtk_widget_hide(wizard_touch_selector);
#else
			gtk_widget_hide(scrolled_window);
#endif
			gtk_widget_show_all(table);
		} else {
			new_dialog->wizard_mode = 1;
			gtk_label_set_text(mode_label, "Advanced");
			gtk_widget_set_sensitive(new_dialog->ok_button, FALSE);
			gtk_widget_hide(table);
#ifdef MAEMO_CHANGES
			gtk_widget_show_all(wizard_touch_selector);
#else
			gtk_tree_selection_unselect_all(selection);
			gtk_widget_show_all(scrolled_window);
#endif
		}
	}
 	if (response == GTK_RESPONSE_ACCEPT) {
		if (new_dialog->wizard_mode) {
#ifdef MAEMO_CHANGES
			if (hildon_touch_selector_get_selected(HILDON_TOUCH_SELECTOR(wizard_touch_selector), 0, &iter)) {
#else
			if (gtk_tree_selection_get_selected(selection, NULL, &iter)) {
#endif
				gtk_tree_model_get_value(GTK_TREE_MODEL(list_store), &iter, 2, &value);
				text = g_strdup(g_value_get_string(&value));
				g_value_unset(&value);
				if (create_new_publisher(subscriber, new_dialog, text)) {
					new_publisher = text;
				} else {
					g_free(text);
				}				
			}			
		} else {
			publisher = gtk_entry_get_text(GTK_ENTRY(entry));
#ifdef MAEMO_CHANGES
			if ((index = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(touch_selector), 0)) != -1) {
#else
			if ((index = gtk_combo_box_get_active(GTK_COMBO_BOX(combo_box))) != -1) {
#endif
				provider = providers_list[index];
				text = g_strconcat(publisher, ":", provider, NULL);
				if (create_new_publisher(subscriber, new_dialog, text)) {
					new_publisher = text;
				} else {
					g_free(text);
				}
			}
		}
	}

	gtk_widget_destroy(new_dialog->dialog);
	microfeed_configuration_free(configuration);

	return new_publisher;
}
