#! /usr/bin/env python
# -*- coding: UTF-8 -*-

import os, sys, time, urllib, dbus, telepathy
import gobject
import mafw_support

from telepathy_support import *

from config import *

UPDATE_TIMEOUT  = 5

class MediaIMStatusUpdater(mafw_support.MafwRendererMonitor):
  __config          = None
  __session         = None
  __timer           = None

  # Media profile type state.
  __media_profile   = None

  # Media information
  __metadata        = None
  __updated_at      = None

  #   User-Specified IM presence information for restoration and detecting
  # modification during runtime.
  __user_presences  = None
  __curr_presences  = None

  def __init__(self, session, config):
    self.__config   = config
    self.__session  = session

    proxy  = bus.get_object(mafw_support.MAFW_GSTREAMER_PLUGIN, mafw_support.MAFW_RENDERER_PATH)
    mafw_support.MafwRendererMonitor.__init__(self, proxy)

    self.__timer    = gobject.timeout_add(1000, self.__timer_process)

  def on_media_changed(self, index, uri):
    self.__metadata = dict()
    self.__metadata['uri'] = uri

  def on_state_started(self):
    #   This is to force the timer process to redisplay track information after
    # a pause.
    self.__updated_at = time.time()

  def on_state_paused(self):
    if self.__media_profile:
      self.__reset_im_status(self.__config.accounts(self.__media_profile))

  def on_state_stopped(self):
    if self.__media_profile:
      self.__reset_im_status(self.__config.accounts(self.__media_profile))

  def on_metadata(self, key, value):
    self.__metadata[key]  = value
    self.__updated_at     = time.time()

  def on_buffering_info(self, status):
    self.__updated_at     = time.time()

  def __timer_process(self):
    self.__config.load()

    #   This is how we debounce skipping many tracks, the track has to be
    # playing for at least 5 seconds before we do the update. This also makes
    # sure we have a complete set of metadata :)
    if self.__metadata and self.__updated_at and time.time() - self.__updated_at > UPDATE_TIMEOUT:
      self.__on_metadata_complete()
      self.__updated_at = None

    return True

  def __detect_media_profile(self):
    profile = None

    if self.__metadata['uri'].startswith('iradiosource::'):
      profile = u'Radio'

    elif self.__metadata.keys().count('video-codec') > 0:
      profile = u'Video'

    else:
      profile = u'Music'

    return profile

  def __on_metadata_complete(self):
    media_profile = self.__detect_media_profile()

    #   Check to see if we're switching media types, if so reset im status,
    # this is required to ensure that configured accounts are restored to user
    # defined state if they were required in the previous profile but not in the
    # new one.
    if self.__media_profile and media_profile != self.__media_profile:
      self.__reset_im_status(self.__config.accounts(self.__media_profile))

    self.__media_profile = media_profile
    debug('INFO: MediaIMStatusUpdater: Detected "%s" media profile.' % media_profile)

    message = self.__generate_status_message()
    debug('INFO: MediaIMStatusUpdater: Generated from Template: ' + message)

    # If this media profile is enabled, then update IM status.
    if self.__config.is_enabled(self.__media_profile):
      self.__update_im_status(self.__config.accounts(self.__media_profile), message)

  def __generate_status_message(self):
    message = self.__config.template(self.__media_profile)

    if self.__media_profile == u'Video':
      fname = urllib.unquote(self.__metadata['uri'].replace('localtagfs::videos/', ''))
      message = message.replace('%f', os.path.splitext(os.path.basename(fname))[0])
    else:
      for key in ['title', 'artist', 'organization']:
        value = None

        try:
          value = self.__metadata[key]
        except KeyError:
          value = u'<unknown>'

        message = message.replace('%' + key[0], value)

    return message    

  def __update_im_status(self, accounts, message):
    if self.__user_presences == None:
      self.__user_presences = dict()

    if self.__curr_presences == None:
      self.__curr_presences = dict()

    for account in accounts:
      ac        = self.__session.get_object(telepathy.ACCOUNT_MANAGER, TELEPATHY_ACCOUNT_PREFIX + account)
      ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

      up,ua,um  = ac_iface.Get(telepathy.ACCOUNT, 'RequestedPresence')

      if not self.__user_presences.has_key(account) or self.__curr_presences[account] != (up, ua, um):
        debug('INFO: MediaIMStatusUpdater: Caching user presence info')
        self.__user_presences[account] = (up, ua, um)

      debug('INFO: MediaIMStatusUpdater: Updating presence information on account: %s' % account)

      ac_iface.Set(telepathy.ACCOUNT, 'RequestedPresence',
                   dbus.Struct((up, ua, message), signature = 'uss'))

      self.__curr_presences[account] = (up, ua, message)

  def __reset_im_status(self, accounts):
    if not self.__user_presences:
      return

    for account in accounts:
      if not self.__user_presences.has_key(account):
        continue

      ac        = self.__session.get_object(telepathy.ACCOUNT_MANAGER, TELEPATHY_ACCOUNT_PREFIX + account)
      ac_iface  = dbus.Interface(ac, dbus.PROPERTIES_IFACE)

      debug('INFO: MediaIMStatusUpdater: Updating presence information on account: %s' % account)

      ac_iface.Set(telepathy.ACCOUNT, 'RequestedPresence',
                   dbus.Struct(self.__user_presences[account], signature = 'uss'))

    self.__user_presences = None
    self.__curr_presences = None

if __name__ == '__main__':
  import gobject, dbus.glib

  mafw_support.DEBUG = True

  bus     = dbus.SessionBus()
  config  = Config(CONFIG_PATH)
  service = MediaIMStatusUpdater(bus, config)
  main    = gobject.MainLoop()

  main.run()

