#include "textselectionwidget.h"
#include <QDebug>
#include <QApplication>
#include <QVBoxLayout>
#include <QApplication>
#include <QStyle>

TextSelectionWidget::TextSelectionWidget(QAbstractKineticScroller *s, QWidget *parent) : QWidget(parent), margin(60), startPosition(0), startPoint(0, 0), endPosition(0), endPoint(0, 0), movingStartHandle(false), movingEndHandle(false), startHandlePoint(0, 0), endHandlePoint(0, 0), scroller(s), panningThreshold(0)
{
    textLayout = new QTextLayout();
    if(scroller)
        panningThreshold = scroller->panningThreshold();
}

void TextSelectionWidget::setText(const QString &text)
{
    textLayout->setText(text);
    textLayout->beginLayout();
    qreal height = 10;
    qreal widthUsed = 0;
    while(1){
        QTextLine line = textLayout->createLine();
        if(!line.isValid())
            break;

        line.setLineWidth(width()-margin);
        height += 10;
        line.setPosition(QPointF(2, height));
        height += line.height();
        lineHeight = line.height() + 10;
        widthUsed = qMax(widthUsed, line.naturalTextWidth());

    }
    setMinimumHeight(height);
    textLayout->endLayout();
    endPosition = textLayout->text().length();

    QTextLine firstLine = textLayout->lineAt(0);
    startHandlePoint = QPoint(firstLine.x() + 15, firstLine.y() + 15);
    QTextLine lastLine = textLayout->lineAt(textLayout->lineCount() - 1);
    endHandlePoint = QPoint(lastLine.cursorToX(endPosition) - 15, lastLine.y() + (lastLine.height() - 15));


    update();

}

QString TextSelectionWidget::getText()
{
    QString text = textLayout->text();
    text.truncate(endPosition);
    text = text.mid(startPosition);
    return text;
}

void TextSelectionWidget::paintEvent(QPaintEvent *event)
{
    QPainter painter(this);

    QVector<QTextLayout::FormatRange> selection;
    QTextLayout::FormatRange range;
    range.start = startPosition;
    range.length = endPosition - startPosition;
    QTextCharFormat format;
    
    format.setBackground(QBrush(QApplication::style()->standardPalette().highlight().color()));
    range.format = format;
    selection.push_back(range);


    textLayout->draw(&painter, QPoint(0, 0), selection);

    //draw handles
    painter.setRenderHint(QPainter::Antialiasing, true);
    QPen pen;
    pen.setColor(Qt::white);
    pen.setWidth(2);
    painter.setPen(pen);
    painter.setBrush(QApplication::style()->standardPalette().highlight());

    QPoint startHandleCenter = startHandlePoint - QPoint(8, 21);
    painter.drawEllipse(startHandleCenter, 6, 6);

    QPoint endHandleCenter = endHandlePoint - QPoint(-8, 21); 
    painter.drawEllipse(endHandleCenter, 6, 6);
    
}

QSize TextSelectionWidget::sizeHint() const
{
    QSize s = textLayout->boundingRect().size().toSize();
    s.setWidth(s.width() + margin);
    return s;
}

void TextSelectionWidget::mousePressEvent(QMouseEvent *event)
{
    QPoint temp(startHandlePoint.x() - 15, startHandlePoint.y() - 27);
    startHandleRect = QRect(temp, QSize(30, lineHeight));
    if(startHandleRect.contains(event->pos())){
        movingStartHandle = true;
        movingEndHandle = false;
        //A hack to prevent scrolling when moving handle
        if(scroller)
            scroller->setPanningThreshold(1000);
    }else {
        QTextLine lastLine = textLayout->lineAt(textLayout->lineCount() - 1);
        temp = QPoint(endHandlePoint.x() - 15, endHandlePoint.y() - 27);
        endHandleRect = QRect(temp, QSize(30, lineHeight));
        if(endHandleRect.contains(event->pos())){
            movingEndHandle = true;
            movingStartHandle = false;
            if(scroller)
                scroller->setPanningThreshold(1000);
        } else {
            movingStartHandle = false;
            movingEndHandle = false;
            if(scroller)
                scroller->setPanningThreshold(panningThreshold);
        }
    }

}

void TextSelectionWidget::mouseMoveEvent(QMouseEvent *event)
{
    if(movingStartHandle){
        startPoint = event->pos(); 
        int line = event->pos().y() / lineHeight;
        if(line >= 0 && line < textLayout->lineCount())
        {
            QTextLine textLine = textLayout->lineAt(line);
            startPosition = textLine.xToCursor(event->pos().x());
            startHandlePoint = QPoint(textLine.cursorToX(startPosition) + 15, textLine.y() + 15);
        }
        update();
    } else if(movingEndHandle){
        endPoint = event->pos(); 
        int line = event->pos().y() / lineHeight;
        if(line >= 0 && line < textLayout->lineCount())
        {
            QTextLine textLine = textLayout->lineAt(line);
            endPosition = textLine.xToCursor(event->pos().x());
            endHandlePoint = QPoint(textLine.cursorToX(endPosition) - 15, textLine.y() + textLine.height() -  15);
        }
        update();
    }


}

void TextSelectionWidget::mouseReleaseEvent(QMouseEvent *event)
{
    //Make sure panning treshold is set back to original or the scrollarea won't work
    if(scroller)
        scroller->setPanningThreshold(panningThreshold);
}
