#include "startwindow.h"
#include "mainwindow.h"

#include <QScrollArea>
#include <QFileDialog>
#include <QMenuBar>

#include <QMaemo5InformationBox>

#include "shared.h"


StartWindow::StartWindow(QWidget *parent) : QMainWindow(parent), readingUI(0), searcCatalogsView(0)
{
    setWindowTitle("MeBook");
    
    QScrollArea *scrollArea = new QScrollArea(this);
    setCentralWidget(scrollArea);

    startUpList = new StartUpList(this);
    scrollArea->setWidget(startUpList);
    scrollArea->setWidgetResizable(true);
    //Pass scrollArea's QAbstractKineticScroller to StartUpList. It's used to detect if user is tapping an item or scrolling the QAbstractKineticScroller.
    //TODO: There might be a better way to do this..
    QAbstractKineticScroller *scroller = scrollArea->property("kineticScroller").value<QAbstractKineticScroller*>();
    startUpList->setScroller(scroller);

    scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(startUpList, SIGNAL(openBook(Book*)), this, SLOT(openBook(Book*)));
    connect(startUpList, SIGNAL(searchCatalogs()), this, SLOT(searchCatalogs()));
    
    setAttribute(Qt::WA_Maemo5StackedWindow);

    connect(startUpList, SIGNAL(openFile()), this, SLOT(openFile()));

    QMenuBar *menuBar = new QMenuBar(this);
    QAction *settings = menuBar->addAction(tr("Settings"));
    connect(settings, SIGNAL(triggered()), this, SLOT(openSettingsDialog()));

    setMenuBar(menuBar);

    readSettings();

    library = new Library("/home/user/MyDocs/books", "/home/user/MyDocs/books/bookdb.db", this);

    connect(library, SIGNAL(bookAdded(Book*)), startUpList, SLOT(addBook(Book*)));
    library->loadLibrary();
/*    QList<Book*> books = library->getBooks(); 
    for(QList<Book*>::iterator iter = books.begin(); iter != books.end(); ++iter){
        startUpList->addBook((*iter));
    }*/

    feedBooksCatalog = new FeedBooks(this);
    connect(feedBooksCatalog, SIGNAL(bookSaved(Book*)), library, SLOT(addToLibrary(Book*)));
    connect(feedBooksCatalog, SIGNAL(message(const QString)), this, SLOT(showMessage(const QString)));
    connect(feedBooksCatalog, SIGNAL(openBook(Book*)), this, SLOT(openBook(Book*)));

    osso_context = osso_initialize(OSSO_NAME.toUtf8(), "0.0.1", TRUE, NULL);

}

StartWindow::~StartWindow()
{
    if(osso_context)
        osso_deinitialize(osso_context);
}

void StartWindow::resizeEvent(QResizeEvent *event)
{
    
//    startUpList->setFixedWidth(event->size().width()-10);
}


void StartWindow::readSettings()
{

    settings = new QSettings("mebook", "MeBook");

    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);


}

void StartWindow::openFile()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Open book"), "/home/user/MyDocs", tr("eBook Files (*.epub)"));
    
    if(!filename.isEmpty()){
        Book *book = new Book(filename);
        library->addToLibrary(book);
        startUpList->addBook(book);
        openBook(book);
    }
}

void StartWindow::openBook(Book *book)
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);

    if(!readingUI){
        readingUI = new MainWindow(osso_context, this);
        connect(readingUI, SIGNAL(closing()), this, SLOT(bookViewClosed()));
    }

    readingUI->setFont(settings->value("font-family").toString(), settings->value("font-size").toInt());
    readingUI->setColorMode(!settings->value("colors").toBool());
    readingUI->enableBacklightForcing(settings->value("backlight").toBool());
    readingUI->setViewMode(static_cast<BookViewMode>(settings->value("viewmode").toInt()));

    readingUI->openBook(book);
    readingUI->show();

    //Hide other windows 
    if(searcCatalogsView){
        if(searcCatalogsView->isVisible())
            searcCatalogsView->hide();
    }

}

void StartWindow::openSettingsDialog()
{
    SettingsDialog *settingsDialog = new SettingsDialog(this);    
    settingsDialog->setOrientation(settings->value("orientation").toInt());
    settingsDialog->setColors(settings->value("colors").toInt());
    settingsDialog->setFontFamily(settings->value("font-family").toString());
    settingsDialog->setFontSize(settings->value("font-size").toInt());
    settingsDialog->setBacklightForce(settings->value("backlight").toBool());
    settingsDialog->setViewMode(static_cast<BookViewMode>(settings->value("viewmode").toInt()));


    if(settingsDialog->exec() == QDialog::Accepted){
        int orientation = settingsDialog->getOrientation();
        if(orientation != settings->value("orientation")){
            settings->setValue("orientation", orientation); 
            setAttribute(static_cast<Qt::WidgetAttribute>(orientation), true);
        }

        int colors = settingsDialog->getColors();
        if(colors != settings->value("colors")){
            settings->setValue("colors", colors);
        }

        settings->setValue("font-family", settingsDialog->getFontFamily());
        settings->setValue("font-size", settingsDialog->getFontSize());
        settings->setValue("backlight", settingsDialog->getBacklighForce());
        settings->setValue("viewmode", settingsDialog->getViewMode());
    }
}

void StartWindow::searchCatalogs()
{
    if(!searcCatalogsView){
        searcCatalogsView = new SearchCatalogsView(feedBooksCatalog, this);
        connect(searcCatalogsView, SIGNAL(closing()), this, SLOT(searchCatalogsClosed()));
    }
    setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
    searcCatalogsView->show();
}

void StartWindow::searchCatalogsClosed()
{
    setAttribute(static_cast<Qt::WidgetAttribute>(settings->value("orientation").toInt()), true);
}

void StartWindow::bookViewClosed()
{
    delete readingUI;
    readingUI = 0;
}

void StartWindow::showMessage(const QString message)
{
    QMaemo5InformationBox::information(this, message, QMaemo5InformationBox::DefaultTimeout); 
}
