#include "bookview.h"

#include <QDebug>
#include <QAction>
#include <QWebElement>
#include <QWebFrame>
#include <QMouseEvent>
#include <QWebSettings>
#include <QApplication>
#include <QDesktopWidget>
#include <QAbstractKineticScroller>
#include <QFontMetrics>
#include <QFontDatabase>
#include <math.h>
#include <QTime>
#include <QMenu>
#include <QMessageBox>

#include "addbookmarkdialog.h"
#include "addannotationdialog.h"

#include "libosso.h"


BookView::BookView(QWidget *parent) : QGraphicsView(parent), autoScrollStep(1), autoScrollBy(0), currentSection(0), currentPage(0), isTap(false), tapMove(0)
{
    setMouseTracking(false); 

    snapEnabled = true;

    scene = new QGraphicsScene();
    setScene(scene);
    scene->setBackgroundBrush(Qt::white);

    pageWidgetCurrent = NULL;
    pageWidgetNext = NULL;
    scrollablePage = NULL;
    

    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    
    renderer = new Renderer(this);
    connect(renderer, SIGNAL(tileRendered(RenderTile)), this, SLOT(addTile(RenderTile)));
    connect(this, SIGNAL(requestRender(QString, int)), renderer, SLOT(addToQueue(QString, int)));

    renderer->setRenderSettings(&renderSettings);
  
    //Set PageMode as default.
    setViewMode(PageMode);

    autoScrollTimer.setInterval(100);
    connect(&autoScrollTimer, SIGNAL(timeout()), this, SLOT(handleAutoScroll()));

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(openContextMenu(const QPoint&)));
}

BookView::~BookView()
{
    delete scene;
}
//TODO: Handle invalid book data. 
void BookView::setBook(Book *openBook)
{
    if(openBook){
        book = openBook;
        

        if(!book->isLoaded())
            book->load();

        renderer->setAnnotations(book->getAnnotations());
        renderer->setResourceManager(book->getResourceManager());

        QString content;
        //TODO: Refactor
        if(book->getProgress().isValid()){
            if(viewMode==ScrollingBookMode){
                content =  book->getBook();
                emit requestRender(content, 0);
            } else if(viewMode==PageMode){
                content = book->getSection(book->getProgress().getSection());
                currentSection = book->getProgress().getSection();
                currentPage = book->getProgress().getPage();
                tiles.clear();
                emit requestRender(content, currentSection);
            }
        }else if(viewMode == ScrollingBookMode){ 
            content =  book->getBook();
            tiles.clear();
            emit requestRender(content, 0);
            scrollablePage->setPosition(0);
        } else if(viewMode== PageMode){
            content = book->getCover();
            currentSection = 0;
            currentPage = 0;
            tiles.clear();
            emit requestRender(content, currentSection);
        }



    }
}

void BookView::goToSection(int section, int page)
{ 
    if(viewMode == PageMode){
        if(tiles.contains(section)){
            currentSection = section;
            //TODO: Somehow make sure page exists.
            currentPage = page;
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else{
            QString content = book->getSection(section);
            currentSection = section;
            currentPage = page;
            pageWidgetCurrent->setPixmap(loadingPixmap());
            tiles.clear();
            emit requestRender(content, currentSection);
        }
    }else{
        scrollablePage->scrollToAnchor("#section" + QString::number(section));
    }
}

void BookView::goToChapter(QString chapter)
{
    goToSection(book->getSectionNumber(chapter));
}

void BookView::goToBookmark(int id)
{
    QList<Bookmark> bookmarks = book->getBookmarks();
    //TODO: Handle error
    if(id < bookmarks.count() && id >= 0){
        Bookmark bm = bookmarks.at(id);
        if(viewMode == PageMode)
            goToSection(bm.getSection(), bm.getPage());
        else
            scrollablePage->setPosition(bm.getScroll());
    }else{
        qDebug() << "Can't find bookmark!";
    }
}

void BookView::goToAnnotation(int id)
{
    QList<Annotation> annotations = book->getAnnotations();
    if(id < annotations.count() && id >= 0){
        Annotation annotation = annotations[id];

        if(viewMode == PageMode)
            goToSection(annotation.getSection(), annotation.getPage());
        else
            scrollablePage->setPosition(annotation.getScroll());
    }
}

void BookView::setFont(const QString &family, int size)
{
    renderSettings.setFont(QFont(family, size));    
}

void BookView::setNightMode(bool toggle)
{
    if(toggle){
        renderSettings.setTextColor(QColor(Qt::white));
        renderSettings.setBackgroundColor(QColor(Qt::black));
        scene->setBackgroundBrush(QBrush(Qt::black));
    }
    else{
        renderSettings.setTextColor(QColor(Qt::black));
        renderSettings.setBackgroundColor(QColor(Qt::lightGray));
        scene->setBackgroundBrush(QBrush(Qt::lightGray));
    }
    
}

void BookView::setViewMode(BookViewMode mode)
{
    viewMode = mode;    

    bool orientation = testAttribute(Qt::WA_Maemo5LandscapeOrientation);
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    int pageWidth = 0;
    int pageHeight = 0;
    qDebug() << "Orientation: " << orientation;
    if(orientation == false){
        pageWidth = 480;
        pageHeight = 800;
    }
    else{
        pageWidth = 800;
        pageHeight = 480;
    }

    if(viewMode == PageMode){
        scene->clear();

        pageWidgetCurrent = new Page();
        pageWidgetNext = new Page();

        scene->addItem(pageWidgetCurrent);
        scene->addItem(pageWidgetNext);
//        connect(renderer, SIGNAL(contentHeight(int)), this, SLOT(contentHeight(int)));
        scene->setSceneRect(QRectF(0, 0, pageWidth, pageHeight));
//        ensureVisible(QRectF(0, 0, pageWidth, pageHeight));
        pageWidgetNext->clearPixmap();
        tiles.clear();
        pageWidgetCurrent->setGeometry(0, 0, pageWidth, pageHeight);
        pageWidgetNext->setGeometry(pageWidth, 0, pageWidth, pageHeight);
        pageWidgetCurrent->setPixmap(loadingPixmap());
        renderer->enablePagination(true);
        renderer->enablePixmapRendering(true);

        

    } else{
        scene->clear();

        scrollablePage = new ScrollablePage();
        scene->addItem(scrollablePage);
        connect(renderer, SIGNAL(webPageLoaded(QWebPage*)), this, SLOT(webPageLoaded(QWebPage*)));
        scrollablePage->setGeometry(0, 0, pageWidth, pageHeight);
        scene->setSceneRect(0, 0, pageWidth, pageHeight);
        renderer->enablePagination(false);
        renderer->enablePixmapRendering(false);

        //scene->removeItem(pageWidgetNext);

    }

    pageSize = QSizeF(pageWidth, pageHeight);    
    renderer->setPageSize(pageSize);
    renderSettings.setMaxImageSize(pageWidth-30, pageHeight-30);

}

void BookView::addBookmark(QString name, QColor color)
{
    int scroll = 0;
    if(viewMode == ScrollingBookMode)
        scroll = scrollablePage->getPosition(); 
    Bookmark bookmark(name, color, currentSection, currentPage, scroll);
    book->addBookmark(bookmark);
}

void BookView::mousePressEvent(QMouseEvent *event)
{
    if(viewMode == PageMode)
        swipeStart = event->pos();    

    isTap = true;
    tapMove = 0;

    QGraphicsView::mousePressEvent(event);  
}

void BookView::mouseMoveEvent(QMouseEvent *event)
{
   if(viewMode == PageMode){
       if(!swipeStart.isNull() && event->pos().x() > 0){
            float moveX = event->pos().x() - swipeStart.x();
            swipeStart = event->pos();
            pageWidgetCurrent->moveBy(moveX, 0);

            tapMove += moveX;
            if((tapMove > 10 || tapMove < -10) && isTap){
                isTap = false;
                tapMove = 0;
            }

            int pageWidth = pageWidgetNext->geometry().width();
            float nextX = (pageWidgetCurrent->pos().x() >= 0) ? pageWidgetCurrent->pos().x() - pageWidth : pageWidgetCurrent->pos().x() + pageWidth;
            pageWidgetNext->setPos(nextX, 0);

            if(nextX > 0){
                pageWidgetNext->setPixmap(getNextPagePixmap());
            }else{
                pageWidgetNext->setPixmap(getPreviousPagePixmap());
            }
       }
   }
    QGraphicsView::mouseMoveEvent(event);  
}

void BookView::mouseReleaseEvent(QMouseEvent *event)
{
    if(viewMode == PageMode){
        int pageWidth = pageSize.width();
        if(pageWidgetCurrent->x() < -(pageWidth/2)){
            pageWidgetCurrent->setPos(-pageWidth, 0);
            pageWidgetNext->setPos(0, 0);
            Page *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            nextPage();
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else if(pageWidgetCurrent->x() > (pageWidth/2) && (currentSection != 0 || currentPage != 0)){
            pageWidgetCurrent->setPos(pageWidth, 0);
            pageWidgetNext->setPos(0, 0);
            Page *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            previousPage(); 
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
            
        }else{
            pageWidgetCurrent->setPos(0, 0);
            pageWidgetNext->setPos(pageWidth, 0);
        }
        swipeStart = QPoint();
    }


    //Test if the user tapped a annotation
    if(isTap){
        QString annotationHit;
        bool hit = false;
        if(viewMode == PageMode){
            hit = renderer->hitAnnotation(event->pos(), annotationHit, &tiles[currentSection][currentPage]);
        } else{
            QPoint t = QPoint(event->pos().x(), scrollablePage->getPosition() + event->pos().y());
            hit = renderer->hitAnnotation(t, annotationHit);
        }

        if(hit){
            Annotation *a = book->getAnnotation(annotationHit);
            if(a){
                QMessageBox annotationBox(this);
                annotationBox.setText(a->getAnnotation());
                annotationBox.setWindowTitle(tr("Annotation"));
                annotationBox.setStandardButtons(QMessageBox::Close);
                annotationBox.exec();
            }
        }
    }


    QGraphicsView::mouseReleaseEvent(event);
}

void BookView::resizeEvent(QResizeEvent *event)
{
    //Make sure pageWidget stays fully visible when returning to application from app switcher
    QPointF viewPos = mapToScene(pos());
    if(viewPos != QPointF(0, 0)){
        QPointF move = QPointF(0, 0) - viewPos;
        translate(viewPos.x(), viewPos.y());
    }
    ensureVisible(QRectF(0, 0, pageSize.width(), pageSize.height()));
    QGraphicsView::resizeEvent(event);
}


QPixmap BookView::loadingPixmap()
{
    QPixmap tile(20, 20);
    QPainter painter;
    painter.begin(&tile);
    painter.setPen(Qt::NoPen);
    painter.setBrush(Qt::lightGray);
    painter.drawRect(0, 0, 20, 20);
    painter.setBrush(Qt::white);
    painter.drawRect(0, 0, 10, 10);
    painter.drawRect(10, 10, 10, 10);
    painter.end();
    QPixmap pixmap(pageSize.toSize());
    painter.begin(&pixmap);
    painter.drawTiledPixmap(0, 0, pageSize.width(), pageSize.height(), tile);
    painter.end();
    return pixmap;
}

void BookView::nextPage()
{
    if(tiles.keys().contains(currentSection)){
        QList<RenderTile> pages = tiles[currentSection];
        if(currentPage < pages.count()-1){
            currentPage++;
        }else{
            currentSection++;
            currentPage = 0;
            if(currentSection >= book->getNumberOfSections())
                currentSection = 0;
        }
    }

}

void BookView::previousPage()
{
    if(tiles.keys().contains(currentSection)){
        QList<RenderTile> pages = tiles[currentSection];
        if(0 < currentPage)
            currentPage--;
        else{
            currentSection--;
            if(tiles.keys().contains(currentSection)){
                pages = tiles[currentSection];
                currentPage = pages.count()-1;
            }
        }
    }

    if(currentSection < 0)
        currentSection = 0;
}

QPixmap BookView::getCurrentPagePixmap()
{
    if(tiles.keys().contains(currentSection)){
        QList<RenderTile> pages = tiles[currentSection];
        if(currentPage < pages.count())
            return pages[currentPage].getPixmap();
    }else{
        QString content = book->getSection(currentSection); 
        currentPage = 0;
        tiles.clear();
        emit requestRender(content, currentSection);
        //TODO: Loading pixmap..
        return loadingPixmap();
    }
    return loadingPixmap();
}

QPixmap BookView::getNextPagePixmap()
{
    if(tiles.keys().contains(currentSection)){
        QList<RenderTile> pages = tiles[currentSection];
        if(currentPage+1 < pages.count()){
            return pages[currentPage+1].getPixmap();
        }else if(tiles.keys().contains(currentSection+1)){
            pages = tiles[currentSection+1];
            return pages[0].getPixmap(); 
        }/*else{
            if(currentSection+1 < book->getNumberOfSections()){
                QString content = book->getSection(currentSection+1);
                emit requestRender(content, currentSection+1);
                return loadingPixmap();
            }else if(tiles.keys().contains(0)){ 
                pages = tiles[0];
                return pages[0].getPixmap();
            }else{
                QString content = book->getSection(0);
                emit requestRender(content, 0);
                return loadingPixmap();
            }
        }*/
    }
    return loadingPixmap();
}

QPixmap BookView::getPreviousPagePixmap()
{
    if(tiles.keys().contains(currentSection)){
        QList<RenderTile> pages = tiles[currentSection];
        if(currentPage-1 >= 0){
            return pages[currentPage-1].getPixmap();
        }else if(tiles.keys().contains(currentSection-1)){
            pages = tiles[currentSection-1];
            return pages.last().getPixmap(); 
        }/*else{
            if(currentSection-1 >= 0){
                //TODO: Set current page to last page of rendered section
                QString content = book->getSection(currentSection-1);
                emit requestRender(content, currentSection-1);
                return loadingPixmap();
            }
            else return QPixmap();
        }*/
    }
    return loadingPixmap();
}


//TODO: Autoscroll
void BookView::volumeUpPressed(){
   if(viewMode == PageMode){
        nextPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy += autoScrollStep; 
        autoScrollTimer.start();
   }
}

void BookView::volumeDownPressed()
{
   if(viewMode == PageMode){
        previousPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy -= autoScrollStep; 
        autoScrollTimer.start();
   }
}


//Progress saved by sectionnumber, page and scrollposition.
void BookView::saveProgress()
{
    if(viewMode == PageMode){
        int scrollPosition = book->getProgress().getScroll();
        int section = currentSection;
        int page = currentPage;

        BookProgress progress;
        progress.setProgress(scrollPosition, section, page);
        book->setProgress(progress);
    }else{
        int scrollPosition = scrollablePage->getPosition();
        int section = book->getProgress().getSection();
        int page = book->getProgress().getPage();

        BookProgress progress;
        progress.setProgress(scrollPosition, section, page);
        book->setProgress(progress);
    }

}

void BookView::addTile(RenderTile tile)
{
    if(tiles.keys().contains(tile.getSection()))
        tiles[tile.getSection()].push_back(tile);
    else{
        QList<RenderTile> pages;
        pages.push_back(tile);
        tiles.insert(tile.getSection(), pages);
    }

    if(viewMode == PageMode){
        if(tile.getID() == currentPage && tile.getSection() == currentSection){
            pageWidgetCurrent->setPixmap(tile.getPixmap());
        }
    }else{ 
        pageWidgetCurrent->addPixmap(tile.getPixmap());
    }
    
    //Set first page as cover if no cover image found.
    if(tile.getSection() == 0 && tile.getID() == 0 && book->getCoverImage().isNull()){
        book->addCoverImage(tile.getPixmap());
    }
}

void BookView::contentHeight(int height)
{
    pageWidgetCurrent->setContentHeight(height);
    pageWidgetNext->setContentHeight(height);
}


void BookView::webPageLoaded(QWebPage *page){
    scrollablePage->setPage(page);
    if(book->getProgress().isValid()){
        scrollablePage->setPosition(book->getProgress().getScroll());
    }
}

void BookView::handleAutoScroll()
{
    if(autoScrollBy != 0)
        scrollablePage->scrollBy(autoScrollBy);
    else
        autoScrollTimer.stop();
}

void BookView::openContextMenu(const QPoint &pos)
{
    QMenu *menu = new QMenu();
    menu->addAction(tr("Add bookmark"), this, SLOT(addBookmark()));
    menu->addAction(tr("Add annotation"), this, SLOT(addAnnotation()));

    QString temp;
    if(viewMode == PageMode){
        temp = renderer->getParagraphAt(pos, &tiles[currentSection][currentPage]);
    } else{
        QPoint t = QPoint(pos.x(), scrollablePage->getPosition() + pos.y());
        temp = renderer->getParagraphAt(t);
    }

    if(temp.isEmpty()) temp = "Couldn't get paragraph!";

    annotationParagraph = temp;

    menu->exec(mapToGlobal(pos));
}

void BookView::addBookmark()
{
    AddBookmarkDialog dialog;
    if(dialog.exec() == QDialog::Accepted){
        addBookmark(dialog.getName(), dialog.getColor());        
    }
}

void BookView::addAnnotation()
{
    AddAnnotationDialog dialog;
    dialog.setParagraph(annotationParagraph);
    if(dialog.exec() == QDialog::Accepted){
        int scrollPosition = book->getProgress().getScroll();
        if(viewMode == ScrollingBookMode)
            scrollPosition = scrollablePage->getPosition();
        Annotation annotation(currentSection, currentPage, scrollPosition, annotationParagraph, dialog.getSelectedText(), dialog.getAnnotation(), dialog.getColor());

        book->addAnnotation(annotation);
        renderer->setAnnotations(book->getAnnotations());
        if(viewMode == PageMode){
            renderer->renderTile(tiles[currentSection][currentPage]);
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }
    }
}
