#include "library.h"

#include <QSqlError>
#include <QBuffer>
#include <QImageWriter>
#include <QImageReader>
#include <QByteArray>

Library::Library(const QString &lD, const QString &databasename, QObject *parent) : QObject(parent), libraryDirectory(lD)
{
    //Create library directory if it doesn't exits before connecting to database
    QDir dir(libraryDirectory);
    if(!dir.exists())
        dir.mkdir(dir.absolutePath());

    connectdb(databasename);
}

Library::~Library()
{
    for(QList<Book*>::iterator iter = books.begin(); iter != books.end(); ++iter)
    {
        saveBookProgress(*iter);
        saveBookmarks(*iter);
        saveAnnotations(*iter);
        saveBooktodb(*iter);
    }
}

void Library::loadLibrary()
{
    scanDirectory(libraryDirectory);
    loadBooksfromdb();
    checkForChanges();
}

void Library::scanDirectory(const QString &directory)
{
    QDir dir(directory);
    if(dir.exists()){
        QStringList sbooks = dir.entryList(QStringList("*.epub"), QDir::Files);
        for(QStringList::iterator iter = sbooks.begin(); iter != sbooks.end(); ++iter)
        {
            (*iter) = dir.absolutePath() + "/" + (*iter);
        }
        filenames.append(sbooks);
        QStringList dirs = dir.entryList(QDir::Dirs | QDir::NoSymLinks);

        QStringList::const_iterator iter;
        for(iter = dirs.begin(); iter != dirs.end(); ++iter){
            if((*iter) != "." && (*iter) != "..")
                scanDirectory(dir.absolutePath() + "/" + *iter);
        }
    }
}

void Library::connectdb(const QString &databasefile)
{
    database = QSqlDatabase::addDatabase("QSQLITE");
    database.setDatabaseName(databasefile);

    if(!database.open()){
        qDebug() << "Cannot connect to database! " << database.lastError();
        return;
    }

    if(!database.tables().contains("recent")){
        QSqlQuery query("CREATE TABLE recent(INTEGER PRIMARY KEY, filename)", database);
    }
    if(!database.tables().contains("lastspot")){
        QSqlQuery query("CREATE TABLE lastspot(filename, scroll, section, page)", database);
    }
    if(!database.tables().contains("bookmarks")){
        QSqlQuery query("CREATE TABLE bookmarks(filename, name, color, scroll, section, page)", database);
    }
    if(!database.tables().contains("annotations")){
        QSqlQuery query("CREATE TABLE annotations(filename, section, page, scroll, paragraph, text, color, annotation)", database);
    }
    if(!database.tables().contains("books")){
        QSqlQuery query("CREATE TABLE books(id INTEGER PRIMARY KEY, filename, title, author, subject, cover)", database);
    }

}

void Library::saveBookProgress(Book *book)
{
    
    BookProgress progress = book->getProgress();
    if(!progress.isValid())
        return;
    QSqlQuery selectQuery("SELECT * FROM lastspot WHERE filename=:filename", database);
    selectQuery.bindValue(":filename", book->getFilename());
    selectQuery.exec();
    if(!selectQuery.next()){
        QSqlQuery insertQuery("INSERT INTO lastspot VALUES(:filename, :scroll, :section, :page)");
        insertQuery.bindValue(":filename", book->getFilename());
        insertQuery.bindValue(":scroll", progress.getScroll());
        insertQuery.bindValue(":section", progress.getSection());
        insertQuery.bindValue(":page", progress.getPage());

        if(!insertQuery.exec())
            qDebug() << "Database error: " << insertQuery.lastError();
    }else{
        QSqlQuery updateQuery("UPDATE lastspot SET scroll=:scroll, section=:section, page=:page WHERE filename=:filename", database);
        updateQuery.bindValue(":scroll", progress.getScroll());
        updateQuery.bindValue(":section", progress.getSection());
        updateQuery.bindValue(":page", progress.getPage());
        updateQuery.bindValue(":filename", book->getFilename());

        if(!updateQuery.exec())
            qDebug() << "Database error: " << updateQuery.lastError();
    }
}

void Library::loadBookProgress(Book *book)
{
    QSqlQuery query("SELECT * FROM lastspot WHERE filename=:filename", database);
    query.bindValue(":filename", book->getFilename());
    if(query.exec()){
        if(query.next()){

            BookProgress progress;
            progress.setProgress(query.value(1).toInt(), query.value(2).toInt(), query.value(3).toInt());

            book->setProgress(progress);
        }
    }
    else{
        qDebug() << "Database error: " << query.lastError();
    }
}

void Library::saveBookmarks(Book *book)
{
    QList<Bookmark> bookmarks = book->getBookmarks();
    for(int i = 0; i < bookmarks.count(); ++i){
        Bookmark bookmark = bookmarks[i];
        QSqlQuery selectQuery("SELECT * FROM bookmarks WHERE filename=:filename AND name=:name", database);
        selectQuery.bindValue(":filename", book->getFilename());
        selectQuery.bindValue(":name", bookmark.getName());
        selectQuery.exec();
        if(!selectQuery.next()){
                QSqlQuery insertQuery("INSERT INTO bookmarks VALUES(:filename, :name, :color, :scroll, :section, :page)");
                insertQuery.bindValue(":filename", book->getFilename());
                insertQuery.bindValue(":name", bookmark.getName());
                insertQuery.bindValue(":color", bookmark.getColor().name());
                insertQuery.bindValue(":scroll", bookmark.getScroll());
                insertQuery.bindValue(":section", bookmark.getSection());
                insertQuery.bindValue(":page", bookmark.getPage());

                if(!insertQuery.exec())
                    qDebug() << "Database error: " << insertQuery.lastError();
            
        }/*else{
            qDebug() << "Updating bookmark";
            QSqlQuery updateQuery("UPDATE bookmarks SET name=:name, color=:color, scroll=:scroll, section=:section, page=:page WHERE filename=:filename", database);
            updateQuery.bindValue(":name", bookmark.getName());
            updateQuery.bindValue(":color", bookmark.getColor().name());
            updateQuery.bindValue(":scroll", bookmark.getScroll());
            updateQuery.bindValue(":section", bookmark.getSection());
            updateQuery.bindValue(":page", bookmark.getPage());
            updateQuery.bindValue(":filename", book->getFilename());

            if(!updateQuery.exec())
                qDebug() << "Database error: " << updateQuery.lastError();
        }*/
    }
}

void Library::loadBookmarks(Book *book)
{
    QSqlQuery query("SELECT * FROM bookmarks WHERE filename=:filename", database);
    query.bindValue(":filename", book->getFilename());
    if(query.exec()){
        while(query.next()){
            Bookmark bookmark(query.value(1).toString(), QColor(query.value(2).toString()), query.value(4).toInt(), query.value(5).toInt(), query.value(3).toInt());

            book->addBookmark(bookmark);
        }
    }
    else{
        qDebug() << "Database error: " << query.lastError();
    }
}

void Library::saveAnnotations(Book *book)
{

    QList<Annotation> annotations = book->getAnnotations();
    for(int i = 0; i < annotations.count(); ++i){
        Annotation annotation = annotations[i];
        QSqlQuery selectQuery("SELECT * FROM annotations WHERE filename=:filename AND annotation=:annotation", database);
        selectQuery.bindValue(":filename", book->getFilename());
        selectQuery.bindValue(":annotation", annotation.getAnnotation());
        selectQuery.exec();
        if(!selectQuery.next()){
                QSqlQuery insertQuery("INSERT INTO annotations VALUES(:filename, :section, :page, :scroll, :paragraph, :text, :color, :annotation)");
                insertQuery.bindValue(":filename", book->getFilename());
                insertQuery.bindValue(":section", annotation.getSection());
                insertQuery.bindValue(":page", annotation.getPage());
                insertQuery.bindValue(":scroll", annotation.getScroll());
                insertQuery.bindValue(":paragraph", annotation.getParagraph());
                insertQuery.bindValue(":text", annotation.getText());
                insertQuery.bindValue(":color", annotation.getColor().name());
                insertQuery.bindValue(":annotation", annotation.getAnnotation());

                if(!insertQuery.exec())
                    qDebug() << "Database error: " << insertQuery.lastError();
            
        }/*else{
            qDebug() << "Updating bookmark";
            QSqlQuery updateQuery("UPDATE bookmarks SET name=:name, color=:color, scroll=:scroll, section=:section, page=:page WHERE filename=:filename", database);
            updateQuery.bindValue(":name", bookmark.getName());
            updateQuery.bindValue(":color", bookmark.getColor().name());
            updateQuery.bindValue(":scroll", bookmark.getScroll());
            updateQuery.bindValue(":section", bookmark.getSection());
            updateQuery.bindValue(":page", bookmark.getPage());
            updateQuery.bindValue(":filename", book->getFilename());

            if(!updateQuery.exec())
                qDebug() << "Database error: " << updateQuery.lastError();
        }*/
    }
}

void Library::loadAnnotations(Book *book)
{

    QSqlQuery query("SELECT * FROM annotations WHERE filename=:filename", database);
    query.bindValue(":filename", book->getFilename());
    if(query.exec()){
        while(query.next()){
            Annotation annotation(query.value(1).toInt(), query.value(2).toInt(), query.value(3).toInt(), query.value(4).toString(), query.value(5).toString(), query.value(7).toString(), QColor(query.value(6).toString()));

            book->addAnnotation(annotation);
        }
    }
    else{
        qDebug() << "Database error: " << query.lastError();
    }
}

bool Library::checkForChanges()
{

    QStringList filesindb;
    for(QList<Book*>::iterator iter = books.begin(); iter != books.end(); ++iter)
    {
        Book *book = *iter;
        if(!QFile::exists(book->getFilename())){
            books.removeOne(book);

        }else{
            filesindb.push_back(book->getFilename());
        }
    }

    QStringList newfiles;
    for(QStringList::const_iterator iter = filenames.begin(); iter != filenames.end(); ++iter)
    {
        if(!filesindb.contains((*iter))){
            newfiles.push_back(*iter);
        }
    }

    for(QStringList::const_iterator iter = newfiles.begin(); iter != newfiles.end(); ++iter)
    { 
        Book *book = new Book(*iter, true);         
        qDebug() << "Found a new Book";
        book->loadMetaData();
        addToLibrary(book);
    }

}

void Library::loadBooksfromdb()
{

    QSqlQuery query("SELECT * FROM books");

    if(!query.exec())
        return;

    while(query.next()){
        QString filename = query.value(1).toString();
        QString title = query.value(2).toString();
        QString author = query.value(3).toString();
        QString subject = query.value(4).toString();
        QByteArray ba = query.value(5).toByteArray();
        QPixmap cover;
        cover.loadFromData(ba, "PNG");

        Book *book = new Book(filename, true);         
        book->setTitle(title);
        book->setAuthor(author);
        book->setSubject(subject);
        if(!cover.isNull())
            book->addCoverImage(cover);

        loadBookProgress(book);
        loadBookmarks(book);
        loadAnnotations(book);

        addToLibrary(book);
    }
}

void Library::saveBooktodb(Book *book)
{
    
    QSqlQuery selectQuery("SELECT * FROM books WHERE filename=:filename", database);
    selectQuery.bindValue(":filename", book->getFilename());
    selectQuery.exec();
    if(!selectQuery.next()){
        QSqlQuery insertQuery("INSERT INTO books VALUES(NULL, :filename, :title, :author, :subject, :cover)");
        insertQuery.bindValue(":filename", book->getFilename());
        insertQuery.bindValue(":title", book->getTitle());
        insertQuery.bindValue(":author", book->getAuthor());
        insertQuery.bindValue(":subject", book->getSubject());

        QPixmap pixmap = book->getCoverImage();
        QByteArray ba;
        QBuffer buffer(&ba);
        buffer.open(QIODevice::WriteOnly);
        if(!pixmap.isNull())
            pixmap.scaledToWidth(100);
        pixmap.save(&buffer, "PNG");
        insertQuery.bindValue(":cover", ba);

        if(!insertQuery.exec())
            qDebug() << "Database error: " << insertQuery.lastError();
    }else{
        QByteArray bat = selectQuery.value(5).toByteArray();
        QPixmap covert;
        covert.loadFromData(bat, "PNG");

        //Updating only the cover for now
        if(covert.isNull()){
            int id = selectQuery.value(0).toInt();
            QSqlQuery updateQuery("UPDATE books SET cover=:cover WHERE id=:id", database);
            QPixmap pixmap = book->getCoverImage();
            QByteArray ba;
            QBuffer buffer(&ba);
            buffer.open(QIODevice::WriteOnly);
            pixmap.save(&buffer, "PNG");
            updateQuery.bindValue(":cover", ba);
            updateQuery.bindValue(":id", id);

            if(!updateQuery.exec())
                qDebug() << "Database error: " << updateQuery.lastError();
        }
    }
}

void Library::addToLibrary(Book *book)
{
    books.push_back(book);
    book->changeLibraryStatus(true);
    emit bookAdded(book);
}

void Library::addToLibrary(const QString& filename)
{
    Book *book = new Book(filename, true);
    book->loadMetaData();
    addToLibrary(book);
}

QList<Book*> Library::getBooks()
{
    return books;
}
