#include "book.h"

#include <QPainter>
#include <QDebug>
Book::Book() : generator(0), loaded(false), inLibrary(false), lengthInWords(0)
{

}

Book::Book(const QString &bookFilename, bool loadLater) : generator(0), loaded(false), inLibrary(false), lengthInWords(0)
{
    filename = bookFilename;
    if(!loadLater){
        setUpGenerator();
        openBook(filename);
    }

}

Book::~Book()
{
    delete generator;
    //TODO: delete book data.
}

void Book::openBook(const QString &bookFilename)
{
    if(!generator)
        setUpGenerator();
    generator->readMetaData();
    generator->readContents();
    loaded = true; 
}

void Book::load()
{    
    if(!generator)
        setUpGenerator();
    generator->readMetaData();
    generator->readContents();
    loaded = true; 
}

void Book::setUpGenerator()
{
    generator = new EpubGenerator(filename);

    connect(generator, SIGNAL(setTitle(QString)), this, SLOT(setTitle(QString)));
    connect(generator, SIGNAL(setAuthor(QString)), this, SLOT(setAuthor(QString)));
    connect(generator, SIGNAL(setPublisher(QString)), this, SLOT(setPublisher(QString)));
    connect(generator, SIGNAL(setSubject(QString)), this, SLOT(setSubject(QString)));
    connect(generator, SIGNAL(setDescription(QString)), this, SLOT(setDescription(QString)));
    connect(generator, SIGNAL(setCategory(QString)), this, SLOT(setCategory(QString)));
    connect(generator, SIGNAL(setCopyright(QString)), this, SLOT(setCopyright(QString)));
    connect(generator, SIGNAL(addTOCItem(QString)), this, SLOT(addTOCItem(QString)));
    connect(generator, SIGNAL(addSection(QString*)), this, SLOT(addSection(QString*)));
    connect(generator, SIGNAL(addCoverImage(QPixmap)), this, SLOT(addCoverImage(QPixmap)));
    connect(generator, SIGNAL(addChapter(QString, QString*)), this, SLOT(addChapter(QString, QString*)));
}

void Book::loadMetaData()
{
    if(!generator)
        setUpGenerator();

    generator->readMetaData();
}

bool Book::isLoaded()
{
    return loaded;
}

QString Book::getFirstChapter()
{
    if(!toc.isEmpty())
        return toc.first();

    return QString();
}

QString Book::getNextChapter(QString &current)
{
    if(toc.contains(current))
    {
        int i = toc.indexOf(current);
        i++;
        i = (i<toc.count()) ? i : 0;
        if(document.contains(toc.at(i))){
            current = toc.at(i);
            return *document.value(toc.at(i)); 
        }
    }

    return QString();
}

QString Book::getCover()
{
    if(!sections.isEmpty())
        return *sections.first();
    if(!document.isEmpty())
        return *document.value(toc.first());

    return QString();

}

const QString& Book::getSection(int number)
{
    if(number < sections.count() && number >= 0)
        return *sections.at(number);

    return QString();
}

int Book::getSectionNumber(QString title)
{
    QString *section = document.value(title);    
    if(!section){
        return 0;
    }
    return sections.indexOf(section);
}

const QString& Book::getSectionName(int number)
{
    if(number >= 0 && number < sections.count()){
        QString *section = sections[number];
        QString title = document.key(section);
        return title;
    }

    return QString();
}

QString Book::getNextSection(int &number)
{
    if(!sections.isEmpty()){
        number = (number < sections.count()-1) ? (number+1) : 0;
        return *sections.at(number); 
    }

    return QString();
}

QString Book::getPreviousSection(int &number)
{

    if(!sections.isEmpty()){
        number = (number > 0) ? (number-1) : 0;
        return *sections.at(number); 
    }

    return QString();
}

QString Book::getChapter(QString chapter)
{
    if(!document.isEmpty())
        return *document.value(chapter);

    return QString();
}

QString Book::getBook()
{
    QString ret;
    int i = 0;
    for(QList<QString*>::iterator iter = sections.begin(); iter != sections.end(); ++iter)
    {
        ret += "<div id=section"+QString::number(i)+">";
        ret += *(*iter);
        ret += "</div>";
        i++;
    }
    return ret;
}

QList<QString> Book::getTOC()
{
    return toc;    
}

void Book::setFilename(QString f)
{
    filename = f;
}

QString Book::getFilename()
{
    return filename;
}

const QString& Book::getTitle()
{
    return title;
}


const QString& Book::getAuthor()
{
    return author;
}


const QString& Book::getCategory()
{
    return category;
}

const QPixmap& Book::getCoverImage()
{
    return coverImage;
}

const QString& Book::getSubject()
{
    return subject;
}

const QString& Book::getDescription()
{
    return description;
}

const QString& Book::getPublishDate()
{
    return publishDate;
}

int Book::getNumberOfSections()
{
    return sections.count();
}

unsigned int Book::getLengthInWords()
{
    return lengthInWords;
}

void Book::setLengthInWords(unsigned int length)
{
    lengthInWords = length;
}

BookProgress Book::getProgress()
{
    return bookProgress; 
}

void Book::setProgress(BookProgress p)
{
    bookProgress = p;
}

QList<Bookmark> Book::getBookmarks()
{
    return bookmarks;
}

void Book::addBookmark(Bookmark b)
{
    bookmarks.push_back(b);
}

QList<Annotation> Book::getAnnotations()
{
    return annotations;
}

bool Book::isInLibrary()
{
    return inLibrary;
}

void Book::changeLibraryStatus(bool s)
{
    inLibrary = s;
}

Annotation* Book::getAnnotation(const QString &text)
{

    for(QList<Annotation>::iterator iter = annotations.begin(); iter != annotations.end(); ++iter){
        Annotation a = *iter;
        if(a.getParagraph() == text)
            return &(*iter);
    }

    return 0;
}

void Book::addAnnotation(Annotation a)
{
    annotations.push_back(a);
}

bool Book::hasTOC()
{
    return !toc.isEmpty();
}

QNetworkAccessManager *Book::getResourceManager()
{
    if(generator)
        return static_cast<QNetworkAccessManager*>(generator);

    return NULL;
}

void Book::setTitle(QString s)
{
    title = s;
}

void Book::setAuthor(QString s)
{
    //TODO: A system for cleaning book data.
    if(s.contains("aut: "))s = s.remove("aut: ");
    if(s.contains("Author: "))s = s.remove("Author: ");
    s = s.remove(QRegExp("\\([^)]*\\)"));
    author = s;
}


void Book::setPublisher(QString s)
{
    publisher = s;
}
void Book::setDescription(QString s)
{
    description = s;
}
void Book::setSubject(QString s)
{
    subject = s;
}
void Book::setPublishDate(QString s)
{
    publishDate = s;
}
void Book::setCategory(QString s)
{
    category = s;
}
void Book::setCopyright(QString s)
{
    copyright = s;
}

void Book::addChapter(QString link, QString *chapter)
{
    document.insert(link, chapter);
}

void Book::addTOCItem(QString s){
    toc.push_back(s);
}

void Book::addSection(QString *s)
{
    //If this is the first section try to catch the cover picture
    if(sections.count() == 0 && coverImage.isNull()){
//        QString str("<img src=\"images/cover.png\" alt=\"Cover\" style=\"height: 100%; max-width: 100%;\" />");
        QRegExp rx("<img[^>]+>");
        int pos = rx.indexIn(*s); 
        if(pos > -1){
            QString img = rx.cap(0);
            QRegExp rx2("src=['\"a-z0-9_./-]+");
            pos = rx2.indexIn(img);
            if(pos > -1){
                img = rx2.cap(0);
                img = img.remove("src=");
                img = img.remove("\"");
                img = img.remove("'");
                img = img.remove("../");
                QPixmap cover;
                int size = 0;
                char *data = generator->readFile(img, size);
                if(data && size != -1){
                    cover.loadFromData(reinterpret_cast<unsigned char*>(data), size);
                    addCoverImage(cover);
                }
            }
        }
    }
    sections.push_back(s);    
}

void Book::addCoverImage(QPixmap cover)
{
    coverImage = cover;
    emit coverImageAdded();
}
