#include "bookview.h"

#include <QDebug>
#include <QAction>
#include <QWebElement>
#include <QWebFrame>
#include <QMouseEvent>
#include <QWebSettings>
#include <QApplication>
#include <QDesktopWidget>
#include <QAbstractKineticScroller>
#include <QFontMetrics>


BookView::BookView(QWidget *parent) : QWebView(parent)
{
    setMouseTracking(false); 
    viewMode = ScrollingSectionMode;

    snapEnabled = true;

    setCSSSetting("white-space", "normal");
    setHTMLAttribute("width", "450");
}

BookView::~BookView()
{
}
//TODO: Handle invalid book data. Fix screen clutter when changing books.
void BookView::setBook(Book *openBook)
{
    //book = openBook;
    if(openBook){
        
        QRect screenGeometry = QApplication::desktop()->screenGeometry();
        if(screenGeometry.width() < screenGeometry.height())
            page()->setPreferredContentsSize(QSize(480, 800));
        else
            page()->setPreferredContentsSize(QSize(800, 480));

    
        book = openBook;
        if(!book->isLoaded())
            book->load();
        //Clear cache so all images get reloaded. If cache is not cleared we might end up having wrong cover image etc.
        QWebSettings *settings = page()->settings()->globalSettings();
        settings->clearMemoryCaches();

        //Using custom QNetworkAccessManager so css and images load properly.
        QNetworkAccessManager *manager = book->getResourceManager();    
        if(page()->networkAccessManager() != manager){ 
            if(manager){
                page()->setNetworkAccessManager(manager);
            }
        }

        QString content;

        if(book->getProgress().isValid()){
            if(viewMode==ScrollingSectionMode){
                content = book->getSection(book->getProgress().getLink().toInt());        
                currentSection = book->getProgress().getLink().toInt();
                connect(this, SIGNAL(loadFinished(bool)), this, SLOT(loadingPageFinished(bool)));
            }else if(viewMode==ScrollingBookMode){
                //TODO:Scroll to right spot.
            } else if(viewMode==PageMode){

            }
        }else if(viewMode == ScrollingBookMode){ 
            content =  book->getBook();
        } else if(viewMode == ScrollingSectionMode || viewMode== PageMode){
            content = book->getCover();
            currentSection = 0;
        }

        if(!content.isEmpty()){
            setContent(content);
            //reload();
        }


    }
}

void BookView::goToSection(int section)
{ 
    if(viewMode == ScrollingBookMode){
        QWebElement element = page()->mainFrame()->findFirstElement("#section"+QString::number(section));
        int y  = page()->mainFrame()->scrollPosition().y() - element.geometry().y();
        page()->mainFrame()->scroll(0, -y);
    } else if(viewMode == ScrollingSectionMode || viewMode == PageMode){
        setContent(book->getSection(section));
        currentSection = section;
    }
}

void BookView::goToChapter(QString chapter)
{
    goToSection(book->getSectionNumber(chapter));
}

void BookView::setFont(const QString &family, int size)
{
    
    QWebSettings *settings = page()->settings()->globalSettings();
    if(!family.isEmpty()){
        settings->setFontFamily(QWebSettings::SerifFont, family);
        settings->setFontFamily(QWebSettings::SansSerifFont, family);
        settings->setFontFamily(QWebSettings::StandardFont, family);
    }
    if(size != 0)
        settings->setFontSize(QWebSettings::MinimumFontSize, size);
}

void BookView::setNightMode(bool toggle)
{
    if(toggle){
        setCSSSetting("color", "#FFFFFF");
        setCSSSetting("border-color", "#FFFFFF");
        setCSSSetting("background-color", "#181818");
    }
    else{
        setCSSSetting("color", "#000000");
        setCSSSetting("border-color", "#000000");
        setCSSSetting("background-color", "#ececec");
    }
    
}

void BookView::mousePressEvent(QMouseEvent *event)
{
    if(viewMode == PageMode || viewMode == ScrollingSectionMode)
        swipeStart = event->pos();    
}

void BookView::mouseMoveEvent(QMouseEvent *event)
{
    if(viewMode == ScrollingSectionMode){
       if(!swipeStart.isNull() && event->pos().x() > 0){
            if(swipeStart.x() - event->pos().x() > 100){
                QString chapter = book->getNextSection(currentSection);
                setContent(chapter);
                swipeStart = QPoint(0, 0);
            } else if(swipeStart.x() - event->pos().x() < -100){
                QString chapter = book->getPreviousSection(currentSection);
                setContent(chapter);
                swipeStart = QPoint(0, 0);
            }
       }

   } else if(viewMode == PageMode)
   {
       if(!swipeStart.isNull() && event->pos().x() > 0){
            if(swipeStart.x() - event->pos().x() > 100){
                currentPage++;
                setPage(pages[currentPage]);
                swipeStart = QPoint(0, 0);
            } else if(swipeStart.x() - event->pos().x() < -100){
                currentPage--;
                setPage(pages[currentPage]);
                swipeStart = QPoint(0, 0);
            }
       }
   }
}

//TODO: Snap between lines.
void BookView::volumeUpPressed(){
   if(snapEnabled){
        previousOffset = page()->mainFrame()->scrollPosition();
        int max = page()->mainFrame()->contentsSize().height() - page()->viewportSize().height();
        if(previousOffset.y() < max){
            int jump = page()->viewportSize().height();

            page()->mainFrame()->setScrollPosition(previousOffset + QPoint(0, jump));
        }else{
            QString chapter = book->getNextSection(currentSection);
            setContent(chapter);

        }
   }
}

void BookView::volumeDownPressed()
{
   if(snapEnabled){
        previousOffset = page()->mainFrame()->scrollPosition();
        if(previousOffset.y() != 0){
            int jump = page()->viewportSize().height();
            page()->mainFrame()->setScrollPosition(previousOffset + QPoint(0, -jump));
        }else{
            QString chapter = book->getPreviousSection(currentSection);
            setContent(chapter);
        }
   }
}



QList<QWebElement> BookView::hitElements(const QLine &line)
{
    QList<QWebElement> hitElements;

    const int points = 30;
    QPoint delta(line.dx() / points, line.dy() / points);
    
    QPoint point = line.p1();
    for(int i = 0; i < points -1; ++i)
    {
        point += delta;
        QWebHitTestResult hit = page()->currentFrame()->hitTestContent(point);
        if(!hit.element().isNull())
            hitElements.push_back(hit.element());
    }

    return hitElements;
}

void BookView::setContent(QString content)
{
    parseLinks(content);
    //For some reason a dummy url has to used or else the custom QNetworkAccessManager won't get any calls.
    if(viewMode == PageMode){
        generatePages(content);
    }else{
        setHtml(content, QUrl(dummyURL));
        makeDOMChangesEffective();
    }
}

//Urls in text cause trouble because they can't be reflowed
//TODO: Replace removed url with a link
void BookView::parseLinks(QString &content)
{
    content = content.replace(QRegExp("[\\[\\s:]\\http://[\\]/#.a-z0-9]*"), "<a href=#>(link)</a>");
}



void BookView::setCSSSetting(QString setting, QString value)
{
    cssSettings.insert(setting, value);
}

void BookView::setHTMLAttribute(QString attribute, QString value)
{
    htmlAttributes.insert(attribute, value);
}

void BookView::makeDOMChangesEffective()
{
    QWebElement documentElement = page()->mainFrame()->documentElement();
    //Change DOM
    setDOMElementSettings(documentElement);
//    page()->mainFrame()->setHtml(documentElement.toOuterXml(), QUrl(dummyURL));
    update();
}

void BookView::setDOMElementSettings(QWebElement &parentElement)
{
    QWebElement element = parentElement.firstChild(); 
    while(!element.isNull())
    {
        //CSS
        for(QHash<QString, QString>::iterator iter = cssSettings.begin(); iter != cssSettings.end(); ++iter){
            element.setStyleProperty(iter.key(), (*iter));
            element.setStyleProperty("line-height", "1.5em");
/*            if(element.tagName() == "P")
                element.setStyleProperty("margin", "0px");
            else{
                element.setStyleProperty("margin", "0px");
                element.setStyleProperty("padding", "0px");
            }*/
        }
        margins.insert(element.tagName(), element.styleProperty("margin", QWebElement::CascadedStyle));
        linewidths.insert(element.tagName(), element.styleProperty("line-height", QWebElement::CascadedStyle));

        //HTML attributes
        for(QHash<QString, QString>::iterator iter = htmlAttributes.begin(); iter != htmlAttributes.end(); ++iter){
            if(element.hasAttribute(iter.key()))
                element.setAttribute(iter.key(), *iter);
        }
        setDOMElementSettings(element);
        element = element.nextSibling();
    }
}

void BookView::parseContent(QWebElement &parentElement, QStringList spots)
{

    QWebElement element = parentElement.firstChild(); 
    while(!element.isNull())
    {
        QString xml = element.toInnerXml();
        for(QStringList::const_iterator iter = spots.constBegin(); iter != spots.constEnd(); ++iter){ 
            if(xml.contains(*iter)){
                QStringList list = xml.split(*iter);
                list.insert(1, "<br><br><br>");
                xml = list.join("joinspot");
            }
        }
        element.setInnerXml(xml);
        element = element.nextSibling();
    }
}

void BookView::generatePages(QString content)
{
    setHtml(content, QUrl(dummyURL));
    makeDOMChangesEffective();
    QWebPage *section = page();
    qDebug() << "Content height: " << section->mainFrame()->contentsSize().height();
    qDebug() << "Page height: " << section->preferredContentsSize().height();
    if(section->mainFrame()->contentsSize().height() > section->preferredContentsSize().height()){
        int contentHeight = section->mainFrame()->contentsSize().height();
        int pageHeight = section->preferredContentsSize().height();

        int numPages = contentHeight / pageHeight;
        pages.clear();

        /*QWebElement documentElement = section->mainFrame()->documentElement();
        QWebElement bodyelement = documentElement.findFirst("body");
        QWebElementCollection children = bodyelement.findAll("p");
        QStringList paragraphs;
        for(QWebElementCollection::const_iterator iter = children.constBegin(); iter != children.constEnd(); ++iter){
            paragraphs.push_back((*iter).toOuterXml());
        }
        //Get the element that holds the text.
        QWebElement firstElement  = bodyelement;
        QWebElement element;
        while(!firstElement.isNull()){
            element = firstElement;
            firstElement = firstElement.firstChild();

        }

        QWebElement mainElement = children.first().parent();
        mainElement.removeAllChildren();
        qDebug() << "Main: " << mainElement.toOuterXml();

        QWebPage *newPage = new QWebPage();
        //newPage->setPreferredContentsSize(QSize(480, 800));
        qDebug() << "contentheight: " << newPage->mainFrame()->contentsSize().height();
        //int over = newPage->mainFrame()->contentsSize().height() - 800;
        QWebSettings *settings = page()->settings()->globalSettings();
        QFontMetrics metrics(QFont(settings->fontFamily(QWebSettings::StandardFont), QWebSettings::MinimumFontSize));
         
        qDebug() << "Margins: " << margins;
        qDebug() << "Line-heights: " << linewidths;
        int fontHeight = settings->fontSize(QWebSettings::MinimumFontSize);
        int margins = 0;
        int lineHeight = 2*fontHeight;

        int rows = (800-margins) / lineHeight;
        int charPerRow = (480 - margins) / (fontHeight / 2);
        int charPerPage = rows * charPerRow;

        qDebug() << "Max rows: " << rows;
        qDebug() << "Line height: " << lineHeight;
        qDebug() << "Font: " << fontHeight;
        qDebug() << "Margins: " << margins;

        QStringList pagesAsString;
        for(QStringList::iterator iter = paragraphs.begin(); iter != paragraphs.end(); ++iter){
            QString para = *iter;
//            qDebug() << "Rect needed: " << metrics.boundingRect(0, 0, 480, 800, Qt::AlignJustify, para);
            //if(para.size()>charPerPage)
           // pagesAsString.push_back(para.left(charPerPage)+"</p>");
            //else
               pagesAsString.push_back(para);
        }
        for(QStringList::iterator iter = pagesAsString.begin(); iter != pagesAsString.end(); ++iter)
        {
            QWebPage *wp = new QWebPage();
            wp->mainFrame()->setHtml(documentElement.toOuterXml(), QUrl(dummyURL));
            QWebElement doc = wp->mainFrame()->documentElement();
            QWebElement me = doc.findFirst("body");
            QWebElement firstElement2  = me;
            QWebElement element2;
            while(!firstElement2.isNull()){
                element2 = firstElement2;
                firstElement2 = firstElement2.firstChild();

            }
            element2.parent().appendInside(*iter);
            pages.push_back(wp);
        }*/
        QWebElement documentElement = section->mainFrame()->documentElement();
        QStringList list = documentElement.toPlainText().split(" ");
        int numWords = list.count();
        int numWordsOnPage = numWords / numPages;
        qDebug() << "NumWordsOnPage: " << numWordsOnPage;

        QList<QString> chopPoints;

        for(int i = 1; i < numPages; ++i){
            int take = i*numWordsOnPage;
            if(take < list.count()){
                chopPoints.push_back(list[take-2]+" "+list[take-1]+" "+list[take]); 
            }
        }
        parseContent(documentElement, chopPoints);
                /*pageAsString.remove(footer);
                int rectHeight = 0;
                QStringList pageStringList = pageAsString.split(" ");
                QString remove = pageStringList.last();
                pageStringList.pop_back();
                pageAsString.remove(pageAsString.size()-1, 1);
                qDebug() << "over: " << over << " font height: " << metrics.height();
                while(rectHeight < over)
                {
                    QRect r = metrics.boundingRect(QRect(0, 0, 480, over), Qt::AlignLeft, remove);
                    rectHeight = r.height();
                    if(!pageStringList.isEmpty())
                        remove.prepend(" " + pageStringList.last());
                    else break;
                    pageStringList.pop_back();
                    qDebug() << "over: " << over << " rectheight: " << rectHeight;
                    qDebug() << "remove: " << remove;

                }*/
                /*content.prepend(remove);
                pageAsString = pageStringList.join(" ");
                pageAsString.append(footer);*/
//                newPage->mainFrame()->setHtml(pageAsString, QUrl(dummyURL));
            //    pageList.push_back(content.truncate(pos));
           // }
//        }
/*        int startFrom = 0;
        for(int i = 0; i < numPages; ++i){
            int pageFull = 0;
//            QWebSettings *settings = page()->settings()->globalSettings();
//            settings->clearMemoryCaches();
            QWebPage *newPage = new QWebPage(this);
            newPage->mainFrame()->setHtml(content, QUrl(dummyURL));
            QWebElement documentElement = newPage->mainFrame()->documentElement();
            QWebElement child = documentElement.findFirst("h3");
            QWebElement parent = child.parent();
            int counter = 0;
            while(!child.isNull()){
                qDebug() << "Counter: " << counter << " StartFrom: " << startFrom;
                QWebElement nextChild = child.nextSibling();
                if(child.geometry().height() != 0){
                    if(pageFull < pageHeight && counter > startFrom){
//                        qDebug() << "Adding child : " << child.localName() << " " << child.geometry().height();
                        pageFull += child.geometry().height(); 
                        counter++;
                        startFrom = counter;
                    }
                    else {
                        child.removeFromDocument();
                        counter++;
 //                       qDebug() << "Removing child: " << child.localName();
                    }
                    
                }
                child = nextChild;
            }

            pages.push_back(newPage);
            
        }*/

        if(!pages.isEmpty())
            setPage(pages.first());
        currentPage = 0;
        qDebug() << "Number of pages" << numPages;
    }
}

//Progress saved by sectionnumber, element and scrollposition.
void BookView::saveProgress()
{
    QList<QWebElement> elements = hitElements(QLine(1, 1, page()->viewportSize().width() -1, page()->viewportSize().height()));
    QString link;
    if(viewMode == ScrollingSectionMode)
        link = QString::number(currentSection);

    QString element;
    QString text;
    if(!elements.isEmpty()){
        QWebElement webElement;
        for(QList<QWebElement>::iterator iter = elements.begin(); iter != elements.end(); ++iter){
            if(!(*iter).isNull() && (*iter).localName() != "html" && (*iter).localName() != "body"){
                webElement = (*iter);
                break;
            }
        }
        if(!webElement.classes().isEmpty())
            element = webElement.classes().first();
        else
            element = webElement.localName();
        text = webElement.toPlainText();
        QStringList tlist = text.split(" ");
        if(tlist.count() > 3)
            text = tlist[0] + tlist[1] + tlist[2];

    }

    int scrollPosition = page()->mainFrame()->scrollPosition().y();

    BookProgress progress;
    progress.setProgress(link, element, text, scrollPosition);
    book->setProgress(progress);

}

void BookView::loadingPageFinished(bool ok)
{
    QAbstractKineticScroller *scroller = property("kineticScroller").value<QAbstractKineticScroller*>();
    scroller->scrollTo(QPoint(0, book->getProgress().getScroll()));
    disconnect(this, SIGNAL(loadFinished(bool)), this, SLOT(loadingPageFinished(bool)));
}
