#include "book.h"

#include <QPainter>
#include <QDebug>
Book::Book(const QString &bookFilename, bool loadLater)
{
    generator = new EpubGenerator(bookFilename);

    connect(generator, SIGNAL(setTitle(QString)), this, SLOT(setTitle(QString)));
    connect(generator, SIGNAL(setAuthor(QString)), this, SLOT(setAuthor(QString)));
    connect(generator, SIGNAL(setPublisher(QString)), this, SLOT(setPublisher(QString)));
    connect(generator, SIGNAL(setSubject(QString)), this, SLOT(setSubject(QString)));
    connect(generator, SIGNAL(setDescription(QString)), this, SLOT(setDescription(QString)));
    connect(generator, SIGNAL(setCategory(QString)), this, SLOT(setCategory(QString)));
    connect(generator, SIGNAL(setCopyright(QString)), this, SLOT(setCopyright(QString)));
    connect(generator, SIGNAL(addTOCItem(QString)), this, SLOT(addTOCItem(QString)));
    connect(generator, SIGNAL(addSection(QString*)), this, SLOT(addSection(QString*)));
    connect(generator, SIGNAL(addCoverImage(QPixmap)), this, SLOT(addCoverImage(QPixmap)));
    connect(generator, SIGNAL(addChapter(QString, QString*)), this, SLOT(addChapter(QString, QString*)));

    loaded = false;
    filename = bookFilename;
    if(!loadLater)
        openBook(filename);
    else
        generator->readMetaData();


}

Book::~Book()
{
    delete generator;
    //TODO: delete book data.
}

void Book::openBook(const QString &bookFilename)
{
    generator->readMetaData();
    generator->readContents();
    loaded = true; 
}

void Book::load()
{    
    generator->readContents();
    loaded = true; 
}

bool Book::isLoaded()
{
    return loaded;
}

QString Book::getFirstChapter()
{
    if(!toc.isEmpty())
        return toc.first();

    return QString();
}

QString Book::getNextChapter(QString &current)
{
    if(toc.contains(current))
    {
        int i = toc.indexOf(current);
        i++;
        i = (i<toc.count()) ? i : 0;
        if(document.contains(toc.at(i))){
            current = toc.at(i);
            return *document.value(toc.at(i)); 
        }
    }

    return QString();
}

QString Book::getCover()
{
    if(!sections.isEmpty())
        return *sections.first();
    if(!document.isEmpty())
        return *document.value(toc.first());

    return QString();

}

QString Book::getSection(int number)
{
    if(sections.count() > number && number >= 0)
        return *sections.at(number);

    return QString();
}

int Book::getSectionNumber(QString title)
{
    QString *section = document.value(title);    
    if(!section){
        return 0;
    }
    return sections.indexOf(section);
}

QString Book::getNextSection(int &number)
{
    if(!sections.isEmpty()){
        number = (number < sections.count()-1) ? (number+1) : 0;
        return *sections.at(number); 
    }

    return QString();
}

QString Book::getPreviousSection(int &number)
{

    if(!sections.isEmpty()){
        number = (number > 0) ? (number-1) : 0;
        return *sections.at(number); 
    }

    return QString();
}

QString Book::getChapter(QString chapter)
{
    if(!document.isEmpty())
        return *document.value(chapter);

    return QString();
}

QString Book::getBook()
{
    QString ret;
    int i = 0;
    for(QList<QString*>::iterator iter = sections.begin(); iter != sections.end(); ++iter)
    {
        ret += "<div id=section"+QString::number(i)+">";
        ret += *(*iter);
        ret += "</div>";
        i++;
    }
    return ret;
}

QList<QString> Book::getTOC()
{
    return toc;    
}

QString Book::getFilename()
{
    return filename;
}

QString Book::getTitle()
{
    return title;
}


QString Book::getAuthor()
{
    return author;
}


QString Book::getCategory()
{
    return category;
}

QPixmap Book::getCoverImage()
{
    return coverImage;
}

BookProgress Book::getProgress()
{
    return bookProgress; 
}

void Book::setProgress(BookProgress p)
{
    bookProgress = p;
}

bool Book::hasTOC()
{
    return !toc.isEmpty();
}

QNetworkAccessManager *Book::getResourceManager()
{
    if(generator)
        return static_cast<QNetworkAccessManager*>(generator);

    return NULL;
}

void Book::setTitle(QString s)
{
    title = s;
}

void Book::setAuthor(QString s)
{
    //TODO: A system for cleaning book data.
    if(s.contains("aut: "))s = s.remove("aut: ");
    if(s.contains("Author: "))s = s.remove("Author: ");
    s = s.remove(QRegExp("\\([^)]*\\)"));
    author = s;
}


void Book::setPublisher(QString s)
{
    publisher = s;
}
void Book::setDescription(QString s)
{
    description = s;
}
void Book::setSubject(QString s)
{
    subject = s;
}
void Book::setPublishDate(QString s)
{
    //TODO: Save date to a QString, impossible to know what format date is in epub files.
    publishDate = QDate::fromString(s, "ddMMyy");
}
void Book::setCategory(QString s)
{
    category = s;
}
void Book::setCopyright(QString s)
{
    copyright = s;
}

void Book::addChapter(QString link, QString *chapter)
{
    document.insert(link, chapter);
}

void Book::addTOCItem(QString s){
    toc.push_back(s);
}

void Book::addSection(QString *s)
{
    sections.push_back(s);    
}

void Book::addCoverImage(QPixmap cover)
{
    coverImage = cover;
}
