/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "FunctionInput.h"

#include <Lum/Base/String.h>

#include <Lum/Button.h>
#include <Lum/Grid.h>
#include <Lum/Panel.h>
#include <Lum/Tab.h>

FunctionInput::FunctionInput()
 : backspaceAction(new Lum::Model::Action()),
   clearAction(new Lum::Model::Action())
{
  Observe(backspaceAction);
  Observe(clearAction);
}

void FunctionInput::SetString(Lum::String *string)
{
  this->string=string;
}

void FunctionInput::SetCalculateAction(Lum::Model::Action* action)
{
  calculateAction=action;
}

Lum::Object* FunctionInput::CreateButton(const std::wstring& label, Lum::Model::Action *action) const
{
  return Lum::Button::Create(label,action,true,true);
}

Lum::Object* FunctionInput::CreateTextButton(const std::wstring& label, const std::wstring& text)
{
  Lum::Model::ActionRef action;

  action=new Lum::Model::Action();

  Observe(action);

  if (text.empty()) {
    actionTextMap[action.Get()]=label;
  }
  else {
    actionTextMap[action.Get()]=text;
  }

  return CreateButton(label,action);
}

void FunctionInput::CalcSize()
{
  Lum::Grid   *grid;
  Lum::HPanel *panel;
  Lum::Tab    *tab;

  panel=Lum::HPanel::Create(true,true);

  tab=Lum::Tab::Create(false,true);

  grid=new Lum::Grid();
  grid->SetSize(2,1);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"\x03c0"/*L"pi"*/,L"pi"));
  grid->SetObject(1,0,CreateTextButton(L"e"));
  tab->Add(L"Const.",grid);

  grid=new Lum::Grid();
  grid->SetSize(3,2);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"^"));
  grid->SetObject(1,0,CreateTextButton(L"ln",L"ln("));
  grid->SetObject(2,0,CreateTextButton(L"lg",L"lg("));

  grid->SetObject(0,1,CreateTextButton(L"\x221a"/*L"sqrt"*/,L"sqrt("));
  grid->SetObject(1,1,CreateTextButton(L"abs",L"abs("));

  tab->Add(L"F(x)",grid);

  grid=new Lum::Grid();
  grid->SetSize(3,4);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"asinh",L"asinh("));
  grid->SetObject(1,0,CreateTextButton(L"acosh",L"acosh("));
  grid->SetObject(2,0,CreateTextButton(L"atanh",L"atanh("));

  grid->SetObject(0,1,CreateTextButton(L"sinh",L"sinh("));
  grid->SetObject(1,1,CreateTextButton(L"cosh",L"cosh"));
  grid->SetObject(2,1,CreateTextButton(L"tanh",L"tanh"));

  grid->SetObject(0,2,CreateTextButton(L"asin",L"asin("));
  grid->SetObject(1,2,CreateTextButton(L"acos",L"acos("));
  grid->SetObject(2,2,CreateTextButton(L"atan",L"atan("));

  grid->SetObject(0,3,CreateTextButton(L"sin",L"sin("));
  grid->SetObject(1,3,CreateTextButton(L"cos",L"cos("));
  grid->SetObject(2,3,CreateTextButton(L"tan",L"tan("));

  tab->Add(L"Trig.",grid);

  panel->Add(tab);

  panel->AddSpace(true);
  grid=new Lum::Grid();
  grid->SetSize(4,5);
  grid->SetEqualDimensions(true,true);
  grid->SetSpace(true,true);

  grid->SetObject(0,0,CreateTextButton(L"("));
  grid->SetObject(1,0,CreateTextButton(L")"));
  grid->SetObject(2,0,CreateButton(L"<-",backspaceAction));
  grid->SetObject(3,0,CreateButton(L"Clr",clearAction));

  grid->SetObject(0,1,CreateTextButton(L"7"));
  grid->SetObject(1,1,CreateTextButton(L"8"));
  grid->SetObject(2,1,CreateTextButton(L"9"));
  grid->SetObject(3,1,CreateTextButton(L"/"));

  grid->SetObject(0,2,CreateTextButton(L"4"));
  grid->SetObject(1,2,CreateTextButton(L"5"));
  grid->SetObject(2,2,CreateTextButton(L"6"));
  grid->SetObject(3,2,CreateTextButton(L"*"));

  grid->SetObject(0,3,CreateTextButton(L"1"));
  grid->SetObject(1,3,CreateTextButton(L"2"));
  grid->SetObject(2,3,CreateTextButton(L"3"));
  grid->SetObject(3,3,CreateTextButton(L"\x2212",L"-"));

  grid->SetObject(0,4,CreateTextButton(L"0"));
  grid->SetObject(1,4,CreateTextButton(Lum::Base::GetDecimalPointWString()));
  if (calculateAction.Valid()) {
    grid->SetObject(2,4,CreateButton(L"=",calculateAction));
  }
  else {
    grid->SetObject(2,4,CreateTextButton(L"x"));
  }
  grid->SetObject(3,4,CreateTextButton(L"+"));

  panel->Add(grid);

  container=panel;

  Component::CalcSize();
}

void FunctionInput::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==backspaceAction && backspaceAction->IsFinished()) {
    string->ActionBackspace();
  }
  else if (model==clearAction && clearAction->IsFinished()) {
    string->ActionEraseAll();
  }
  else if (dynamic_cast<Lum::Model::Action*>(model)!=NULL &&
           dynamic_cast<Lum::Model::Action*>(model)->IsFinished()) {
    std::map<Lum::Model::Action*,std::wstring>::const_iterator iter;

    iter=actionTextMap.find(dynamic_cast<Lum::Model::Action*>(model));
    if (iter!=actionTextMap.end()) {
      string->ActionInsertString(iter->second);
    }
  }

  Component::Resync(model,msg);
}

