/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Configuration.h"

#include <iostream>

#include <Lum/Base/Path.h>
#include <Lum/Base/String.h>

#include <Lum/Config/Config.h>

#include <Lum/Manager/FileSystem.h>

std::vector<Unit*>    units;
std::vector<Formula*> formulas;

std::map<std::wstring,Unit*> unitMap;

Lum::Model::ULongRef calcTrigoMode=new Lum::Model::ULong(1);

Alternative::Alternative(const std::wstring& name,
                         const std::wstring& symbol,
                         const std::wstring& from,
                         const std::wstring& to)
 : name(name),
   symbol(symbol),
   from(from),
   to(to)
{
  // no code
}

std::wstring Alternative::GetName() const
{
  return name;
}

std::wstring Alternative::GetSymbol() const
{
  return symbol;
}

std::wstring Alternative::GetFrom() const
{
  return from;
}

std::wstring Alternative::GetTo() const
{
  return to;
}

Unit::Unit(const std::wstring& type)
 : type(type)
{
  // no code
}

std::wstring Unit::GetType() const
{
  return type;
}

size_t Unit::GetAlternativeCount() const
{
  return alternatives.size();
}

Alternative* Unit::GetAlternative(size_t index) const
{
  assert(index<alternatives.size());

  return alternatives[index];
}

void Unit::AddAlternative(Alternative* alternative)
{
  alternatives.push_back(alternative);
}

Variable::Variable(const std::wstring& name,
                   const std::wstring& unit,
                   const std::wstring& description)
 : name(name),
   unit(unit),
   description(description)
{
  // no code
}

std::wstring Variable::GetName() const
{
  return name;
}

std::wstring Variable::GetDescription() const
{
  return description;
}

std::wstring Variable::GetUnit() const
{
  return unit;
}

Formula::Formula(const std::wstring& name,
                 const std::wstring& category,
                 const std::wstring& unit,
                 const std::wstring& formula,
                 const std::wstring& description)
 : name(name),
   category(category),
   unit(unit),
   formula(formula),
   description(description)
{
  // no code
}

std::wstring Formula::GetName() const
{
  return name;
}

std::wstring Formula::GetCategory() const
{
  return category;
}

std::wstring Formula::GetFormula() const
{
  return formula;
}

std::wstring Formula::GetDescription() const
{
  return description;
}

std::wstring Formula::GetUnit() const
{
  return unit;
}

void Formula::AddVariable(Variable* variable)
{
  variables.push_back(variable);
}

const std::vector<Variable*>& Formula::GetVariables() const
{
  return variables;
}

bool LoadConfig(std::list<ConfigPlugin*>& plugins)
{
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path(Lum::Base::Path::GetApplicationConfigPath());

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"MathJinni") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid config file!" << std::endl;
    delete top;
    return false;
  }

  for (std::list<ConfigPlugin*>::iterator iter=plugins.begin(); iter!=plugins.end(); ++iter) {
    (*iter)->LoadConfig(top);
  }

  delete top;

  return true;
}

bool SaveConfig(std::list<ConfigPlugin*>& plugins)
{
  Lum::Config::Node *top;
  Lum::Base::Path   path(Lum::Base::Path::GetApplicationConfigPath());
  std::wstring      config;
  bool              res;

  top=new Lum::Config::Node();
  top->SetName(L"MathJinni");

  for (std::list<ConfigPlugin*>::iterator iter=plugins.begin(); iter!=plugins.end(); ++iter) {
    (*iter)->StoreConfig(top);
  }

  Lum::Base::Status status;

  status=path.CreateDirRecursive();

  if (!status) {
    std::cerr << "Cannot create config directory '" << Lum::Base::WStringToString(path.GetDir()) << "': " << Lum::Base::WStringToString(status.GetDescription()) << std::endl;
    return false;
  }

  res=Lum::Config::SaveConfigToXMLFile(path.GetPath(),top);

  delete top;

  return res;
}

bool LoadData(std::list<ConfigPlugin*>& plugins)
{
  std::wstring           appDataDir;
  Lum::Config::Node      *top;
  Lum::Config::ErrorList errors;
  Lum::Base::Path        path;

  if (!Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::appDataDir,
                                                      appDataDir)) {
    return false;
  }

  path.SetNativeDirAndFile(appDataDir,L"data.xml");

  top=Lum::Config::LoadConfigFromXMLFile(path.GetPath(),errors);

  if (top==NULL) {
    top=Lum::Config::LoadConfigFromXMLFile(L"data/data.xml",errors);
  }

  if (top==NULL) {
    return false;
  }

  if (top->GetName()!=L"MathJinni-Data") {
    std::cerr << "'" << Lum::Base::WStringToString(path.GetPath()) << "' is a valid data file!" << std::endl;
    delete top;
    return false;
  }

  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    Lum::Config::Node *node=*iter;

    if (node->GetName()==L"unit") {
      std::wstring type;

      if (!node->GetAttribute(L"type",type)) {
        std::cerr << "unit node is missing attribute 'type'" << std::endl;
        continue;
      }

      Unit *unit=new Unit(type);

      for (Lum::Config::Node::NodeList::const_iterator iter2=node->GetChildren().begin();
           iter2!=node->GetChildren().end();
           ++iter2) {
        Lum::Config::Node *node=*iter2;

        if (node->GetName()==L"alternative") {
          std::wstring name;
          std::wstring symbol;
          std::wstring from;
          std::wstring to;

          if (!(node->GetAttribute(L"name",name) &&
                node->GetAttribute(L"symbol",symbol) &&
                node->GetAttribute(L"from",from) &&
                node->GetAttribute(L"to",to))) {
            std::cerr << "type/alternative node is missing attribute 'name', 'symbol', 'from' or 'to'" << std::endl;
            break;
          }

          unit->AddAlternative(new Alternative(name,symbol,from,to));
        }
      }

      units.push_back(unit);
      unitMap[unit->GetType()]=unit;
    }
    else if (node->GetName()==L"formula") {
      std::wstring name;
      std::wstring category;
      std::wstring unit;
      std::wstring formula;
      std::wstring description;

      if (!(node->GetAttribute(L"name",name) &&
            node->GetAttribute(L"category",category) &&
            node->GetAttribute(L"unit",unit) &&
            node->GetAttribute(L"formula",formula) &&
            node->GetAttribute(L"description",description))) {
        std::cerr << "formula node is missing attribute 'name', 'category', 'unit', 'formula' or 'description'" << std::endl;
        continue;
      }

      Formula *f=new Formula(name,category,unit,formula,description);

      for (Lum::Config::Node::NodeList::const_iterator iter2=node->GetChildren().begin();
           iter2!=node->GetChildren().end();
           ++iter2) {
        Lum::Config::Node *node=*iter2;

        if (node->GetName()==L"variable") {
          std::wstring name;
          std::wstring unit;
          std::wstring description;

          if (!(node->GetAttribute(L"name",name) &&
                node->GetAttribute(L"unit",unit) &&
                node->GetAttribute(L"description",description))) {
            std::cerr << "formula/variable node is missing attribute 'name', 'unit' or 'description'" << std::endl;
            break;
          }

          f->AddVariable(new Variable(name,unit,description));
        }
      }

      formulas.push_back(f);
    }
  }

  delete top;

  for (std::list<ConfigPlugin*>::iterator iter=plugins.begin(); iter!=plugins.end(); ++iter) {
    (*iter)->LoadData();
  }

  return true;
}

