/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Plotter.h"

#include <math.h>

#include "Util.h"

Functions::Function::Function(const std::wstring& function, Parser::Expression *expression)
 : function(function),
   expression(expression)
{
  // no code
}

Functions::Functions(Parser::Variable* plotVar)
 : plotVar(plotVar),
   top(10.0),
   bottom(-10.0),
   left(-10.0),
   right(10.0)
{
  // no code
}

size_t Functions::GetRows() const
{
  return functions.size();
}

size_t Functions::GetColumns() const
{
  return 1;
}

std::wstring Functions::GetString(size_t column, size_t row) const
{
  assert(column==1);
  assert(row>0 && row<=functions.size());

  return functions[row-1].function;
}

Lum::Object* Functions::GetObject(size_t column, size_t row) const
{
  return NULL;
}

Parser::Variable* Functions::GetPlotVar() const
{
  return plotVar;
}

Parser::Expression* Functions::GetExpression(size_t row) const
{
  return functions[row-1].expression;
}

std::vector<double>& Functions::GetValue(size_t row)
{
return functions[row-1].values;
}

void Functions::AddFunction(const std::wstring& function, Parser::Expression* expression)
{
  Function f(function,expression);

  functions.push_back(f);

  NotifyInsert(functions.size(),1);
}

void Functions::SetFunction(size_t row, const std::wstring& function, Parser::Expression* expression)
{
  assert(row>=1 && row<=functions.size());

  functions[row-1].function=function;
  functions[row-1].expression=expression;
  functions[row-1].values.clear();

  RedrawRow(row);
}

void Functions::RemoveFunction(size_t row)
{
  assert(row>=1 && row<=functions.size());

  delete functions[row-1].expression;
  functions.erase(functions.begin()+row-1);;

  NotifyDelete(row,1);
}

void Functions::SetRegion(double top, double bottom,double left, double right)
{
  this->top=top;
  this->bottom=bottom;
  this->left=left;
  this->right=right;

  Notify();
}

double Functions::GetTop() const
{
  return top;
}

double Functions::GetBottom() const
{
  return bottom;
}

double Functions::GetLeft() const
{
  return left;
}

double Functions::GetRight() const
{
  return right;
}

double exponent(double x)
{
  if (x>=0) {
    return floor(log10(x));
  }
  else {
    return floor(log10(-x));
  }
}

Plotter::Plotter()
: top(0.0),
  bottom(0.0),
  left(0.0),
  right(0.0),
  areaSelection(false),
  backgroundColor(1,1,1,Lum::OS::Display::whiteColor),
  gridAColor(0.5,0.5,0.5,Lum::OS::Display::graphScaleColor),
  gridBColor(0.7,0.7,0.7,Lum::OS::Display::graphScaleColor),
  axisColor(1,0,0,Lum::OS::Display::blackColor),
  lineColor(0,0,0,Lum::OS::Display::blackColor)
{
  SetBackground(Lum::OS::display->GetFill(Lum::OS::Display::boxedBackgroundFillIndex));
}

Plotter::~Plotter()
{
  // no code
}

bool Plotter::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<Functions*>(model);

  Control::SetModel(this->model);

  return this->model.Valid();
}

void Plotter::CalcSize()
{
  width=100;
  height=100;
  minWidth=width;
  minHeight=height;

  Control::CalcSize();
}

int Plotter::TransformX(double x)
{
  return (int)floor((x-model->GetLeft())*(width-1)/(model->GetRight()-model->GetLeft()));
}

double Plotter::TransformX(size_t x)
{
  return x*(model->GetRight()-model->GetLeft())/(width-1)+model->GetLeft();
}

int Plotter::TransformY(double y)
{
  return (int)floor(height-(y-model->GetBottom())*(height-1)/(model->GetTop()-model->GetBottom()));
}

double Plotter::TransformY(size_t y)
{
  return (height-y)*(model->GetTop()-model->GetBottom())/(height-1)+model->GetBottom();
}

void Plotter::Draw(int x, int y, size_t w, size_t h)
{
  Lum::Control::Draw(x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (!model.Valid()) {
    return;
  }

  Lum::OS::DrawInfo *draw=GetDrawInfo();
  Lum::OS::FontRef  font=Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote);
  double            a,b,startStep,endStep;
  int               count;

  draw->PushForeground(backgroundColor);
  draw->FillRectangle(this->x,this->y,width,height);
  draw->PopForeground();

  // Now check if we need to calculate some value again

  for (size_t f=1; f<=model->GetRows(); f++) {
    if (top!=model->GetTop() || bottom!=model->GetBottom() ||
        left!=model->GetLeft() || right!=model->GetRight() ||
        model->GetValue(f).size()!=width)
    {
      std::vector<double> &values(model->GetValue(f));
      Parser::Variable    *variable=model->GetPlotVar();
      Parser::Expression  *expression=model->GetExpression(f);
      double              left=model->GetLeft();
      double              right=model->GetRight();

      assert(variable!=NULL && expression!=NULL);

      values.resize(width);

      for (size_t i=0; i<width; i++) {
        Parser::ReturnCode result;

        variable->value=left+i*(right-left)/(width-1);
        expression->Calculate(values[i],result);
      }
    }
  }

  top=model->GetTop();
  bottom=model->GetBottom();
  left=model->GetLeft();
  right=model->GetRight();

  // Vertical grid lines

  if (left>0) {
    a=0;
  }
  else {
    a=-pow(10,exponent(left)+1);
  }

  if (right>0) {
    b=pow(10,exponent(right)+1);
  }
  else {
    b=0;
  }

  endStep=pow(10,exponent(right-left));

  count=1;
  startStep=endStep;
  while (width*startStep/(5*(right-left))>=5) {
    startStep/=5;
    count++;
  }

  for (double step=startStep; step<=endStep; step*=5) {
    double c=a;
    double first=left,last=right;
    bool   hasFirst=false;

    while (c<b) {
      if (c>=left && c<=right) {
        size_t i;

        i=TransformX(c);

        if ((startStep!=endStep && count==2) || (startStep==endStep && count==1)) {
          if (this->width-i>=font->height/2) {
            last=c;
          }
          if (!hasFirst && i>=font->height/2) {
            hasFirst=true;
            first=c;
          }
        }


        if (count<=2) {
          draw->PushForeground(gridAColor); // halfShine
          draw->DrawLine(this->x+i,this->y,this->x+i,this->y+this->height-1);
          draw->PopForeground();
        }
        else {
          draw->PushForeground(gridBColor); // shine
          draw->DrawLine(this->x+i,this->y,this->x+i,this->y+this->height-1);
          draw->PopForeground();
        }
      }
      c+=step;
    }

    if ((startStep!=endStep && count==2) || (startStep==endStep && count==1)) {
      std::wstring label;

      draw->PushFont(font);
      draw->PushForeground(Lum::OS::Display::blackColor);

      // bottom
      label=DoubleToWStringExp(first);
      draw->DrawString(this->x+TransformX(first),
                       this->y+(height-font->height)/2+font->ascent,
                       label);

      // top
      label=DoubleToWStringExp(last);
      draw->DrawString(this->x+TransformX(last)-font->StringWidth(label),
                       this->y+(height-font->height)/2+font->ascent,
                       label);

      draw->PopForeground();
      draw->PopFont();
    }

    count--;
  }

  // Horizontal grid lines

  if (bottom>0) {
    a=0;
  }
  else {
    a=-pow(10,exponent(bottom)+1);
  }

  if (top>0) {
    b=pow(10,exponent(top)+1);
  }
  else {
    b=0;
  }

  endStep=pow(10,exponent(top-bottom));

  count=1;
  startStep=endStep;
  while (height*startStep/(5*(top-bottom))>=5) {
    startStep/=5;
    count++;
  }

  for (double step=startStep; step<=endStep; step*=5) {
    double c=a;
    double first=bottom,last=top;
    bool   hasFirst=false;

    while (c<b) {
      if (c>=bottom && c<=top) {
        size_t i;

        i=TransformY(c);

        if ((startStep!=endStep && count==2) || (startStep==endStep && count==1)) {
          if (i>=font->height/2) {
            last=c;
          }
          if (!hasFirst && this->height-i>=font->height/2) {
            hasFirst=true;
            first=c;
          }
        }

        if (count<=2) {
          draw->PushForeground(gridAColor); // halfShine
          draw->DrawLine(this->x,this->y+i,this->x+width-1,this->y+i);
          draw->PopForeground();
        }
        else {
          draw->PushForeground(gridBColor); // shine
          draw->DrawLine(this->x,this->y+i,this->x+width-1,this->y+i);
          draw->PopForeground();
        }
      }
      c+=step;
    }

    // Labeling of axis
    if ((startStep!=endStep && count==2) || (startStep==endStep && count==1)) {
      std::wstring label;

      draw->PushFont(font);
      draw->PushForeground(Lum::OS::Display::blackColor);

      // bottom
      label=DoubleToWStringExp(first);
      draw->DrawString(this->x+width/2-font->StringWidth(label)/2,
                       this->y+TransformY(first)-font->height/2+font->ascent,
                       label);

      // top
      label=DoubleToWStringExp(last);
      draw->DrawString(this->x+width/2-font->StringWidth(label)/2,
                       this->y+TransformY(last)-font->height/2+font->ascent,
                       label);

      draw->PopForeground();
      draw->PopFont();
    }

    count--;
  }

  // Draw possible area selection

  if (areaSelection) {
    size_t x1,y1,x2,y2;

    if (assx<asex) {
      x1=assx;
      x2=asex;
    }
    else {
      x1=asex;
      x2=assx;
    }

    if (assy<asey) {
      y1=assy;
      y2=asey;
    }
    else {
      y1=asey;
      y2=assy;
    }

    draw->PushForeground(Lum::OS::Display::graphScaleColor); // halfShine
    draw->FillRectangle(x1,y1,x2-x1+1,y2-y1+1);
    draw->PopForeground();

    draw->PushForeground(Lum::OS::Display::blackColor);
    draw->DrawRectangle(x1,y1,x2-x1+1,y2-y1+1);
    draw->PopForeground();
  }


  // Draw X-Axis

  draw->PushForeground(axisColor);
  if (top>=0 && bottom<=0) {
    int i=TransformY(0.0);
    draw->DrawLine(this->x,this->y+i,this->x+width-1,this->y+i);
  }

  // Draw Y-Axis

  if (left<=0 && right>=0) {
    int i=TransformX(0.0);
    draw->DrawLine(this->x+i,this->y,this->x+i,this->y+this->height-1);
  }

  draw->PopForeground();

  // Draw data

  draw->PushClip(this->x,this->y,this->width,this->height);
  draw->PushForeground(lineColor);

  for (size_t f=1; f<=model->GetRows(); f++) {
    std::vector<double> &values(model->GetValue(f));
    double              last=0;

    for (size_t i=0; i<width; i++) {
      if (i==0 || !isfinite(last)) {
        draw->DrawPoint(this->x+i,TransformY(values[i]));
      }
      else if (!isfinite(values[i])) {
        // do nothing
      }
      else {
        draw->DrawLine(this->x+i-1,this->y+TransformY(last),this->x+i,this->y+TransformY(values[i]));
      }
      last=values[i];
    }
  }

  draw->PopForeground();
  draw->PopClip();
}

void Plotter::VAutosize()
{
  if (!model.Valid() || model->GetRows()==0) {
    return;
  }

  double top;
  double bottom;

  top=model->GetValue(1)[0];
  bottom=top;

  for (size_t f=1; f<=model->GetRows(); f++) {
    std::vector<double> &values(model->GetValue(f));

    for (size_t i=1; i<values.size(); i++) {
      if (values[i]>top) {
        top=values[i];
      }
      if (values[i]<bottom) {
        bottom=values[i];
      }
    }
  }

  if (top==INFINITY) {
    top=1000.0;
  }
  if (bottom==-INFINITY) {
    bottom=-1000;
  }

  top=top*1.1;
  bottom=bottom*1.1;

  model->SetRegion(top,bottom,model->GetLeft(),model->GetRight());
}

void Plotter::RedrawSelectionArea(size_t assx, size_t assy,
                                  size_t asex, size_t asey,
                                  size_t oex, size_t oey)
{
  size_t x1,y1,x2,y2;

  x1=std::min(oex,std::min(assx,asex));
  y1=std::min(oey,std::min(assy,asey));

  x2=std::max(oex,std::max(assx,asex));
  y2=std::max(oey,std::max(assy,asey));

  Redraw(x1,y1,x2-x1+1,y2-y1+1);
}

void Plotter::HandleMouseMoveEvent(const Lum::OS::MouseEvent& event)
{

  if (!PointIsIn(event)) {
    return;
  }

  if ((int)asex==event.x && (int)asey==event.y) {
    return;
  }

  size_t ox,oy; // Old end values

  ox=asex;
  oy=asey;

  asex=event.x;
  asey=event.y;

  if (areaSelection && (assx==asex || assy==asey)) {
    RedrawSelectionArea(assx,assy,ox,oy,ox,oy);
    areaSelection=false;
  }
  else if (assx!=asex && assy!=asey) {
    areaSelection=true;

    RedrawSelectionArea(assx,assy,asex,asey,ox,oy);
  }
}

bool Plotter::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!model.Valid() || model->GetRows()==0) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down &&
      PointIsIn(event) &&
      event.button==Lum::OS::MouseEvent::button1) {
    assx=event.x;
    assy=event.y;
    asex=event.x;
    asey=event.y;
    return true;
  }
  else if (event.IsGrabEnd() &&
           event.type==Lum::OS::MouseEvent::up &&
           event.button==Lum::OS::MouseEvent::button1) {
    HandleMouseMoveEvent(event);

    if (areaSelection) {
      areaSelection=false;
      RedrawSelectionArea(assx,assy,asex,asey,asex,asey);

      SetRegionFromAreaSelection();
    }
    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::move && event.IsGrabed()) {
    HandleMouseMoveEvent(event);
    return true;
  }

  return false;
}

void Plotter::SetRegionFromAreaSelection()
{
  assert(model.Valid());

  if (assx<asex && assy<asey) {
    model->SetRegion(TransformY(assy-this->y),TransformY(asey-this->y),
                     TransformX(assx-this->x),TransformX(asex-this->x));
  }
  else if (assx>asex && assy>asey) {
    double hScale;
    double vScale;

    hScale=(model->GetRight()-model->GetLeft())/(TransformX(assx-this->x)-TransformX(asex-this->x));
    vScale=(model->GetTop()-model->GetBottom())/(TransformY(assy-this->y)-TransformY(asey-this->y));

    model->SetRegion(model->GetTop()*vScale,model->GetBottom()*vScale,
                     model->GetLeft()*hScale,model->GetRight()*hScale);
  }
}

void Plotter::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==this->model) {
    Redraw();
  }

  Control::Resync(model,msg);
}

