/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Calc.h"

#include <Lum/Base/String.h>

#include <Lum/Panel.h>
#include <Lum/TextValue.h>

#include "Configuration.h"
#include "FunctionInput.h"
#include "Util.h"

Calc::Calc()
: calculate(new Lum::Model::Action()),
  history(new Lum::Model::StringTable()),
  calcInput(new Lum::Model::String(L"")),
  calcResult(new Lum::Model::String(L""))
{
  Observe(calculate);
}

void Calc::Calculate()
{
  Parser::ReturnCode result;
  size_t             pos;
  Parser::Expression *expression;
  double             value;

  expression=parser.Parse(Lum::Base::WStringToString(calcInput->Get()),result,pos);

  if (result==Parser::OK) {
    if (history->GetRows()==0 || history->GetEntry(1)!=calcInput->Get()) {
      history->Prepend(calcInput->Get());
    }

    expression->Calculate(value,result,(Parser::TrigonometryMode)calcTrigoMode->Get());

    this->calcResult->Set(DoubleToWStringFloat(value));

    /*
    std::cout << "---" << std::endl;
    std::cout << std::setprecision(20);
    std::cout.unsetf(std::ios::scientific|std::ios::fixed|std::ios::showpoint);
    std::cout << value << std::endl;
    std::cout.setf(std::ios::fixed);
    std::cout  << value << std::endl;
    std::cout.unsetf(std::ios::fixed);
    std::cout.setf(std::ios::scientific,std::ios::floatfield);
    std::cout  << value << std::endl;*/

    delete expression;
  }
  else {
    comboInput->GetString()->SetCursorPos(pos);
    this->calcResult->Set(GetErrorText(result));
  }
}

void Calc::CalcSize()
{
  FunctionInput  *input;
  Lum::Panel     *panel;
  Lum::TextValue *value;

  panel=new Lum::VPanel();

  comboInput=new Lum::TextEditCombo();
  comboInput->SetFlex(true,false);
  comboInput->SetModel(calcInput);
  comboInput->SetTableModel(history);
  comboInput->GetString()->SetReturnAction(calculate);
  comboInput->GetString()->SetRequestsKeyboard(false);
  panel->Add(comboInput);

  panel->AddSpace();

  value=new Lum::TextValue();
  value->SetFlex(true,false);
  value->SetAlignment(Lum::TextValue::right);
  value->SetModel(calcResult);
  panel->Add(value);

  panel->AddSpace();

  input=new FunctionInput();
  input->SetFlex(true,true);
  input->SetString(comboInput->GetString());
  input->SetCalculateAction(calculate);
  panel->Add(input);

  container=panel;

  Component::CalcSize();
}

void Calc::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==calculate && calculate->IsFinished()) {
    Calculate();
  }

  Lum::Component::Resync(model,msg);
}

void Calc::LoadData()
{
  // no code
}

void Calc::LoadConfig(Lum::Config::Node *top)
{
  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    Lum::Config::Node *node=*iter;

    if (node->GetName()==L"calculator") {
      size_t mode;

      if (node->GetAttribute(L"trigometricalMode",mode)) {
        calcTrigoMode->Set(mode);
      }
    }
  }
}

void Calc::StoreConfig(Lum::Config::Node *top)
{
  Lum::Config::Node *node;

  node=new Lum::Config::Node(L"calculator");
  node->SetAttribute(L"trigometricalMode",*calcTrigoMode);

  top->Add(node);
}
