/*
  MathJinni - A simple formular calculator
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Plot.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Dlg/ActionDialog.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/TextValue.h>

#include "FunctionInput.h"
#include "Util.h"

class FunctionDialog : public Lum::Dlg::ActionDialog
{
private:
  Parser                &parser;
  Lum::Model::StringRef function;
  Lum::String           *functionControl;
  Lum::Model::ActionRef okAction;
  Parser::Expression*   expression;

public:
  FunctionDialog(Parser& parser, Lum::Model::String* function);

  Lum::Object* GetContent();
  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions);

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg);

  std::wstring GetFunction() const;
  Parser::Expression* GetExpression() const;
};

FunctionDialog::FunctionDialog(Parser& parser, Lum::Model::String* function)
 : parser(parser),
   function(function),
   functionControl(NULL),
   okAction(new Lum::Model::Action()),
   expression(NULL)
{
  Observe(okAction);

  Observe(GetClosedAction());
}

Lum::Object* FunctionDialog::GetContent()
{
  FunctionInput *input;
  Lum::Label    *label;
  Lum::Panel    *panel;

  panel=Lum::VPanel::Create(true,true);

  label=Lum::Label::Create(true,false);

  functionControl=Lum::String::Create(true,false);
  functionControl->SetRequestsKeyboard(false);
  functionControl->SetModel(function);
  label->AddLabel(L"Function:",functionControl);

  panel->Add(label);
  //panel->Add(functionControl);

  panel->AddSpace();

  input=new FunctionInput();
  input->SetFlex(true,true);
  input->SetString(functionControl);
  panel->Add(input);

  return panel;
}

void FunctionDialog::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
}

void FunctionDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    delete expression;
    expression=NULL;
  }
  else if (model==okAction && okAction->IsFinished()) {
    Parser::ReturnCode result;
    size_t             pos;

    delete expression;

    expression=parser.Parse(Lum::Base::WStringToString(function->Get()),result,pos);

    if (result!=Parser::OK) {
      functionControl->SetCursorPos(pos);
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Parse error!",GetErrorText(result));

      delete expression;
      expression=NULL;
    }
    else {
      Exit();
    }
  }

  Lum::Dialog::Resync(model,msg);
}

std::wstring FunctionDialog::GetFunction() const
{
  return function->Get();
}

Parser::Expression* FunctionDialog::GetExpression() const
{
  return expression;
}

class FunctionsDialog : public Lum::Dlg::ActionDialog
{
private:
  Parser                             &parser;
  FunctionsRef                       functions;
  Lum::Model::SingleLineSelectionRef selection;

  Lum::Model::ActionRef              addAction;
  Lum::Model::ActionRef              updateAction;
  Lum::Model::ActionRef              removeAction;

private:
  void AddFunction();

public:
  FunctionsDialog(Parser &parser, Functions* functions);

  Lum::Object* GetContent();
  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions);

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg);
};

FunctionsDialog::FunctionsDialog(Parser &parser, Functions* functions)
 : parser(parser),
   functions(functions),
   selection(new Lum::Model::SingleLineSelection()),
   addAction(new Lum::Model::Action()),
   updateAction(new Lum::Model::Action()),
   removeAction(new Lum::Model::Action())
{
  updateAction->Disable();
  removeAction->Disable();

  Observe(selection);

  Observe(addAction);
  Observe(updateAction);
  Observe(removeAction);

  Observe(GetClosedAction());
}

Lum::Object* FunctionsDialog::GetContent()
{
  Lum::Panel            *horiz;
  Lum::Table            *table;
  Lum::Model::HeaderRef headerModel;

  horiz=Lum::HPanel::Create(true,true);

  horiz->Add(Lum::VPanel::Create(false,true)
             ->Add(Lum::Button::Create(L"_Add",addAction,true,false))
             ->Add(Lum::Button::Create(L"_Modify",updateAction,true,false))
             ->Add(Lum::Button::Create(L"_Remove",removeAction,true,false)));

  horiz->AddSpace();

  headerModel=new Lum::Model::HeaderImpl();
  headerModel->AddColumn(L"Function",Lum::Base::Size::pixel,32000);

  table=new Lum::Table();
  table->SetFlex(true,true);
  table->SetMinWidth(Lum::Base::Size::stdCharWidth,30);
  table->GetTableView()->SetMinHeight(Lum::Base::Size::stdCharHeight,6);
  table->SetShowHeader(true);
  //table->GetTableView()->SetAutoHFit(true);
  table->SetModel(functions);
  table->SetHeaderModel(headerModel);
  table->SetSelection(selection);
  horiz->Add(table);

  return horiz;
}

void FunctionsDialog::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
{
  Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
}

void FunctionsDialog::AddFunction()
{
  FunctionDialog        *dialog;
  Lum::Model::StringRef function=new Lum::Model::String(L"");

  dialog=new FunctionDialog(parser,function);
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->GetExpression()!=NULL) {
    functions->AddFunction(dialog->GetFunction(),dialog->GetExpression());
  }

  delete dialog;
}
void FunctionsDialog::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
    //Exit();
  }
  else if (model==selection && dynamic_cast<const Lum::Model::Selection::Selected*>(&msg)!=NULL) {
    if (selection->HasSelection()) {
      updateAction->Enable();
      removeAction->Enable();
    }
    else {
      updateAction->Disable();
      removeAction->Disable();
    }
  }
  else if (model==addAction && addAction->IsFinished()) {
    AddFunction();
  }
  else if (model==updateAction && updateAction->IsFinished()) {
    if (selection->HasSelection()) {
      FunctionDialog        *dialog;
      Lum::Model::StringRef function=new Lum::Model::String(functions->GetString(1,selection->GetLine()));

      dialog=new FunctionDialog(parser,function);
      dialog->SetParent(GetWindow());

      if (dialog->Open()) {
        dialog->SetExitAction(dialog->GetClosedAction());
        dialog->EventLoop();
        dialog->Close();
      }

      if (dialog->GetExpression()!=NULL) {
        functions->SetFunction(selection->GetLine(),dialog->GetFunction(),dialog->GetExpression());
      }

      delete dialog;
    }
  }
  else if (model==removeAction && removeAction->IsFinished()) {
    if (selection->HasSelection()) {
      functions->RemoveFunction(selection->GetLine());
    }
  }

  Dialog::Resync(model,msg);
}

class RegionDialog : public Lum::Dlg::ActionDialog
{
private:
  Lum::Model::StringRef topValue;
  Lum::Model::StringRef bottomValue;
  Lum::Model::StringRef leftValue;
  Lum::Model::StringRef rightValue;

  double                top;
  double                bottom;
  double                left;
  double                right;

  Lum::Model::ActionRef okAction;

  bool                  success;

public:
  RegionDialog(double top,
               double bottom,
               double left,
               double right)
    : topValue(new Lum::Model::String(DoubleToWStringExp(top))),
      bottomValue(new Lum::Model::String(DoubleToWStringExp(bottom))),
      leftValue(new Lum::Model::String(DoubleToWStringExp(left))),
      rightValue(new Lum::Model::String(DoubleToWStringExp(right))),
      top(top),
      bottom(bottom),
      left(left),
      right(right),
      okAction(new Lum::Model::Action()),
      success(false)
  {
    Observe(GetClosedAction());
    Observe(okAction);
  }

  bool GetInterval(double &top, double &bottom, double &left, double &right)
  {
    if (sscanf(Lum::Base::WStringToString(topValue->Get()).c_str(),"%lf",&top)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the top border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(bottomValue->Get()).c_str(),"%lf",&bottom)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the bottom border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(leftValue->Get()).c_str(),"%lf",&left)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the left border of the diagram is not valid!");
      return false;
    }
    if (sscanf(Lum::Base::WStringToString(rightValue->Get()).c_str(),"%lf",&right)!=1) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"The value of the right border of the diagram is not valid!");
      return false;
    }

    if (left>=right) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"Left border value is bigger than right border value!");
      return false;
    }

    if (bottom>=top) {
      Lum::Dlg::Msg::ShowOk(GetWindow(),L"Input error!",L"Bottom border value is bigger than top border value!");
      return false;
    }

    return true;
  }

  Lum::Object* GetContent()
  {
    Lum::Panel     *vert,*horiz;
    Lum::String    *string;

    vert=Lum::VPanel::Create(true,true);

    // Top

    horiz=Lum::HPanel::Create(true,false);

    horiz->AddSpace(true);

    string=new Lum::String();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
    string->SetModel(topValue);
    horiz->Add(string);

    horiz->AddSpace(true);

    vert->Add(horiz);

    vert->AddSpace();

    // Left & right

    horiz=Lum::HPanel::Create(true,false);

    string=new Lum::String();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
    string->SetModel(leftValue);
    horiz->Add(string);

    horiz->AddSpace(true);

    string=new Lum::String();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
    string->SetModel(rightValue);
    horiz->Add(string);

    vert->Add(horiz);

    vert->AddSpace();

    // Bottom

    horiz=Lum::HPanel::Create(true,false);

    horiz->AddSpace(true);

    string=new Lum::String();
    string->SetRequestsKeyboard(false);
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,10);
    string->SetAlignment(Lum::String::right);
    string->SetInputHelper(new Lum::InputHelperNumber(Lum::InputHelperNumber::typeFloat));
    string->SetModel(bottomValue);
    horiz->Add(string);

    horiz->AddSpace(true);

    vert->Add(horiz);

    return vert;
  }

  void GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
  {
    Lum::Dlg::ActionDialog::CreateActionInfosClose(actions,GetClosedAction());
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      //Exit();
    }
    else if (model==okAction && okAction->IsFinished()) {
      if (GetInterval(top,bottom,left,right)) {
        success=true;
        Exit();
      }
    }

    Dialog::Resync(model,msg);
  }

  bool HasSuccess()
  {
    return success;
  }

  void GetIntervalValues(double& top, double& bottom, double& left, double& right)
  {
    top=this->top;
    bottom=this->bottom;
    left=this->left;
    right=this->right;
  }
};

Plot::Plot()
: editFunctions(new Lum::Model::Action()),
  autosize(new Lum::Model::Action()),
  region(new Lum::Model::Action())
 {
  plotVar=parser.AddVariable("x",0.0);

  functions=new Functions(plotVar);

  Observe(editFunctions);
  Observe(autosize);
  Observe(region);
}

void Plot::EditFunctions()
{
  FunctionsDialog *dialog;

  dialog=new FunctionsDialog(parser,functions);
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  delete dialog;
}

void Plot::Autosize()
{
  plotter->VAutosize();
}

void Plot::SetRegion()
{
  RegionDialog *dialog;
  double       top,bottom,left,right;

  dialog=new RegionDialog(functions->GetTop(),functions->GetBottom(),
                          functions->GetLeft(),functions->GetRight());
  dialog->SetParent(GetWindow());

  if (dialog->Open()) {
    dialog->SetExitAction(dialog->GetClosedAction());
    dialog->EventLoop();
    dialog->Close();
  }

  if (!dialog->HasSuccess()) {
    delete dialog;
    return;
  }

  dialog->GetIntervalValues(top,bottom,left,right);

  functions->SetRegion(top,bottom,left,right);

  delete dialog;
}

void Plot::CalcSize()
{
  Lum::Panel     *panel;

  panel=new Lum::VPanel();

  plotter=new Plotter();
  plotter->SetFlex(true,true);
  plotter->SetModel(functions);

  panel->Add(plotter);
  panel->AddSpace();
  panel->Add(Lum::ButtonRow::Create(true,false)
             ->Add(Lum::Button::Create(L"_f(x)",editFunctions,true,false))
             ->Add(Lum::Button::Create(L"_Autosize",autosize,true,false))
             ->Add(Lum::Button::Create(L"_Region",region,true,false)));

  container=panel;

  Component::CalcSize();
}

void Plot::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==editFunctions && editFunctions->IsFinished()) {
    EditFunctions();
  }
  else if (model==autosize && autosize->IsFinished()) {
    Autosize();
  }
  else if (model==region && region->IsFinished()) {
    SetRegion();
  }

  Lum::Component::Resync(model,msg);
}

void Plot::LoadConfig(Lum::Config::Node *top)
{
  for (Lum::Config::Node::NodeList::const_iterator iter=top->GetChildren().begin();
       iter!=top->GetChildren().end();
       ++iter) {
    Lum::Config::Node *node=*iter;

    if (node->GetName()==L"function") {
      std::wstring function;

      if (node->GetAttribute(L"function",function)) {
        Parser::Expression* expression;
        Parser::ReturnCode  result;
        size_t              pos;

        expression=parser.Parse(Lum::Base::WStringToString(function),result,pos);

        if (result!=Parser::OK || expression==NULL) {
          delete expression;
        }
        else {
          functions->AddFunction(function,expression);
        }
      }
    }
  }
}

void Plot::StoreConfig(Lum::Config::Node *top)
{
  if (functions.Valid()) {
    for (size_t row=1; row<=functions->GetRows(); row++) {
      Lum::Config::Node *node;

      node=new Lum::Config::Node();
      node->SetName(L"function");
      node->SetAttribute(L"function",functions->GetString(1,row));

      top->Add(node);
    }
  }
}

