/**
 * mr_biorhythm_chart.c
 *
 * A GTK+ widget for display biorhythm data
 * 
 *
 * Authors:
 *   Salvador Cabrera  <salvadorcabrera@gmail.com>
 */

#include <gtk/gtk.h>

#include "mr_biorhythm_chart.h"
#include <math.h>

typedef struct _MrBiorhythmChartPrivate MrBiorhythmChartPrivate;

struct _MrBiorhythmChartPrivate
 {
        GDate *birth_date;
        GDate *reference_date;
        GDate *highlight_date;
        cairo_surface_t *cachedChart;
 };

typedef enum
{
  PROP_0,

  PROP_BIRTH_DATE,
  PROP_REFERENCE_DATE,
  PROP_HIGHLIGHT_DATE
};

 enum
 {
   HIGHLIGHT_CHANGED,
   LAST_SIGNAL
 };

#define MR_BIORHYTHM_CHART_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MR_TYPE_BIORHYTHM_CHART, MrBiorhythmChartPrivate))

G_DEFINE_TYPE (MrBiorhythmChart, mr_biorhythm_chart, GTK_TYPE_DRAWING_AREA);

static guint mr_biorhythm_chart_signals[LAST_SIGNAL] = { 0 };

static gboolean mr_biorhythm_chart_expose (GtkWidget *biorhythm, GdkEventExpose *event);

static void mr_biorhythm_chart_finalize(GObject *object);


static void drawCacheChart( GtkWidget *biorhythm );



static void
mr_biorhythm_chart_set_property (GObject      *object,
                        guint         property_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{

    MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (object);

   	GtkWidget *widget;
    GdkRegion *region;
    GDate *tempDate;

  switch (property_id)
    {

    case PROP_BIRTH_DATE:

        g_date_free (private->birth_date);
        tempDate = (GDate*) g_value_get_pointer(value);
        private->birth_date = g_date_new_dmy( tempDate->day, tempDate->month, tempDate->year );
	
	    widget = GTK_WIDGET (object);

	    if (!widget->window) return;

	    region = gdk_drawable_get_clip_region (widget->window);

        drawCacheChart( widget );

	    /* redraw the cairo canvas completely by exposing it */
	    gdk_window_invalidate_region (widget->window, region, TRUE);
	    gdk_window_process_updates (widget->window, TRUE);

      break;

    case PROP_REFERENCE_DATE:
        g_date_free (private->reference_date);
        tempDate = (GDate*) g_value_get_pointer(value);
        private->reference_date = g_date_new_dmy( tempDate->day, tempDate->month, tempDate->year );

	    widget = GTK_WIDGET (object);

	    if (!widget->window) return;

	    region = gdk_drawable_get_clip_region (widget->window);

        drawCacheChart( widget );

	    /* redraw the cairo canvas completely by exposing it */
	    gdk_window_invalidate_region (widget->window, region, TRUE);
	    gdk_window_process_updates (widget->window, TRUE);

      break;

    case PROP_HIGHLIGHT_DATE:
        if( private->highlight_date != NULL )
        {
            g_date_free (private->highlight_date);
        }
        tempDate = (GDate*) g_value_get_pointer(value);
        private->highlight_date = g_date_new_dmy( tempDate->day, tempDate->month, tempDate->year );

	    widget = GTK_WIDGET (object);

        g_signal_emit ( widget,
                         mr_biorhythm_chart_signals[HIGHLIGHT_CHANGED],
                         0);


	    if (!widget->window) return;

	    region = gdk_drawable_get_clip_region (widget->window);

        drawCacheChart( widget );

	    /* redraw the cairo canvas completely by exposing it */
	    gdk_window_invalidate_region (widget->window, region, TRUE);
	    gdk_window_process_updates (widget->window, TRUE);

      break;

    default:
      /* We don't have any other property... */
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}


static void
mr_biorhythm_chart_get_property (GObject      *object,
                        guint         property_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{

    MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (object);

  switch (property_id)
    {

    case PROP_BIRTH_DATE:
        g_value_set_pointer( (GValue*) value, (gpointer) private->birth_date);
        break;

    case PROP_REFERENCE_DATE:
        g_value_set_pointer( (GValue*) value, (gpointer) private->reference_date);
        break;

    case PROP_HIGHLIGHT_DATE:
        g_value_set_pointer( (GValue*) value, (gpointer) private->reference_date);
        break;

    default:
        /* We don't have any other property... */
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
         break;
    }

}


static void
mr_biorhythm_chart_class_init (MrBiorhythmChartClass *class)
{
	GObjectClass *obj_class;
	GtkWidgetClass *widget_class;

	obj_class = G_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);	
    
    GParamSpec *pspec;

    obj_class->set_property = mr_biorhythm_chart_set_property;
    obj_class->get_property = mr_biorhythm_chart_get_property;

    pspec = g_param_spec_pointer ("birth-date",
                               "Birth date",
                               "Sets birth date used to compute the biorhythms",
                                G_PARAM_READWRITE);

     g_object_class_install_property (obj_class,
                                   PROP_BIRTH_DATE,
                                   pspec);

    pspec = g_param_spec_pointer ("reference-date",
                               "Reference date",
                               "Sets date to be used as a reference",
                                G_PARAM_READWRITE);

     g_object_class_install_property (obj_class,
                                   PROP_REFERENCE_DATE,
                                   pspec);

    pspec = g_param_spec_pointer ("highlight-date",
                               "Highlight date",
                               "Sets the date to highlight",
                                G_PARAM_READWRITE);

     g_object_class_install_property (obj_class,
                                   PROP_HIGHLIGHT_DATE,
                                   pspec);

    mr_biorhythm_chart_signals[HIGHLIGHT_CHANGED] = g_signal_new (
             "highlight-changed",
             G_OBJECT_CLASS_TYPE (obj_class),
             G_SIGNAL_RUN_FIRST,
             0,
             NULL,
             NULL,
             g_cclosure_marshal_VOID__VOID,
             G_TYPE_NONE, 
             0);

	widget_class->expose_event = mr_biorhythm_chart_expose;
	obj_class->finalize = mr_biorhythm_chart_finalize;

	
    g_type_class_add_private (obj_class, sizeof (MrBiorhythmChartPrivate));
}

static void
mr_biorhythm_chart_init (MrBiorhythmChart *biorhythm )
{
    MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (biorhythm);

    private->birth_date = g_date_new_dmy( 28, 3, 1987 );
    private->reference_date = g_date_new_dmy( 1, 11, 2009 );
    private->highlight_date = NULL;

    private->cachedChart = NULL;
}

static void
draw (GtkWidget *biorhythm, cairo_t *cr)
{

    int i, j;
    MrBiorhythmChartPrivate *private;
    GDate *tempDate;
    gint daysLive;
    gint initialDayChart;
    GString *dayNumberString;
    gdouble daySize;
    PangoLayout *layout;
    PangoFontDescription *fontDescription;
    int pangoContextWidth, pangoContextHeight;
    GArray *availableMonths;
    GDate *endDate;
    char monthString[128];    
    gint bottomLabelHeight;
    gint availableHeightChart;

    private = MR_BIORHYTHM_CHART_GET_PRIVATE( biorhythm );
    daysLive = g_date_days_between( private->birth_date, private->reference_date );

    daySize = biorhythm->allocation.width / 30;
    
    layout = pango_cairo_create_layout (cr);
  
    fontDescription = pango_font_description_from_string ( "Sans  20");
    pango_layout_set_font_description (layout, fontDescription);


    dayNumberString = g_string_new("");

    tempDate = g_date_new();

     g_date_set_day( tempDate, g_date_get_day( private->reference_date ) );
     g_date_set_month( tempDate, g_date_get_month( private->reference_date ) );
     g_date_set_year( tempDate, g_date_get_year( private->reference_date ) );

    g_date_subtract_days( tempDate, 15 );

    
    availableMonths = g_array_new( FALSE, FALSE, sizeof( guint ) );
    guint currentMonth = g_date_get_month( tempDate );
    g_array_append_val (availableMonths, currentMonth);

    for( i = 0; i < 30; i++ )
    {
        if( currentMonth !=  g_date_get_month( tempDate ) )
        {
            guint month = g_date_get_month( tempDate );
            
            g_array_append_val (availableMonths, month);
            currentMonth = g_date_get_month( tempDate );            
        }

        g_string_printf( dayNumberString, "%d", g_date_get_day( tempDate ) );        

        pango_layout_set_text (layout, dayNumberString->str, -1);
        pango_layout_get_size (layout, &pangoContextWidth, &pangoContextHeight);
        pango_cairo_update_layout (cr, layout);       
        cairo_move_to(cr, (i * daySize) + ( (daySize / 2)  - ((pangoContextWidth / PANGO_SCALE) / 2)   ), biorhythm->allocation.height - (pangoContextHeight / PANGO_SCALE) ); 

        cairo_set_source_rgb ( cr, 1, 1, 1);  
       
        pango_cairo_show_layout (cr, layout);
        g_date_add_days ( tempDate, 1 );
    }

    pango_font_description_free (fontDescription);
    fontDescription = pango_font_description_from_string ( "Sans  30");
    pango_layout_set_font_description (layout, fontDescription);

     g_date_set_day( tempDate, g_date_get_day( private->reference_date ) );
     g_date_set_month( tempDate, g_date_get_month( private->reference_date ) );
     g_date_set_year( tempDate, g_date_get_year( private->reference_date ) );

    g_date_subtract_days( tempDate, 15 );

    endDate = g_date_new();
     g_date_set_day( endDate, g_date_get_day( private->reference_date ) );
     g_date_set_month( endDate, g_date_get_month( private->reference_date ) );
     g_date_set_year( endDate, g_date_get_year( private->reference_date ) );
    g_date_add_days( endDate, 15 );
    

    gfloat globalOffset = 0;

    for( i = 0; i < availableMonths->len; i++  )
    {
            guint monthDays;
            guint startDay;
            guint finishDay;
            guint visibleDays;
            gint monthTextWidth;
            gint monthTextHeight;
            const char *pointer;
            gfloat totalTextWidth;
            gfloat emptySpaceWidth;
            gfloat totalEmptySpaceWidth;
            gfloat visibleArea;
            gfloat startOffset;
            GDate *limitTempDate;
            gfloat labelStartPosition;
            gfloat labelEndPosition;

            pointer = (const char*) &monthString;
        
            limitTempDate = g_date_new();
            
            startDay = g_date_get_day( tempDate );
            
            g_date_set_month( limitTempDate, g_date_get_month( tempDate ) );
            g_date_set_year( limitTempDate, g_date_get_year( tempDate ) );
    
            g_date_strftime( (gchar*)  &monthString, sizeof( monthString ), "%B", tempDate );
            pango_layout_set_text (layout, pointer, -1 );
            pango_layout_get_size (layout, &monthTextWidth, &monthTextHeight);

            monthDays = g_date_get_days_in_month( g_date_get_month(tempDate), g_date_get_year(tempDate) );
            g_date_set_day( limitTempDate, monthDays );

            if( g_date_compare(  limitTempDate, endDate ) <= 0 )
            {
                finishDay = monthDays;
            }
            else
            {
                finishDay = g_date_days_between( tempDate, endDate  );
            }

			/* Freeing don't needed dates */
			g_date_free( limitTempDate ); 
			

            visibleDays = finishDay - startDay;

            totalTextWidth = (monthTextWidth / PANGO_SCALE) * 3;
            totalEmptySpaceWidth = (daySize * monthDays) - totalTextWidth;            
            emptySpaceWidth = totalEmptySpaceWidth / 4;
            visibleArea = daySize * visibleDays;            
            startOffset = (startDay - 1 ) * daySize;

            for( j = 0;  j < 3; j++)
            {
                labelStartPosition = j * ( emptySpaceWidth + (monthTextWidth / PANGO_SCALE) );
                labelStartPosition += emptySpaceWidth;
                labelEndPosition = labelStartPosition + (monthTextWidth / PANGO_SCALE);
                if( labelEndPosition > startOffset )
                {
                            pango_cairo_update_layout (cr, layout);
                            cairo_move_to(cr, labelStartPosition + ( globalOffset ) - startOffset, 50 ); 
                            cairo_set_source_rgba ( cr, 1, 1, 1, 0.35);
                            pango_cairo_show_layout (cr, layout);
                }         
            }

            globalOffset += visibleDays * daySize;
            g_date_add_days( tempDate, visibleDays +1 );

    }

	/* We don't need endDate anymore */
	g_date_free( endDate );
    
    pango_font_description_free( fontDescription );
    g_string_free( dayNumberString, TRUE );  
    g_object_unref(layout);
    g_date_free( tempDate );
    
    bottomLabelHeight = pangoContextHeight / PANGO_SCALE;    
   
    availableHeightChart = biorhythm->allocation.height - bottomLabelHeight - 2;

    cairo_set_line_width( cr, 1 );

    for( i  = 0; i < 30 + 1; i++)
    {
        cairo_move_to( cr, (daySize * i) + 1, 0 );
        cairo_line_to( cr, daySize * i, (availableHeightChart + 1) );
        cairo_set_source_rgb ( cr, 1, 1, 1);
    	cairo_stroke (cr); 
    }

    cairo_move_to( cr, 0, ( availableHeightChart  / 2 ) + 1 );
    cairo_line_to( cr, biorhythm->allocation.width, (availableHeightChart  / 2) + 1 ) ;
    cairo_stroke (cr); 

    cairo_move_to( cr, 0, availableHeightChart  + 2 );
    cairo_line_to( cr, biorhythm->allocation.width, availableHeightChart   + 2 ) ;
    cairo_stroke (cr); 

    initialDayChart = daysLive - 15;    

    for( i = 0 ; i < biorhythm->allocation.width; i++ )
    {
       cairo_rectangle( cr, i ,  availableHeightChart  - ( (sin( ( (initialDayChart - 0.5  + (i * (30.0 / (double) biorhythm->allocation.width) ) ) / (double) MR_PHYSIC_CYCLE) * 2 * G_PI ) + 1) * ( availableHeightChart  / 2 )   ) + 0 , 1, 1 );
      cairo_set_source_rgb ( cr, 1, 0, 0);
       cairo_fill( cr );
    }

    for( i = 0 ; i < biorhythm->allocation.width; i++ )
    {
       cairo_rectangle( cr, i ,  availableHeightChart  - ( (sin( ( (initialDayChart  - 0.5 + (i * (30.0 / (double) biorhythm->allocation.width) ) ) / (double) MR_EMOTIONAL_CYCLE) * 2 * G_PI ) + 1) * ( availableHeightChart  / 2 )   ) + 0 , 1, 1 );
      cairo_set_source_rgb ( cr, 0, 0, 1);
       cairo_fill( cr);
    }

    for( i = 0 ; i < biorhythm->allocation.width; i++ )
    {
       cairo_rectangle( cr, i ,  availableHeightChart  - ( (sin( ( (initialDayChart - 0.5  + (i * (30.0 / (double) biorhythm->allocation.width) ) ) / (double) MR_INTELLECTUAL_CYCLE) * 2 * G_PI ) + 1) * ( availableHeightChart  / 2 )   ) + 0 , 1, 1 );
      cairo_set_source_rgb ( cr, 0, 1, 0);
       cairo_fill( cr);
    }

/*
    cairo_rectangle ( cr, 30, 0, 60, availableHeightChart );
    cairo_set_source_rgba( cr, 1, 0.94, 0.0, 0.25 );
    cairo_fill ( cr );
*/

    if( private->birth_date != NULL && private->highlight_date != NULL )
    {
        gint highlightDaysOffset;
        highlightDaysOffset = g_date_days_between( private->birth_date, private->highlight_date ) - initialDayChart;

        cairo_rectangle ( cr, daySize * highlightDaysOffset, 0, daySize, availableHeightChart );
        cairo_set_source_rgba( cr, 0.12, 0.61, 0.87, 0.25 );
        cairo_fill ( cr );
    }

    g_array_free (availableMonths, TRUE);    

}

static void drawPhysicCycle()
{

}

static void drawEmotionalCycle()
{

}

static void drawIntelectualCycle()
{

}

static void drawCycle()
{

}

static void drawCacheChart( GtkWidget *biorhythm )
{
    MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (biorhythm);

    if( private->cachedChart != NULL )
    {
        cairo_surface_destroy( private->cachedChart );
    }

    cairo_t *tempCairoContext;        
    
    private->cachedChart = cairo_image_surface_create ( CAIRO_FORMAT_RGB24,
                                                         biorhythm->allocation.width,
                                                         biorhythm->allocation.height);

    tempCairoContext = cairo_create ( private->cachedChart );

    cairo_rectangle( tempCairoContext,
	        0, 0,
	        biorhythm->allocation.width,
            biorhythm->allocation.height );

    draw( biorhythm, tempCairoContext );

    cairo_surface_flush (private->cachedChart); 

    cairo_destroy( tempCairoContext );
 
}

static gboolean
mr_biorhythm_chart_expose (GtkWidget *biorhythm, GdkEventExpose *event)
{
    MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (biorhythm);

    if( private->cachedChart == NULL )
    {
         drawCacheChart( biorhythm );
    }

	cairo_t *windowContext;
	
	windowContext = gdk_cairo_create( biorhythm->window );

	cairo_rectangle( windowContext,
			event->area.x, event->area.y,
			event->area.width, event->area.height );
	cairo_clip( windowContext );


    cairo_set_source_surface ( windowContext,
                                                         private->cachedChart,
                                                         0,
                                                         0);

    cairo_paint( windowContext );

    cairo_destroy( windowContext );

	return FALSE;
}

GtkWidget *
mr_biorhythm_chart_new (void)
{
	return g_object_new (MR_TYPE_BIORHYTHM_CHART, NULL);
}

static void mr_biorhythm_chart_finalize( GObject *object)
{
	MrBiorhythmChart *biorhythmChart = MR_BIORHYTHM_CHART(object);
	
	MrBiorhythmChartPrivate *private;
    private = MR_BIORHYTHM_CHART_GET_PRIVATE (biorhythmChart);
    
    g_date_free( private->birth_date );
    g_date_free( private->highlight_date );
    g_date_free( private->reference_date );
    cairo_surface_destroy( private->cachedChart );
}
