/*
 *
 * GtkGps: Display GPS information
 * Copyright (C) 2007 Kaj-Michael Lang
 * Originanl non-widget version Copyright Cezary Jackiewicz
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>

#include <glib/gstdio.h>
#include <glib-object.h>
#include <math.h>
#include "gtkgps.h"

static void gtk_gps_finalize (GObject *object);
static void gtk_gps_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void gtk_gps_size_allocate (GtkWidget *widget, GtkAllocation *allocate);
static gboolean gtk_gps_expose (GtkWidget *widget, GdkEventExpose *event);
static void gtk_gps_realize (GtkWidget *widget);
static void gtk_gps_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec);
static void gtk_gps_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec);
static void gtk_gps_paint_by_mode(GtkGps *gps);
static gboolean gtk_gps_refresh_cb(GtkWidget *widget);

G_DEFINE_TYPE(GtkGps, gtk_gps, GTK_TYPE_WIDGET);

#define PADDING (15)

static void
gtk_gps_class_init(GtkGpsClass *class)
{
GObjectClass *object_class;
GtkWidgetClass *widget_class;
	
object_class = (GObjectClass*) class;
widget_class = (GtkWidgetClass*) class;
	
object_class->finalize = gtk_gps_finalize;
object_class->set_property = gtk_gps_set_property;
object_class->get_property = gtk_gps_get_property;
	
widget_class->size_request = gtk_gps_size_request;
widget_class->expose_event = gtk_gps_expose;
widget_class->realize = gtk_gps_realize;
widget_class->size_allocate = gtk_gps_size_allocate;
}

static void
gtk_gps_set_property(GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(object));

gps=GTK_GPS(object);
switch (prop_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	break;
}
}

static void
gtk_gps_get_property(GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(object));

gps=GTK_GPS(object);
switch (prop_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
	break;
}
}

static void
gtk_gps_init(GtkGps *gps)
{
gps->gc_w=NULL;
gps->gc_s=NULL;
gps->gc_sf=NULL;
gps->width=300;
gps->height=300;
gps->size=300;
gps->esid=0;
}

static gboolean 
gtk_gps_cb_button_press(GtkWidget * widget, GdkEventButton * event)
{
GtkGps *gps;

gps=GTK_GPS(widget);

if (event->button==1) {
	switch (gps->display_mode) {
	case GTK_GPS_MODE_SKY:
		gps->display_mode=GTK_GPS_MODE_SIGNAL;
	break;
	case GTK_GPS_MODE_SIGNAL:
	default:
		gps->display_mode=GTK_GPS_MODE_SKY;
	break;
	}
}
gtk_widget_queue_draw_area(widget, 0, 0, gps->width, gps->height);
return FALSE;
}

GtkWidget*
gtk_gps_new(GtkGpsMode display_mode, GpsData *data)
{
GtkGps *gps;
GtkWidget *widget;

gps=gtk_type_new(gtk_gps_get_type ());
gps->display_mode=display_mode;
gps->data=data;
widget=GTK_WIDGET(gps);

/* Allow switching if mode is combined */
if (display_mode==GTK_GPS_MODE_COMBINED) {
	g_signal_connect(G_OBJECT(widget), "button_press_event", G_CALLBACK(gtk_gps_cb_button_press), NULL);
	display_mode=GTK_GPS_MODE_SKY;
}

gps->context=gtk_widget_get_pango_context(widget);
gps->layout=pango_layout_new(gps->context);
gps->fontdesc=pango_font_description_new();
pango_font_description_set_family(gps->fontdesc, "Sans Serif");
pango_font_description_set_size(gps->fontdesc, 10*PANGO_SCALE);
pango_layout_set_font_description(gps->layout, gps->fontdesc);
pango_layout_set_alignment(gps->layout, PANGO_ALIGN_CENTER);

gps->esid=g_timeout_add(1000,(GSourceFunc)gtk_gps_refresh_cb, gps);

return GTK_WIDGET(gps);
}

static void
gtk_gps_finalize(GObject *object)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(object));
gps=GTK_GPS(object);

g_source_remove(gps->esid);
if (GTK_WIDGET(object)->parent && GTK_WIDGET_MAPPED(object)) {
	gtk_widget_unmap(GTK_WIDGET(object));
}

G_OBJECT_CLASS(gtk_gps_parent_class)->finalize(object);
}

static void
gtk_gps_size_request(GtkWidget	*widget, GtkRequisition *requisition)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(widget));
g_return_if_fail(requisition!=NULL);
	
gps = GTK_GPS (widget);
	
requisition->width=300;
gps->width=300;

if (gps->display_mode==GTK_GPS_MODE_SKY) {
	requisition->height=400;
	gps->height=400;
} else {
	requisition->height=200;
	gps->height=200;	
}
gps->size=300;
}

static void
gtk_gps_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(widget));
g_return_if_fail(allocation!=NULL);

gps=GTK_GPS (widget);

widget->allocation = *allocation;

if (GTK_WIDGET_REALIZED (widget)) {
	gdk_window_move_resize (widget->window,
		allocation->x, allocation->y,
		allocation->width, allocation->height);
}

gps->width=allocation->width;
gps->height=allocation->height;
gps->size=MIN(gps->width, gps->height);

gps->fs=(gfloat)gps->size/28.0;
gps->fs=CLAMP(gps->fs, 10, 26);

pango_font_description_set_size(gps->fontdesc, gps->fs*PANGO_SCALE);
pango_layout_set_font_description(gps->layout, gps->fontdesc);
}

static void 
gtk_gps_realize (GtkWidget *widget)
{
GtkGps *gps;
GdkColor color;
GdkWindowAttr attributes;
gint attributes_mask;

g_return_if_fail (GTK_IS_GPS(widget));
GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
gps=GTK_GPS(widget);

attributes.x=widget->allocation.x;
attributes.y=widget->allocation.y;
attributes.width=widget->allocation.width;
attributes.height=widget->allocation.height;
attributes.wclass=GDK_INPUT_OUTPUT;
attributes.window_type=GDK_WINDOW_CHILD;
attributes.event_mask=gtk_widget_get_events(widget) | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK;
attributes.visual=gtk_widget_get_visual(widget);
attributes.colormap=gtk_widget_get_colormap(widget);

attributes_mask=GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

widget->window=gdk_window_new(gtk_widget_get_parent_window(widget), &attributes, attributes_mask);
widget->style=gtk_style_attach(widget->style, widget->window);

gdk_window_set_user_data(widget->window, widget);
gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);

/* No fix */
if (!gps->gc_s) {
	color.red=0;
	color.green=0;
	color.blue=0x1000;
	gps->gc_s=gdk_gc_new(widget->window);
	gdk_gc_set_rgb_fg_color(gps->gc_s, &color);
}

/* Fix color */
if (!gps->gc_sf) {
	color.red=0;
	color.green=0xdfff;
	color.blue=0;
	gps->gc_sf=gdk_gc_new(widget->window);
	gdk_gc_set_rgb_fg_color(gps->gc_sf, &color);
}

if (!gps->gc_w) {
	color.red=0xffff;
	color.green=0xffff;
	color.blue=0xffff;
	gps->gc_w=gdk_gc_new(widget->window);
	gdk_gc_set_rgb_fg_color(gps->gc_w, &color);
}
}

static void
gtk_gps_paint_sky(GtkGps *gps)
{
GdkGC *gc;
GtkWidget *widget;
guint i, x, y, size, halfsize, xoffset, yoffset;
guint x1, y1, x0, y0;
gfloat tmp;
gchar buffer[16];
guint line[12]={0, 30, 60, 90, 120, 150, 180, 210, 240, 270, 300, 330};

widget=GTK_WIDGET(gps);

g_return_if_fail(gps->data);

if (!GTK_WIDGET_MAPPED(widget))
	return;

x0=0;
y0=0;

size=gps->size-PADDING;
halfsize=size/2;
if (gps->width>gps->height) {
	xoffset=x0+(gps->width-gps->height)/2;
	yoffset=y0+5;
} else {
	xoffset=x0+5;
	yoffset=y0+(gps->height-gps->width)/2;
}

/* 90 */
gdk_draw_arc(widget->window,
	     widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
	     FALSE,
	     xoffset + 2, yoffset + 2, size - 4, size - 4, 0, 64 * 360);

/* 60 */
gdk_draw_arc(widget->window,
	     widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
	     FALSE,
	     xoffset + size / 6, yoffset + size / 6,
	     size / 6 * 4, size / 6 * 4, 0, 64 * 360);

/* 30 */
gdk_draw_arc(widget->window,
	     widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
	     FALSE,
	     xoffset + size / 6 * 2, yoffset + size / 6 * 2,
	     size / 6 * 2, size / 6 * 2, 0, 64 * 360);

for (i = 0; i < 12; i++) {
	gfloat sint, cost;

	tmp = (line[i] * (1.f / 180.f)) * G_PI;
	sint=sinf(tmp);
	cost=cosf(tmp);
	/* line */
	gdk_draw_line(widget->window,
		      widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
		      xoffset + halfsize + (halfsize - 2) * sint,
		      yoffset + halfsize - (halfsize - 2) * cost,
		      xoffset + halfsize - (halfsize - 2) * sint,
		      yoffset + halfsize + (halfsize - 2) * cost);

	/* Skip angle text if there is no space */
	if (size<100)
		continue;

	if (line[i] == 0)
		g_snprintf(buffer, 16, "N");
	else
		g_snprintf(buffer, 16, "%d°", line[i]);
	pango_layout_set_text(gps->layout, buffer, -1);
	pango_layout_get_pixel_size(gps->layout, &x, &y);
	gdk_draw_layout(widget->window,	widget->style->fg_gc[GTK_STATE_NORMAL],
			(xoffset + halfsize + (halfsize - size / 12) * sint) - x / 2,
			(yoffset + halfsize - (halfsize - size / 12) * cost) - y / 2,
			gps->layout);

}

if (size>100) {
	tmp = (30 * (1.f / 180.f)) * G_PI;
	/* elevation 30 */
	g_snprintf(buffer, 16, "30°");
	pango_layout_set_text(gps->layout, buffer, -1);
	pango_layout_get_pixel_size(gps->layout, &x, &y);
	gdk_draw_layout(widget->window,	widget->style->fg_gc[GTK_STATE_NORMAL],
		(xoffset + halfsize + size / 6 * 2 * sinf(tmp)) - x / 2,
		(yoffset + halfsize - size / 6 * 2 * cosf(tmp)) - y / 2,
		gps->layout);

	/* elevation 60 */
	g_snprintf(buffer, 16, "60°");
	pango_layout_set_text(gps->layout, buffer, -1);
	pango_layout_get_pixel_size(gps->layout, &x, &y);
	gdk_draw_layout(widget->window, widget->style->fg_gc[GTK_STATE_NORMAL],
		(xoffset + halfsize + size / 6 * sinf(tmp)) - x / 2,
		(yoffset + halfsize - size / 6 * cosf(tmp)) - y / 2, 
		gps->layout);
}

for (i=0;i<gps->data->satinview;i++) {
	/* Sat used or not */
	gc=(gps->data->sat[i].fix==TRUE) ? gps->gc_sf : gps->gc_s;

	tmp = (gps->data->sat[i].azimuth * (1.f / 180.f)) * G_PI;
	x = xoffset + halfsize + (90 - gps->data->sat[i].elevation) * halfsize / 90 * sinf(tmp);
	y = yoffset + halfsize - (90 - gps->data->sat[i].elevation) * halfsize / 90 * cosf(tmp);

	gdk_draw_arc(widget->window, gc, TRUE, x-gps->fs, y-gps->fs, gps->fs+6, gps->fs+6, 0, 64 * 360);

	if (size>80) {
		g_snprintf(buffer, 6, "%02d", gps->data->sat[i].prn);
		pango_layout_set_text(gps->layout, buffer, -1);
		pango_layout_get_pixel_size(gps->layout, &x1, &y1);
		gdk_draw_layout(widget->window, gps->gc_w, x-gps->fs/2-x1/2+2, y-gps->fs/2-y1/2+2, gps->layout);
	}
}

return;
}

static void
gtk_gps_paint_signals(GtkGps *gps)
{
GdkGC *gc;
GtkWidget *widget;
guint step, i, snr_height, bymargin, xoffset, yoffset;
guint x, y, x1, y1;
gchar tmp[32];

widget=GTK_WIDGET(gps);

if (!GTK_WIDGET_MAPPED(widget))
	return;

xoffset=0;
yoffset=0;

/* Bootom margin - 12% */
bymargin = gps->height * 0.88f;

/* Bottom line */
gdk_draw_line(widget->window,
	      widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
	      xoffset + 5, yoffset + bymargin,
	      xoffset + gps->width - 10 - 2, yoffset + bymargin);
gdk_draw_line(widget->window,
	      widget->style->fg_gc[GTK_WIDGET_STATE(widget)],
	      xoffset + 5, yoffset + bymargin - 1,
	      xoffset + gps->width - 10 - 2, yoffset + bymargin - 1);

if (gps->data->satinview==0)
	return;

/* Left margin - 5pix, Right margin - 5pix */
step = (gps->width - 10) / gps->data->satinview;

for (i=0;i<gps->data->satinview;i++) {
	/* Sat used or not */
	gc=(gps->data->sat[i].fix) ? gps->gc_sf : gps->gc_s;

	x = 5 + i * step;
	snr_height = gps->data->sat[i].snr * gps->height * 0.78f / 100;
	y = gps->height * 0.1f + (gps->height * 0.78f - snr_height);

	/* draw sat rectangle... */
	gdk_draw_rectangle(widget->window,
			   gc, TRUE,
			   xoffset + x,
			   yoffset + y, step - 2, snr_height);

	if (gps->data->sat[i].snr>0) {
		/* ...snr.. */
		g_snprintf(tmp, 32, "%02d", gps->data->sat[i].snr);
		pango_layout_set_text(gps->layout, tmp, 2);
		pango_layout_get_pixel_size(gps->layout, &x1, &y1);
		gdk_draw_layout(widget->window,
				widget->style->fg_gc[GTK_STATE_NORMAL],
				xoffset + x + ((step - 2) - x1) / 2,
				yoffset + y - 15,
				gps->layout);
	}

	/* ...and sat number */
	g_snprintf(tmp, 32, "%02d", gps->data->sat[i].prn);
	pango_layout_set_text(gps->layout, tmp, 2);
	pango_layout_get_pixel_size(gps->layout, &x1, &y1);
	gdk_draw_layout(widget->window,
			widget->style->fg_gc[GTK_STATE_NORMAL],
			xoffset + x + ((step - 2) - x1) / 2,
			yoffset + bymargin + 1,
			gps->layout);
}

}

static void
gtk_gps_paint_by_mode(GtkGps *gps)
{
g_return_if_fail(GTK_IS_GPS(gps));

switch (gps->display_mode) {
	case GTK_GPS_MODE_SIGNAL:
		gtk_gps_paint_signals(gps);
	break;
	case GTK_GPS_MODE_SKY:
	default:
		gtk_gps_paint_sky(gps);
	break;
}
}

static gboolean
gtk_gps_refresh_cb(GtkWidget *widget)
{
GtkGps *gps;

g_return_val_if_fail(GTK_IS_GPS(widget), FALSE);

if ((GTK_WIDGET_MAPPED(widget)==FALSE) || (GTK_WIDGET_VISIBLE(widget)==FALSE))
	return TRUE;

gps=GTK_GPS(widget);
gtk_widget_queue_draw_area(widget, 0, 0, gps->width, gps->height);
return TRUE;
}

static gboolean
gtk_gps_expose(GtkWidget *widget, GdkEventExpose *event)
{
GtkGps *gps;

g_return_val_if_fail(GTK_IS_GPS(widget), FALSE);
g_return_val_if_fail(event != NULL, FALSE);

gps=GTK_GPS(widget);
gtk_gps_paint_by_mode(gps);
return TRUE;
}

void
gtk_gps_refresh(GtkWidget *widget)
{
GtkGps *gps;

g_return_if_fail(GTK_IS_GPS(widget));

gps=GTK_GPS(widget);
gtk_widget_queue_draw_area(widget, 0, 0, gps->width, gps->height);
}
