/*
 * This file is part of mapper
 *
 * Copyright (C) 2007-2010 Kaj-Michael Lang
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <string.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gst/gst.h>
#include <gst/app/gstappsrc.h>
#include <gst/app/gstappbuffer.h>
#include <gst/app/gstappsink.h>
#include <espeak/speak_lib.h>

#include "speak.h"

typedef struct _gst_espeak gst_espeak;
struct _gst_espeak {
	GstCaps	*srccaps;
	GstElement *pipeline;
	GstElement *src;
	GstElement *queue;
	GstElement *sink;
	GstBus *bus;
	gboolean eos;
	gboolean speaking;
	gboolean espeak_ok;
	gshort *buffer;
	gint erate;
	gint size;
	gchar *text;
};
static gst_espeak ge;

/***
 * For old gstreamer in Diablo 
 */
#ifndef GST_MESSAGE_SRC_NAME
#define GST_MESSAGE_SRC_NAME(message)   (GST_MESSAGE_SRC(message) ? \
    GST_OBJECT_NAME (GST_MESSAGE_SRC(message)) : "(NULL)")
#endif
#ifndef GST_CHECK_VERSION
#define	GST_CHECK_VERSION(major,minor,micro)	\
    (GST_VERSION_MAJOR > (major) || \
     (GST_VERSION_MAJOR == (major) && GST_VERSION_MINOR > (minor)) || \
     (GST_VERSION_MAJOR == (major) && GST_VERSION_MINOR == (minor) && \
      GST_VERSION_MICRO >= (micro)))
#endif

static gboolean
speak_do_synth(gpointer data)
{
espeak_ERROR ee;

g_return_val_if_fail(ge.text, FALSE);

g_debug("synth: [%s] %zu", ge.text, strlen(ge.text));
ee=espeak_Synth(ge.text, strlen(ge.text)+1, 0, POS_CHARACTER, 0, espeakCHARS_UTF8, NULL, NULL);
if (ee==EE_BUFFER_FULL) { /* Try again */
	g_warning("Retrying synth, buffer full");
	return TRUE;
}
return FALSE;
}

static gboolean
speak_bus_call(GstBus *bus, GstMessage *msg, gpointer data)
{
gchar *debug;
GstState newstate;
GstState oldstate;
GstState pending;
GError *err=NULL;

switch (GST_MESSAGE_TYPE (msg)) {
	case GST_MESSAGE_EOS:
		g_debug("EOS from %s", GST_MESSAGE_SRC_NAME(msg));
		ge.eos=TRUE;
		g_timeout_add(500, (GSourceFunc)speak_stop, NULL);
	break;
	case GST_MESSAGE_ERROR:
		gst_message_parse_error(msg, &err, &debug);
		g_debug("Error: %s", err->message);

		g_free(debug);
		g_error_free(err);

		speak_stop();
	break;
	case GST_MESSAGE_STATE_CHANGED:
		gst_message_parse_state_changed(msg, &oldstate, &newstate, &pending);
		g_debug("GST: %s state changed (%d->%d => %d)", GST_MESSAGE_SRC_NAME(msg), oldstate, newstate, pending);

		/* We are only interested in pipeline messages */
		if (GST_MESSAGE_SRC(msg)!=GST_OBJECT(ge.pipeline))
			return TRUE;

		if (pending==GST_STATE_PAUSED)
			g_idle_add_full(G_PRIORITY_HIGH_IDLE,(GSourceFunc)speak_do_synth, NULL, NULL);
	break;
	default:
		g_debug("GST: From %s -> %s", GST_MESSAGE_SRC_NAME(msg), gst_message_type_get_name(GST_MESSAGE_TYPE(msg)));
	break;
	}
return TRUE;
}

static gboolean
speak_create_pipeline(void)
{
ge.pipeline=gst_pipeline_new("pipeline");
g_return_val_if_fail(ge.pipeline, FALSE);

ge.src=gst_element_factory_make("appsrc", "source");
g_return_val_if_fail(ge.src, FALSE);

#if GST_CHECK_VERSION(0,10,22)
gst_app_src_set_size(GST_APP_SRC(ge.src), -1);
gst_app_src_set_stream_type(GST_APP_SRC(ge.src), GST_APP_STREAM_TYPE_STREAM);
/* g_object_set(GST_APP_SRC(ge.src), "block", TRUE, NULL); */
g_object_set(GST_APP_SRC(ge.src), "is-live", TRUE, NULL);
#endif

ge.srccaps=gst_caps_new_simple ("audio/x-raw-int",
			"depth", G_TYPE_INT, 16,
			"signed", G_TYPE_BOOLEAN, TRUE, 
			"width", G_TYPE_INT,  16,
			"rate", G_TYPE_INT, ge.erate,
			"channels", G_TYPE_INT, 1,
			NULL);
g_return_val_if_fail(ge.srccaps, FALSE);

ge.queue=gst_element_factory_make("queue", "queue");
g_return_val_if_fail(ge.queue, FALSE);

ge.sink=gst_element_factory_make(AUDIO_SINK, "sink");
g_return_val_if_fail(ge.sink, FALSE);

gst_bin_add_many(GST_BIN(ge.pipeline), ge.src, ge.queue, ge.sink, NULL);

if (!gst_element_link_filtered(ge.src, ge.queue, ge.srccaps)) {
	g_warning("Failed to link source to queue with caps.");
	return FALSE;
}
gst_caps_unref(ge.srccaps);

if (!gst_element_link(ge.queue, ge.sink)) {
	g_warning("Failed to link queue to sink.");
	return FALSE;
}

return TRUE;
}

static void
espeak_buffer_free(void *p)
{
g_free(p);
}

static gboolean
speak_start_play(gpointer data)
{
if (gst_element_set_state(ge.pipeline, GST_STATE_PLAYING)==GST_STATE_CHANGE_FAILURE) {
	g_warning("Failed to play pipeline");
	ge.speaking=FALSE;
}
return FALSE;
}

static int 
espeak_cb(short *wav, int numsamples, espeak_EVENT *events)
{
GstBuffer *buf;
gchar *data;

if (wav==NULL) {
#if GST_CHECK_VERSION(0,10,22)
	if (gst_app_src_end_of_stream(GST_APP_SRC(ge.src))!=GST_FLOW_OK)
		g_warning("Failed to push EOS");
#else
	gst_app_src_end_of_stream(GST_APP_SRC(ge.src));
#endif
	g_idle_add_full(G_PRIORITY_HIGH_IDLE, (GSourceFunc)speak_start_play, NULL, NULL);
} else if (numsamples>0) {
	g_debug("Adding speach buffer %d", numsamples);

	numsamples=numsamples*2;
	data=g_memdup(wav, numsamples);
	buf=gst_app_buffer_new(data, numsamples, espeak_buffer_free, data);
	gst_buffer_set_caps(buf, ge.srccaps);
#if GST_CHECK_VERSION(0,10,22)
	if (gst_app_src_push_buffer(GST_APP_SRC(ge.src), buf)!=GST_FLOW_OK)
		g_warning("Failed to push buffer");
#else
	gst_app_src_push_buffer(GST_APP_SRC(ge.src), buf);
#endif
}

return 0;
}

void
speak_set_parameters(guint speed, guint pitch)
{
espeak_SetParameter(espeakRATE, speed, 0);
espeak_SetParameter(espeakPITCH, pitch, 0);
}

gboolean
speak_init(gchar *voice, guint speed, guint pitch)
{
ge.text=NULL;
ge.eos=FALSE;
ge.espeak_ok=FALSE;

ge.erate=espeak_Initialize(AUDIO_OUTPUT_RETRIEVAL, 250, NULL, 0);
if (ge.erate==-1)
	return FALSE;

espeak_SetSynthCallback(espeak_cb);
espeak_SetVoiceByName(voice);
speak_set_parameters(speed, pitch);
espeak_SetParameter(espeakVOLUME, 100, 0);
if (speak_create_pipeline()==FALSE)
	return FALSE;

ge.bus=gst_pipeline_get_bus(GST_PIPELINE(ge.pipeline));
g_return_val_if_fail(ge.bus, FALSE);

gst_bus_add_watch(ge.bus, speak_bus_call, NULL);

ge.espeak_ok=TRUE;
return TRUE;
}

void
speak_deinit(void)
{
if (ge.espeak_ok==FALSE)
	return;

gst_element_set_state(ge.pipeline, GST_STATE_NULL);
gst_object_unref(ge.pipeline);
ge.pipeline=NULL;
if (ge.text)
	g_free(ge.text);
ge.text=NULL;
ge.speaking=FALSE;
espeak_Terminate();
}

gboolean
speak_text(const gchar *text)
{
g_debug("Speak: [%s] (%zu)", text, strlen(text));

if (ge.text)
	g_free(ge.text);
ge.text=g_strdup(text);

if (ge.speaking==FALSE || ge.eos==TRUE) {
	g_debug("Starting playback");
	ge.eos=FALSE;
	if (gst_element_set_state(ge.pipeline, GST_STATE_PAUSED)==GST_STATE_CHANGE_FAILURE) {
		g_warning("Failed to pause pipeline");
		ge.eos=TRUE;
		ge.speaking=FALSE;
		return FALSE;
	}
	ge.speaking=TRUE;
}
return TRUE;
}

gboolean
speak_speaking(void)
{
return ge.speaking;
}

gboolean
speak_stop(void)
{
g_debug("Speak: Stop");
if (ge.text)
	g_free(ge.text);
ge.text=NULL;
ge.eos=TRUE;
ge.speaking=FALSE;
if (gst_element_set_state(ge.pipeline, GST_STATE_NULL)==GST_STATE_CHANGE_FAILURE) {
	g_warning("Failed to stop pipeline");
	return FALSE;
}
return FALSE;
}