/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/* 
 * Maemo hardware specific functions 
 */

#include <config.h>

#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <gtk/gtk.h>

#ifdef WITH_OSSO
#include <libosso.h>

#ifdef WITH_MCE
#include <mce/mode-names.h>
#include <mce/dbus-names.h>
#define MCE_SIGNAL_MATCH "type='signal'," \
	"sender='"    MCE_SERVICE     "'," \
	"interface='" MCE_SIGNAL_IF   "'"

#ifdef MCE_ORIENTATION_PORTRAIT
#define ENABLE_ROTATION
#endif

#endif

#include "mapper.h"
#include "map.h"
#include "map-download.h"
#include "route.h"
#include "mapper-types.h"
#include "gps.h"
#include "settings-gconf.h"
#include "settings.h"
#include "config-gconf.h"
#include "poi.h"
#include "gps-conn.h"
#include "maemo-osso.h"

void
maemo_osso_keep_display_on(MapperApp *app)
{
if (_always_keep_on || _fullscreen) {
	osso_display_state_on(mapper_app.osso);
	osso_display_blanking_pause(mapper_app.osso);
}
}

static void
maemo_osso_set_rotation(MapperApp *app, const gchar *rotation)
{
#ifdef ENABLE_ROTATION
HildonPortraitFlags f;

g_return_if_fail(app);
g_return_if_fail(rotation);

f=(strcmp(rotation, MCE_ORIENTATION_PORTRAIT)==0) ? HILDON_PORTRAIT_MODE_SUPPORT | HILDON_PORTRAIT_MODE_REQUEST : HILDON_PORTRAIT_MODE_SUPPORT;
hildon_gtk_window_set_portrait_flags(GTK_WINDOW(app->ui->mainwindow), f);
#endif
}

gboolean
maemo_osso_rotation_start(MapperApp *app)
{
#ifdef ENABLE_ROTATION
gboolean ok=FALSE;
gchar *rotation;
DBusError error;
DBusMessage *message, *reply;
DBusConnection *dbc=dbus_g_connection_get_connection(app->dbus_conn);

dbus_error_init(&error);
message=dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ);
reply=dbus_connection_send_with_reply_and_block(dbc, message, -1, &error);
dbus_message_unref(message);
if (dbus_message_get_args(reply, NULL, DBUS_TYPE_STRING, &rotation, DBUS_TYPE_INVALID)) {
	maemo_osso_set_rotation(app, rotation);
	ok=TRUE;
}
dbus_message_unref(reply); 
return ok;
#else
return FALSE;
#endif
}

void
maemo_osso_rotation_stop(MapperApp *app)
{
#ifdef ENABLE_ROTATION
DBusError error;
DBusMessage *message, *reply;
DBusConnection *dbc=dbus_g_connection_get_connection(app->dbus_conn);

dbus_error_init(&error);
message=dbus_message_new_method_call(MCE_SERVICE, MCE_REQUEST_PATH, MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ);
reply=dbus_connection_send_with_reply_and_block(dbc, message, -1, &error);
dbus_message_unref(message);
#endif
}

static DBusHandlerResult
maemo_osso_mce_filter_func(DBusConnection *connection, DBusMessage *message, MapperApp *app)
{
DBusError error;

dbus_error_init(&error);

#ifdef ENABLE_ROTATION
if (dbus_message_is_signal(message, MCE_SIGNAL_IF, MCE_DEVICE_ORIENTATION_SIG)) {
	gchar *rotation, *stand, *face;
	gint x, y, z;

	/* Retrieve arguments from a message */
	if (!dbus_message_get_args(message, &error,
			DBUS_TYPE_STRING, &rotation, DBUS_TYPE_STRING, &stand, DBUS_TYPE_STRING, &face,
			DBUS_TYPE_INT32, &x, DBUS_TYPE_INT32, &y, DBUS_TYPE_INT32, &z, DBUS_TYPE_INVALID)) {
		g_debug("DBus error: %s (%s)\n", error.name, error.message);
		dbus_error_free(&error);
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}
	g_debug("R: %s S: %s F: %s, x=%d, y=%d z=%d", rotation, stand, face, x, y, z);
	maemo_osso_set_rotation(app, rotation);
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}
#endif

return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gint
maemo_osso_dbus_cb_default(const gchar *interface, const gchar *method, GArray *arguments, gpointer data, osso_rpc_t *retval)
{
if(!strcmp(method, "top_application"))
	gtk_idle_add((GSourceFunc)mapper_window_present, NULL);
retval->type=DBUS_TYPE_INVALID;
return OSSO_OK;
}

static void
maemo_osso_drop_caches(void)
{
/* XXX drop tile cache too */
poi_icon_hash_clear();
}

static void
maemo_osso_cb_hw_state(osso_hw_state_t *state, gpointer data)
{
static gboolean _must_save_data = FALSE;

if (state->system_inactivity_ind) {
	if (_must_save_data)
		_must_save_data = FALSE;
	else {
		if (_gps->io.conn > RCVR_OFF) {
			if (gconf_client_get_bool(gconf_client, GCONF_KEY_DISCONNECT_ON_COVER, NULL)) {
				gconf_client_clear_cache(gconf_client);
				gps_conn_set_state(_gps, RCVR_OFF);
				gps_disconnect(_gps);
				/* Pretend autoroute is in progress to avoid download. */
				if (_autoroute_data.enabled)
					_autoroute_data.in_progress=TRUE;
			}
		}
	}
} else if (state->save_unsaved_data_ind) {
	config_save();
	_must_save_data=TRUE;
} else if (state->memory_low_ind) {
	maemo_osso_drop_caches();
} else {
	if (_gps->io.conn==RCVR_OFF && _enable_gps) {
		gps_conn_set_state(_gps, RCVR_DOWN);
		gps_connect_later(_gps);
		if (_autoroute_data.enabled)
			_autoroute_data.in_progress=TRUE;
	}
}
}

/**
 * OSSO (de)initialization functions below
 **/
gboolean
maemo_osso_initialize(MapperApp *app)
{
DBusConnection *dbc=dbus_g_connection_get_connection(app->dbus_conn);

app->osso=osso_initialize("org.tal.mapper", VERSION, TRUE, NULL);
if (!app->osso) {
	g_printerr("osso_initialize failed.\n");
	return 1;
}
if (osso_rpc_set_default_cb_f(app->osso, maemo_osso_dbus_cb_default, NULL)!=OSSO_OK) {
	g_printerr("osso_rpc_set_default_cb_f failed.\n");
	return 1;
}
osso_hw_set_event_cb(app->osso, NULL, maemo_osso_cb_hw_state, app);
dbus_bus_add_match(dbc, MCE_SIGNAL_MATCH, NULL);
dbus_connection_add_filter(dbc, (DBusHandleMessageFunction)maemo_osso_mce_filter_func, app, NULL);
return 0;
}

void
maemo_osso_deinitialize(MapperApp *app)
{
DBusConnection *dbc=dbus_g_connection_get_connection(app->dbus_conn);

dbus_bus_remove_match(dbc, MCE_SIGNAL_MATCH, NULL);
osso_deinitialize(app->osso);
}

#endif
