/*
 * This file is part of mapper
 *
 * Copyright (C) 2007 Kaj-Michael Lang
 * Copyright (C) 2006-2007 John Costigan.
 *
 * POI and GPS-Info code originally written by Cezary Jackiewicz.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <stddef.h>
#include <errno.h>
#include <sys/wait.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <fcntl.h>

#include "hildon-mapper.h"
#include <bt-dbus.h>

#include "gps.h"
#include "gps-scan.h"
#include "gps-conn.h"

#include "gps-bluetooth-maemo-marshal.h"

/**
 * Bluetooth device scanning functions below
 */
static void
scan_dev_found_cb(DBusGProxy *sig_proxy, const gchar *bda, const gchar *name, gpointer *class, guchar rssi, gint coff, GpsScanInfo *scan_info)
{
GtkTreeIter iter;

gtk_list_store_append(scan_info->store, &iter);
gtk_list_store_set(scan_info->store, &iter, 0, g_strdup(bda), 1, g_strdup(name), -1);
}

static void 
scan_search_complete_cb(DBusGProxy * sig_proxy, GpsScanInfo *scan_info)
{
gtk_widget_destroy(scan_info->banner);
dbus_g_proxy_disconnect_signal(sig_proxy, BTSEARCH_DEV_FOUND_SIG, G_CALLBACK(scan_dev_found_cb), scan_info);
dbus_g_proxy_disconnect_signal(sig_proxy, BTSEARCH_SEARCH_COMPLETE_SIG, G_CALLBACK(scan_search_complete_cb), scan_info);
}

/**
 * Request bluetooth device scan
 */
GpsScanResult
gps_scan_start_search(GpsScanInfo *scan_info)
{
GError *error=NULL;

g_return_val_if_fail(scan_info, GPS_SCAN_ERROR);

gtk_list_store_clear(scan_info->store);
if (NULL==(scan_info->req_proxy=dbus_g_proxy_new_for_name(scan_info->dbus_conn, BTSEARCH_SERVICE, BTSEARCH_REQ_PATH, BTSEARCH_REQ_INTERFACE))) {
	g_printerr("Failed to create D-Bus request proxy for btsearch.");
	return GPS_SCAN_FAIL;
}

if (NULL==(scan_info->sig_proxy=dbus_g_proxy_new_for_name(scan_info->dbus_conn, BTSEARCH_SERVICE, BTSEARCH_SIG_PATH, BTSEARCH_SIG_INTERFACE))) {
	g_printerr("Failed to create D-Bus signal proxy for btsearch.");
	return GPS_SCAN_FAIL;
}

dbus_g_object_register_marshaller(_bt_maemo_VOID__STRING_STRING_POINTER_UCHAR_UINT, G_TYPE_NONE, G_TYPE_STRING, G_TYPE_STRING, DBUS_TYPE_G_UCHAR_ARRAY, G_TYPE_UCHAR, G_TYPE_UINT, G_TYPE_INVALID);

dbus_g_proxy_add_signal(scan_info->sig_proxy, BTSEARCH_DEV_FOUND_SIG, G_TYPE_STRING, G_TYPE_STRING, DBUS_TYPE_G_UCHAR_ARRAY, G_TYPE_UCHAR, G_TYPE_UINT, G_TYPE_INVALID);
dbus_g_proxy_connect_signal(scan_info->sig_proxy, BTSEARCH_DEV_FOUND_SIG, G_CALLBACK(scan_dev_found_cb), scan_info, NULL);

dbus_g_proxy_add_signal(scan_info->sig_proxy, BTSEARCH_SEARCH_COMPLETE_SIG, G_TYPE_INVALID);
dbus_g_proxy_connect_signal(scan_info->sig_proxy, BTSEARCH_SEARCH_COMPLETE_SIG, G_CALLBACK(scan_search_complete_cb), scan_info, NULL);

if (!dbus_g_proxy_call(scan_info->req_proxy, BTSEARCH_START_SEARCH_REQ, &error, G_TYPE_INVALID, G_TYPE_INVALID)) {
	if (error->domain==DBUS_GERROR && error->code == DBUS_GERROR_REMOTE_EXCEPTION) {
		g_printerr("Caught remote method exception %s: %s", dbus_g_error_get_name(error), error->message);
	} else {
		g_printerr("Error: %s\n", error->message);
	}
	return GPS_SCAN_ERROR;
}

return GPS_SCAN_OK;
}

gboolean
gps_scan_stop_search(GpsScanInfo *scan_info)
{
GError *error=NULL;

dbus_g_proxy_call(scan_info->req_proxy, BTSEARCH_STOP_SEARCH_REQ, &error, G_TYPE_INVALID, G_TYPE_INVALID);
g_object_unref(scan_info->req_proxy);
g_object_unref(scan_info->sig_proxy);
return TRUE;
}
