/*
 * poi-layer.c - Champlain POI layer
 * Copyright (C) 2010 Collabora Ltd
 * @author Cosimo Alfarano <cosimo.alfarano@collabora.co.uk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include "config.h"
#include "poi-layer.h"

#include "poi.h"
#include "poi-manager.h"

#include <hildon/hildon.h>


typedef struct
{
  ChamplainLayer *pois_layer;
  MapBuddyPoiManager *poi_manager;
  GtkWindow *main_window;

} MapBuddyPoiLayerPrivate;

struct _MapBuddyPoiLayer
{
  GObject base;

  MapBuddyPoiLayerPrivate *priv;
};

struct _MapBuddyPoiLayerClass
{
  GObjectClass parent_class;
};



G_DEFINE_TYPE (MapBuddyPoiLayer, map_buddy_poi_layer,
    G_TYPE_OBJECT);

#define POI_LAYER_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE((obj), MAP_BUDDY_TYPE_POI_LAYER, \
                               MapBuddyPoiLayerPrivate))


#define POI_QUARK g_quark_from_static_string ("MapBuddyWindowPOI")
#define POI_MARKER_QUARK g_quark_from_static_string ("MapBuddyWindowPOItMarker")

static gboolean
on_poi_marker_press (ClutterActor *actor,
    ClutterEvent *event,
    gpointer user_data)
{
  MapBuddyPoiLayer *self = MAP_BUDDY_POI_LAYER (user_data);
  MapBuddyPoiLayerPrivate *priv = POI_LAYER_GET_PRIVATE (self);
  MapBuddyPoi *poi = g_object_get_qdata (G_OBJECT (actor), POI_QUARK);
  GtkWidget *dialog;
  GtkWidget *label;
  GHashTable *tags;
  const gchar *tag_value;
  gchar *markup;

  g_debug ("Clicked on contact %s\n",
      champlain_marker_get_text (CHAMPLAIN_MARKER (actor)));

  if (!poi)
    return TRUE;

  dialog = hildon_dialog_new_with_buttons (map_buddy_poi_get_name (poi),
      priv->main_window, GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
      NULL, NULL);

  markup = g_markup_printf_escaped ("<b>%s</b> (%s)",
      map_buddy_poi_get_feature_type (poi),
      map_buddy_poi_get_feature (poi)
      );

  label = gtk_label_new (NULL);
  gtk_label_set_markup (GTK_LABEL (label), markup);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE,
      TRUE, 0);

  g_free (markup);

  tags = map_buddy_poi_get_tags (poi);

  tag_value = g_hash_table_lookup (tags, "website");
  if (tag_value != NULL)
    {
      markup = g_markup_printf_escaped ( "<u>%s</u>",
          tag_value);

      label = gtk_label_new (NULL);
      gtk_label_set_markup (GTK_LABEL (label), markup);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
          label, TRUE, TRUE, 0);

      g_free (markup);
    }

  tag_value = g_hash_table_lookup (tags, "phone");
  if (tag_value != NULL)
    {
      markup = g_markup_printf_escaped ( "<i>%s</i>",
          tag_value);

      label = gtk_label_new (NULL);
      gtk_label_set_markup (GTK_LABEL (label), markup);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
          label, TRUE, TRUE, 0);

      g_free (markup);
    }

  gtk_widget_show_all (dialog);

  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return TRUE;
}

static void
on_poi_added (MapBuddyPoiManager *poi_manager,
    MapBuddyPoi *poi,
    gpointer user_data)
{
  MapBuddyPoiLayer *self = MAP_BUDDY_POI_LAYER (user_data);
  MapBuddyPoiLayerPrivate *priv = POI_LAYER_GET_PRIVATE (self);
  ClutterActor *marker;

  g_return_if_fail (MAP_BUDDY_IS_POI_MANAGER (poi_manager));
  g_return_if_fail (MAP_BUDDY_IS_POI_LAYER (self));

  g_print ("poi-layer: %s: %s %s/%s XY(%f,%f)\n",
      G_STRFUNC,
      map_buddy_poi_get_name (poi),
      map_buddy_poi_get_feature (poi),
      map_buddy_poi_get_feature_type (poi),
      map_buddy_poi_get_lat (poi),
      map_buddy_poi_get_lon (poi));

  marker = g_object_get_qdata (G_OBJECT (poi), POI_MARKER_QUARK);
  /* if it's a POI on which we already set a qdata, ignore it.
   * Updated POI in manager are actually new GObject, thus without any qdata
   * associated to this quark */
  if (marker != NULL)
    return;

  marker = champlain_marker_new ();

  g_object_set_qdata (G_OBJECT (marker), POI_QUARK, poi);

  champlain_marker_set_image (CHAMPLAIN_MARKER (marker),
      map_buddy_poi_get_icon_small (poi));
  champlain_base_marker_set_position (CHAMPLAIN_BASE_MARKER (marker),
      map_buddy_poi_get_lat (poi), map_buddy_poi_get_lon (poi));

  g_signal_connect (CLUTTER_ACTOR (marker), "button-press-event",
      G_CALLBACK (on_poi_marker_press), self);

  champlain_layer_add_marker (priv->pois_layer,
      CHAMPLAIN_BASE_MARKER (marker));

  g_object_set_qdata_full (G_OBJECT (poi), POI_MARKER_QUARK, marker,
      (GDestroyNotify) clutter_actor_destroy);
}

static void
on_poi_removed (MapBuddyPoiManager *poi_manager,
    MapBuddyPoi *poi,
    gpointer user_data)
{
  g_debug ("poi-layer.c: %s on %s (%d)",
      G_STRFUNC,
      map_buddy_poi_get_name (poi),
      map_buddy_poi_get_id (poi));

  g_object_set_qdata (G_OBJECT (poi), POI_MARKER_QUARK, NULL);
}

static void
map_buddy_poi_layer_dispose (GObject *object)
{

  MapBuddyPoiLayer *self = MAP_BUDDY_POI_LAYER (object);
  MapBuddyPoiLayerPrivate *priv = POI_LAYER_GET_PRIVATE (self);

  if (priv->pois_layer != NULL)
    {
      g_object_unref (priv->pois_layer);
      priv->pois_layer = NULL;
    }

  if (priv->poi_manager != NULL)
    {
      g_object_unref (priv->poi_manager);
      priv->poi_manager = NULL;
    }

  if (priv->main_window != NULL)
    {
      g_object_unref (priv->main_window);
      priv->main_window = NULL;
    }

  G_OBJECT_CLASS (map_buddy_poi_layer_parent_class)->dispose (object);
}

static void
map_buddy_poi_layer_class_init (MapBuddyPoiLayerClass *cls)
{
  g_type_class_add_private (cls, sizeof (MapBuddyPoiLayerPrivate));

  GObjectClass *object_class = G_OBJECT_CLASS (cls);
  object_class->dispose = map_buddy_poi_layer_dispose;
}

static void
map_buddy_poi_layer_init (MapBuddyPoiLayer *self)
{
  MapBuddyPoiLayerPrivate *priv = POI_LAYER_GET_PRIVATE (self);

  self->priv = priv;
}

void
map_buddy_poi_layer_start (MapBuddyPoiLayer *self,
    ChamplainLayer *champlain_layer,
    GtkWindow *main_window)
{
  MapBuddyPoiLayerPrivate *priv = POI_LAYER_GET_PRIVATE (self);

  priv->poi_manager = map_buddy_poi_manager_dup ();
  priv->pois_layer = g_object_ref (champlain_layer);
  priv->main_window = g_object_ref (main_window);

  g_signal_connect (priv->poi_manager, "poi-added",
      G_CALLBACK (on_poi_added), self);
  g_signal_connect (priv->poi_manager, "poi-removed",
      G_CALLBACK (on_poi_removed), self);

  map_buddy_poi_manager_start (priv->poi_manager);
}

MapBuddyPoiLayer *
map_buddy_poi_layer_new (void)
{
  return g_object_new (MAP_BUDDY_TYPE_POI_LAYER, NULL);
}
