/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the Graphics Dojo project on Qt Labs.
**
** This file may be used under the terms of the GNU General Public
** License version 2.0 or 3.0 as published by the Free Software Foundation
** and appearing in the file LICENSE.GPL included in the packaging of
** this file.  Please review the following information to ensure GNU
** General Public Licensing requirements will be met:
** http://www.fsf.org/licensing/licenses/info/GPLv2.html and
** http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
****************************************************************************/

#include "flickable.h"

#include <QtCore>
#include <QtGui>

class FlickableTicker: QObject
{
public:
    FlickableTicker(Flickable *scroller) {
        m_pScroller = scroller;
    }

    void start(int interval) {
        if (!m_timer.isActive())
            m_timer.start(interval, this);
    }

    void stop() {
        m_timer.stop();
    }

protected:
    void timerEvent(QTimerEvent *event) {
        Q_UNUSED(event);
        m_pScroller->tick();
    }

private:
    Flickable *m_pScroller;
    QBasicTimer m_timer;
};

Flickable::Flickable()
{
    m_state = Steady;
    m_threshold = 20; // it was 10
    m_pTicker = new FlickableTicker(this);
    m_timeStamp = QTime::currentTime();
    m_pTarget = 0;
}

Flickable::~Flickable() {
}

void Flickable::setThreshold(int th)
{
    if (th >= 0)
        m_threshold = th;
}

int Flickable::threshold() const
{
    return m_threshold;
}

void Flickable::setAcceptMouseClick(QWidget *target)
{
    target = target;
}

static QPoint deaccelerate(const QPoint &speed, int a = 1, int max = 64)
{
    int x = qBound(-max, speed.x(), max);
    int y = qBound(-max, speed.y(), max);
    x = (x == 0) ? x : (x > 0) ? qMax(0, x - a) : qMin(0, x + a);
    y = (y == 0) ? y : (y > 0) ? qMax(0, y - a) : qMin(0, y + a);
    return QPoint(x, y);
}

void Flickable::handleMousePress(QMouseEvent *event)
{
    event->ignore();

    if (event->button() != Qt::LeftButton) return;
    if (m_ignoreList.removeAll(event)) return;

    switch (m_state) {
        case Steady:
            qDebug() << "handleMousePress: Steady - register click";
            event->accept();
            m_state = Pressed;
            m_pressPos = event->pos();
            break;

        case AutoScroll:
            qDebug() << "handleMousePress: AutoScroll - stop auto scrolling";
            event->accept();
            m_state = Stop;
            m_speed = QPoint(0, 0);
            m_pressPos = event->pos();
            m_offset = scrollOffset();
            m_pTicker->stop();
            break;

        default:
            qDebug() << "handleMousePress: error? " << m_state;
            break;
    }
}

void Flickable::handleMouseRelease(QMouseEvent *pEvent)
{
    pEvent->ignore();

    if (pEvent->button() != Qt::LeftButton) return;
    if (m_ignoreList.removeAll(pEvent)) return;

    QPoint delta;

    switch (m_state) {

        case Pressed:
            qDebug() << "handleMouseRelease: Pressed - fire fake events";

            pEvent->accept();
            m_state = Steady;
            if (m_pTarget) {
                QMouseEvent *m_pPressEvent = new QMouseEvent(
                    QEvent::MouseButtonPress,
                    m_pressPos, Qt::LeftButton,
                    Qt::LeftButton, Qt::NoModifier
                );
                QMouseEvent *m_pReleaseEvent = new QMouseEvent(*pEvent);
                m_ignoreList << m_pPressEvent;
                m_ignoreList << m_pReleaseEvent;
                QApplication::postEvent(m_pTarget, m_pPressEvent);
                QApplication::postEvent(m_pTarget, m_pReleaseEvent);
            }
            break;

        case ManualScroll:
            qDebug() << "handleMouseRelease: ManualScroll";

            pEvent->accept();
            delta = pEvent->pos() - m_pressPos;
            if (m_timeStamp.elapsed() > 100) {
                qDebug() << ">> elapsed>100: ????";
                m_timeStamp = QTime::currentTime();
                m_speed = delta - delta;
                delta = delta;
            }

            m_offset = scrollOffset();
            m_pressPos = pEvent->pos();

            if (m_speed == QPoint(0, 0)) {
                qDebug() << ">> keep it steady?";
                m_state = Steady;
            } else {
                qDebug() << ">> start Autoscroll";
                m_speed /= 4;
                m_state = AutoScroll;
                m_pTicker->start(20);
            }
            break;

        case Stop:
            qDebug() << "handleMouseRelease: Stop - just stopped? make it steady";
            pEvent->accept();
            m_state = Steady;
            m_offset = scrollOffset();
            break;

        default:
            qDebug() << "handleMouseRelease: error? " << m_state;
            break;
    }
}

void Flickable::handleMouseMove(QMouseEvent *event)
{
    event->ignore();

    if (!(event->buttons() & Qt::LeftButton)) return;
    if (m_ignoreList.removeAll(event)) return;

    QPoint delta;

    switch (m_state) {
        case Pressed:
        case Stop:
            delta = event->pos() - m_pressPos;
            if ( delta.x() > m_threshold || delta.x() < -m_threshold ||
                delta.y() > m_threshold || delta.y() < -m_threshold )
            {
                qDebug() << "handleMouseMove (Pressed or Stop): start manual scroll";

                m_timeStamp = QTime::currentTime();
                m_state = ManualScroll;
                delta = QPoint(0, 0);
                m_pressPos = event->pos();
                m_offset = scrollOffset(); // ?!?!?!?!?!? felipe
                event->accept();
                        } else {
                                qDebug() << "handleMouseMove (Pressed or Stop): bellow threshold: " << m_state;
                        }
            break;

        case ManualScroll: {
                qDebug() << "handleMouseMove: ManualScroll";
                event->accept();
                delta = event->pos() - m_pressPos;
                QPoint newpos = m_offset - delta;

                qDebug() << ">> Press: " << m_pressPos.y()
                        << ", Event: " << event->pos().y()
                        << ", Delta: " << delta.y()
                        << ", Scroll: " << m_offset.y()
                        << " = " << newpos.y();

                setScrollOffset(newpos); // offset - delta

                if (m_timeStamp.elapsed() > 100) {
                    qDebug() << ">> elapsed()>100?";
                    m_timeStamp = QTime::currentTime();
                    m_speed = delta - delta;
                    delta = delta;
                }
            }
            break;

        default:
            qDebug() << "handleMouseMove: error?" << m_state;
            break;
    }
}

void Flickable::tick()
{
    if (m_state ==  AutoScroll) {
        m_speed = deaccelerate(m_speed);
        setScrollOffset(m_offset - m_speed);
        m_offset = scrollOffset();
        if (m_speed == QPoint(0, 0)) {
            m_state = Steady;
            m_pTicker->stop();
        }
    } else {
        m_pTicker->stop();
    }
}
