/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "viewportitem.h"

ViewportItem::ViewportItem()
    : QGraphicsWidget(), m_widget(0)
{
    setFlag(QGraphicsItem::ItemHasNoContents, true);
    setFlag(QGraphicsItem::ItemClipsChildrenToShape, true);
    setFlag(QGraphicsItem::ItemClipsToShape, true);
    setAttribute(Qt::WA_OpaquePaintEvent, true);
    setFiltersChildEvents(true);
}

void ViewportItem::setWidget(QGraphicsWidget *widget)
{
    if (m_widget) {
        m_widget->setParentItem(0);
        delete m_widget;
    }

    m_widget = widget;
    if (m_widget) {
        m_widget->setParentItem(this);

        qDebug() << "ViewportItem::setWidget ";
#if (QT_VERSION < QT_VERSION_CHECK(4, 7, 0))
        // before resize to contents
        if (qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
            // felipe: the set zoom will reset this
            // connect(m_widget, SIGNAL(loadFinished(bool)), this, SLOT(resizeWebViewToFrame()));
            resizeWebViewToFrame();
        }
#endif

    }
}

bool ViewportItem::sceneEventFilter(QGraphicsItem *i, QEvent *e) {
#ifdef Q_WS_MAEMO_5
    bool res = false;
    if (i && (i == m_widget) && m_widget->isEnabled()) {
        switch (e->type()) {
        case QEvent::GraphicsSceneMousePress:
        case QEvent::GraphicsSceneMouseMove:
        case QEvent::GraphicsSceneMouseRelease:
        case QEvent::GraphicsSceneMouseDoubleClick: {
            res = handleMouseEvent(static_cast<QGraphicsSceneMouseEvent *>(e));
            break;
        }
        default:
            break;
        }
    }
    // prevent text selection and image dragging
    if (e->type() == QEvent::GraphicsSceneMouseMove) {
        return true;
    }
    return res ? true : QGraphicsWidget::sceneEventFilter(i, e);
#else
    return QGraphicsWidget::sceneEventFilter(i,e);
#endif
}

QSize ViewportItem::viewportSize() const
{
    return size().toSize();
}

QPoint ViewportItem::maximumScrollPosition() const
{
    QSizeF s = m_widget ? m_widget->size() - size() : QSize(0, 0);
    return QPoint(qMax(0, int(s.width())), qMax(0, int(s.height())));
}

QPoint ViewportItem::scrollPosition() const
{
    return m_widget ? -m_widget->pos().toPoint() + m_overShoot : QPoint();
}

void ViewportItem::setScrollPosition(const QPoint &p, const QPoint &overShoot) {
    m_overShoot = overShoot;
    if (!m_widget) return;
    m_widget->setPos(-p + m_overShoot);
}

void ViewportItem::cancelLeftMouseButtonPress(const QPoint & /*globalPressPos*/) {
    //TODO: although my test have shown, that this seems like it isn't necessary
}

#if (QT_VERSION < QT_VERSION_CHECK(4, 7, 0))
void ViewportItem::resizeWebViewToFrame() {
    qDebug() << "resizeWebViewToFrame";
    if (QGraphicsWebView *view = qgraphicsitem_cast<QGraphicsWebView *>(m_widget)) {
        if (view->page() && view->page()->mainFrame()) {
            // reset position and resize to "min" to force recalculation
            m_widget->setPos(QPoint(0,0) + m_overShoot);
            m_widget->resize(size());

            // calculate
            QSizeF s = view->page()->mainFrame()->contentsSize();
            qDebug() << "Content Size: " << s;
            s = s.expandedTo(size());
            view->setGeometry(QRectF(QPointF(0,0), s)); //  view->geometry().topLeft()
        }
    }
}
#endif

