/*
 * tangle-stylable.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-stylable.h"

/**
 * SECTION:tangle-stylable
 * @Short_description: An interface for objects that want to control how a style is applied
 * @Title: TangleStylable
 */
 
GType tangle_stylable_get_type(void) {
	static GType object_type = 0;
	
	if (!object_type) {
		static const GTypeInfo object_info = {
			sizeof(TangleStylableIface),
			NULL,	/* base init */
			NULL,	/* base finalize */
		};
		object_type = g_type_register_static(G_TYPE_INTERFACE, "TangleStylable", &object_info, 0);
	}

	return object_type;
}

gboolean tangle_stylable_apply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	gboolean handled = FALSE;
	TangleStylableIface* stylable_iface;

	g_return_val_if_fail(TANGLE_IS_STYLABLE(stylable), FALSE);
	
	stylable_iface = TANGLE_STYLABLE_GET_IFACE(stylable);
	if (stylable_iface->apply_stylesheet) {
		handled = stylable_iface->apply_stylesheet(stylable, stylesheet);
	}
	
	return handled;
}

gboolean tangle_stylable_unapply_stylesheet(TangleStylable* stylable, TangleStylesheet* stylesheet) {
	gboolean handled = FALSE;
	TangleStylableIface* stylable_iface;
	
	g_return_val_if_fail(TANGLE_IS_STYLABLE(stylable), FALSE);
	
	stylable_iface = TANGLE_STYLABLE_GET_IFACE(stylable);
	if (stylable_iface->unapply_stylesheet) {
		handled = stylable_iface->unapply_stylesheet(stylable, stylesheet);
	}
	
	return handled;
}

gboolean tangle_stylable_set_style_property(TangleStylable* stylable, const gchar* name, const GValue* value) {
	gboolean handled = FALSE;
	TangleStylableIface* stylable_iface;
	
	g_return_val_if_fail(TANGLE_IS_STYLABLE(stylable), FALSE);
	
	stylable_iface = TANGLE_STYLABLE_GET_IFACE(stylable);
	if (stylable_iface->set_style_property) {
		handled = stylable_iface->set_style_property(stylable, name, value);
	}

	return handled;
}

gboolean tangle_stylable_unset_style_property(TangleStylable* stylable, const gchar* name, const GValue* value) {
	gboolean handled = FALSE;
	TangleStylableIface* stylable_iface;
	
	stylable_iface = TANGLE_STYLABLE_GET_IFACE(stylable);
	g_return_val_if_fail(TANGLE_IS_STYLABLE(stylable), FALSE);
	
	if (stylable_iface->unset_style_property) {
		handled = stylable_iface->unset_style_property(stylable, name, value);
	}

	return handled;
}

gboolean tangle_stylable_is_style_property(TangleStylable* stylable, const gchar* name, const GValue* value) {
	gboolean retvalue;
	TangleStylableIface* stylable_iface;
	
	g_return_val_if_fail(TANGLE_IS_STYLABLE(stylable), FALSE);
	
	stylable_iface = TANGLE_STYLABLE_GET_IFACE(stylable);
	if (stylable_iface->is_style_property) {
		retvalue = stylable_iface->is_style_property(stylable, name, value);
	} else {
		retvalue = tangle_object_has_property_value(stylable, name, value);
	}
	
	return retvalue;
}

gboolean tangle_object_has_property_value(GObject* object, const gchar* name, GValue* value) {
	gboolean retvalue = FALSE;
	GParamSpec* param_spec;
	GValue current_value = { 0 };
	
	g_return_val_if_fail(G_IS_OBJECT(object), FALSE);
	
	param_spec = g_object_class_find_property(G_OBJECT_GET_CLASS(object), name);
	g_object_get_property(object, name, &current_value);
	if (!g_param_values_cmp(param_spec, &current_value, value)) {
		retvalue = TRUE;
	}
	g_value_unset(&current_value);

	return retvalue;
}
