/*
 * tangle-drag-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-drag-action.h"
#include "tangle-drop-action.h"
#include "tangle-actor.h"
#include "tangle-widget.h"
#include "marshalers.h"

G_DEFINE_TYPE(TangleDragAction, tangle_drag_action, CLUTTER_TYPE_DRAG_ACTION);

enum {
	PROP_0,
	PROP_USE_CLONE
};

enum {
	DROPPED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

struct _TangleDragActionPrivate {
	guchar opacity_dragged;

	ClutterActor* actor;
	ClutterActor* clone;
	ClutterActor* actor_under;
	TangleDropAction* drop_action;
	gulong enabled_notify_handler_id;
	
	guint use_clone : 1;

	guint descendant_dragging : 1;
	guint meta_enabled : 1;
};

static gboolean change_actor_under(TangleDragAction* drag_action);
static void on_notify_enabled(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void on_notify_descendant_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data);
static void handle_enabled(TangleDragAction* drag_action);

ClutterAction* tangle_drag_action_new() {

	return CLUTTER_ACTION(g_object_new(TANGLE_TYPE_DRAG_ACTION, NULL));
}

ClutterActor* tangle_drag_action_get_drag_actor(TangleDragAction* drag_action) {
	ClutterActor* drag_actor;

	g_return_val_if_fail(TANGLE_IS_DRAG_ACTION(drag_action), NULL);
	
	if (!(drag_actor = clutter_drag_action_get_drag_handle(CLUTTER_DRAG_ACTION(drag_action)))) {
		drag_actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(drag_action));
	}
	
	return drag_actor;
}

static void tangle_drag_action_set_actor(ClutterActorMeta* actor_meta, ClutterActor* actor) {
	TangleDragAction* drag_action;
	
	drag_action = TANGLE_DRAG_ACTION(actor_meta);
	
	if (drag_action->priv->actor) {
		if (TANGLE_IS_WIDGET(drag_action->priv->actor)) {
			g_signal_handlers_disconnect_by_func(drag_action->priv->actor, G_CALLBACK(on_notify_descendant_dragging), drag_action);
		}
		g_object_unref(drag_action->priv->actor);
	}
	drag_action->priv->actor = actor;
	if (drag_action->priv->actor) {
		if (TANGLE_IS_WIDGET(drag_action->priv->actor)) {
			g_signal_connect(drag_action->priv->actor, "notify::descendant-dragging", G_CALLBACK(on_notify_descendant_dragging), drag_action);
		}
		g_object_ref(drag_action->priv->actor);
	}
	
	CLUTTER_ACTOR_META_CLASS(tangle_drag_action_parent_class)->set_actor(actor_meta, actor);
}

static void tangle_drag_action_drag_begin(ClutterDragAction* action, ClutterActor* actor, gfloat event_x, gfloat event_y, ClutterModifierType modifiers) {
	TangleDragAction* drag_action;
	gfloat x, y;

	drag_action = TANGLE_DRAG_ACTION(action);

	if (TANGLE_IS_ACTOR(actor)) {
	        tangle_actor_set_interacting(TANGLE_ACTOR(actor), TRUE);
	        tangle_actor_set_dragging(TANGLE_ACTOR(actor), TRUE);
	}

	if (drag_action->priv->use_clone) {
		g_warn_if_fail(!drag_action->priv->clone);

		drag_action->priv->clone = clutter_clone_new(actor);
		clutter_container_add_actor(CLUTTER_CONTAINER(clutter_actor_get_stage(actor)), drag_action->priv->clone);
		clutter_actor_get_transformed_position(actor, &x, &y);
		clutter_actor_set_position(drag_action->priv->clone, x, y);
		clutter_drag_action_set_drag_handle(action, drag_action->priv->clone);
	}
}

static void tangle_drag_action_drag_motion(ClutterDragAction* action, ClutterActor* actor, gfloat delta_x, gfloat delta_y) {
	TangleDragAction* drag_action;
	
	drag_action = TANGLE_DRAG_ACTION(action);
	
	CLUTTER_DRAG_ACTION_CLASS(tangle_drag_action_parent_class)->drag_motion(action, actor, delta_x, delta_y);

	change_actor_under(drag_action);
	if (drag_action->priv->drop_action) {
		if (tangle_drop_action_drag_motion(drag_action->priv->drop_action, drag_action)) {
			g_object_unref(drag_action->priv->drop_action);
			drag_action->priv->drop_action = NULL;
		}
	}		
}

static void tangle_drag_action_drag_end(ClutterDragAction* action, ClutterActor* actor, gfloat event_x, gfloat event_y, ClutterModifierType modifiers) {
	TangleDragAction* drag_action;
	gboolean dropped_emitted = FALSE;

	drag_action = TANGLE_DRAG_ACTION(action);

	if (TANGLE_IS_ACTOR(actor)) {
	        tangle_actor_set_interacting(TANGLE_ACTOR(actor), FALSE);
	        tangle_actor_set_dragging(TANGLE_ACTOR(actor), FALSE);
	}
	
	change_actor_under(drag_action);
	if (drag_action->priv->drop_action) {
		if (!tangle_drop_action_drag_end(drag_action->priv->drop_action, drag_action)) {
			tangle_drop_action_dropped(drag_action->priv->drop_action, drag_action);
			g_signal_emit(action, signals[DROPPED], 0, drag_action->priv->drop_action);
			dropped_emitted = TRUE;
		}

		g_object_unref(drag_action->priv->drop_action);
		drag_action->priv->drop_action = NULL;
	}

	if (!dropped_emitted) {
		g_signal_emit(action, signals[DROPPED], 0, NULL);
	}
	
	if (drag_action->priv->clone) {
		clutter_actor_destroy(drag_action->priv->clone);
		drag_action->priv->clone = NULL;
	}
}

static void tangle_drag_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleDragAction* drag_action;
	
	drag_action = TANGLE_DRAG_ACTION(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_drag_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleDragAction* drag_action;

	drag_action = TANGLE_DRAG_ACTION(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_drag_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_drag_action_parent_class)->finalize(object);
}

static void tangle_drag_action_dispose(GObject* object) {
        TangleDragAction* drag_action;

	drag_action = TANGLE_DRAG_ACTION(object);

	if (drag_action->priv->actor) {
		if (TANGLE_IS_WIDGET(drag_action->priv->actor)) {
			g_signal_handlers_disconnect_by_func(drag_action->priv->actor, G_CALLBACK(on_notify_descendant_dragging), drag_action);
		}
		g_object_unref(drag_action->priv->actor);
		drag_action->priv->actor = NULL;
	}

	G_OBJECT_CLASS(tangle_drag_action_parent_class)->dispose(object);
}

static void tangle_drag_action_class_init(TangleDragActionClass* drag_action_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(drag_action_class);
	ClutterActorMetaClass* actor_meta_class = CLUTTER_ACTOR_META_CLASS(drag_action_class);
	ClutterDragActionClass* clutter_drag_action_class = CLUTTER_DRAG_ACTION_CLASS(drag_action_class);

	gobject_class->finalize = tangle_drag_action_finalize;
	gobject_class->dispose = tangle_drag_action_dispose;
	gobject_class->set_property = tangle_drag_action_set_property;
	gobject_class->get_property = tangle_drag_action_get_property;
	
	actor_meta_class->set_actor = tangle_drag_action_set_actor;
	
	clutter_drag_action_class->drag_begin = tangle_drag_action_drag_begin;
	clutter_drag_action_class->drag_motion = tangle_drag_action_drag_motion;
	clutter_drag_action_class->drag_end = tangle_drag_action_drag_end;

	/**
	 * TangleDrargAction::dropped:
	 * @drag_action: the object which received the signal
	 * @drop_action: a #TangleDropAction that received the dragged object or NULL if the drop failed
	 *
	 * This signal is emitted after the :drag-end signal.
	 * When a #ClutterActor with #TangleDragAction is dropped over
	 * a #ClutterActor that has #TangleDropAction i.e. mouse button is released,
	 * over the actor, the @drop_action parameter contains the #TangleDropAction.
	 * Otherwise it is NULL.
	 */
	signals[DROPPED] = g_signal_new("dropped", G_TYPE_FROM_CLASS(gobject_class),
	                                G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(TangleDragActionClass, dropped),
					NULL, NULL,
					tangle_marshal_VOID__OBJECT,
					G_TYPE_NONE, 1,
					TANGLE_TYPE_DROP_ACTION);


	g_type_class_add_private (gobject_class, sizeof (TangleDragActionPrivate));
}

static void tangle_drag_action_init(TangleDragAction* drag_action) {
	drag_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(drag_action, TANGLE_TYPE_DRAG_ACTION, TangleDragActionPrivate);
	drag_action->priv->use_clone = 1;
	
        drag_action->priv->meta_enabled = clutter_actor_meta_get_enabled(CLUTTER_ACTOR_META(drag_action));
	drag_action->priv->enabled_notify_handler_id = g_signal_connect(drag_action, "notify::enabled", G_CALLBACK(on_notify_enabled), NULL);
}

static TangleDropAction* get_drop_action_from_actor_hierarchy(ClutterActor* actor) {
	TangleDropAction* drop_action = NULL;
	ClutterActor* parent;
	
	drop_action = TANGLE_DROP_ACTION(tangle_actor_get_action_by_type(actor, TANGLE_TYPE_DROP_ACTION));
	if (!drop_action && (parent = clutter_actor_get_parent(actor))) {
		drop_action = get_drop_action_from_actor_hierarchy(parent);
	}
	
	return drop_action;
}


static gboolean change_actor_under(TangleDragAction* drag_action) {
	gboolean retvalue = FALSE;
	ClutterActor* drag_handle;
	gfloat x, y, w, h;
	ClutterActor* actor;
	
	drag_handle = tangle_drag_action_get_drag_actor(drag_action);
		
	clutter_actor_get_transformed_position(drag_handle, &x, &y);
	clutter_actor_get_size(drag_handle, &w, &h);
	actor = clutter_stage_get_actor_at_pos(CLUTTER_STAGE(clutter_actor_get_stage(drag_handle)), CLUTTER_PICK_REACTIVE, x + w / 2, y + h / 2);

	if (drag_action->priv->actor_under != actor) {
		if (drag_action->priv->actor_under) {
			g_object_unref(drag_action->priv->actor_under);
		}
		drag_action->priv->actor_under = actor;
		if (drag_action->priv->actor_under) {
			g_object_ref(drag_action->priv->actor_under);
		}
		
		if (drag_action->priv->drop_action) {
			tangle_drop_action_drag_end(drag_action->priv->drop_action, drag_action);
			g_object_unref(drag_action->priv->drop_action);
			drag_action->priv->drop_action = NULL;
		}
		if (drag_action->priv->actor_under &&
		    (drag_action->priv->drop_action = get_drop_action_from_actor_hierarchy(drag_action->priv->actor_under))) {			
			if (tangle_drop_action_drag_begin(drag_action->priv->drop_action, drag_action)) {
				drag_action->priv->drop_action = NULL;
			} else {
				g_object_ref(drag_action->priv->drop_action);
			}
		}

		retvalue = TRUE;
	}

	return retvalue;
}

static void on_notify_enabled(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleDragAction* drag_action;
	
	drag_action = TANGLE_DRAG_ACTION(object);
	
	drag_action->priv->meta_enabled = clutter_actor_meta_get_enabled(CLUTTER_ACTOR_META(drag_action));
	handle_enabled(drag_action);
}

static void on_notify_descendant_dragging(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	handle_enabled(TANGLE_DRAG_ACTION(user_data));
}

static void handle_enabled(TangleDragAction* drag_action) {
	ClutterActor* actor;
	gboolean descendant_dragging;
	
	actor = clutter_actor_meta_get_actor(CLUTTER_ACTOR_META(drag_action));
	if (TANGLE_IS_WIDGET(actor) && tangle_widget_get_descendant_dragging(TANGLE_WIDGET(actor))) {
		descendant_dragging = TRUE;
	} else {
		descendant_dragging = FALSE;
	}
	
	g_signal_handler_block(drag_action, drag_action->priv->enabled_notify_handler_id);	
	clutter_actor_meta_set_enabled(CLUTTER_ACTOR_META(drag_action), !descendant_dragging && drag_action->priv->meta_enabled);
	g_signal_handler_unblock(drag_action, drag_action->priv->enabled_notify_handler_id);	
}
