/*
 * tangle-compound-style.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-compound-style.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-compound-style
 * @Short_description: A style that is composed from other styles
 * @Title: TangleComboundStyle
 */

G_DEFINE_TYPE(TangleCompoundStyle, tangle_compound_style, TANGLE_TYPE_STYLE);

enum {
	PROP_0,
	PROP_STYLESHEET
};

struct _TangleCompoundStylePrivate {
	TangleStylesheet* stylesheet;
	
	GHashTable* style_property_references;
	GHashTable* overriden_style_property_references;
};

static void add_style_property_reference(const gchar* name, const GValue* value, gpointer user_data);
static void remove_style_property_reference(const gchar* name, const GValue* value, gpointer user_data);

TangleCompoundStyle* tangle_compound_style_new(GType for_type) {

	return TANGLE_COMPOUND_STYLE(g_object_new(TANGLE_TYPE_COMPOUND_STYLE, "for-type", for_type, NULL));
}

TangleCompoundStyle* tangle_compound_style_new_with_stylesheet(GType for_type, TangleStylesheet* stylesheet) {

	return TANGLE_COMPOUND_STYLE(g_object_new(TANGLE_TYPE_COMPOUND_STYLE, "for-type", for_type, "stylesheet", stylesheet, NULL));
}

void tangle_compound_style_add_style(TangleCompoundStyle* compound_style, TangleStyle* style) {
	TanglePointers* pointers;

	g_return_if_fail(TANGLE_IS_COMPOUND_STYLE(compound_style));
	
	pointers = tangle_pointers_new(2, compound_style, style);
	tangle_style_foreach_style_property(style, add_style_property_reference, pointers);
	tangle_pointers_free(pointers, 2);
}

void tangle_compound_style_remove_style(TangleCompoundStyle* compound_style, TangleStyle* style) {
	TanglePointers* pointers;

	g_return_if_fail(TANGLE_IS_COMPOUND_STYLE(compound_style));
	
	pointers = tangle_pointers_new(2, compound_style, style);
	tangle_style_foreach_style_property(style, remove_style_property_reference, pointers);
	tangle_pointers_free(pointers, 2);
}

static gboolean tangle_compound_style_get_style_property(TangleStyle* style, const gchar* name, GValue* value) {
	gboolean found = TRUE;
	TangleCompoundStyle* compound_style;
	TangleStyle* referenced_style;

	if (!TANGLE_STYLE_CLASS(tangle_compound_style_parent_class)->get_style_property(style, name, value)) {
		compound_style = TANGLE_COMPOUND_STYLE(style);
		referenced_style = g_hash_table_lookup(compound_style->priv->style_property_references, name);
		if (referenced_style) {
			found = tangle_style_get_style_property(referenced_style, name, value);
		}
	}

	return found;
}

static void tangle_compound_style_set_style_property(TangleStyle* style, const gchar* name, const GValue* value) {
	TangleCompoundStyle* compound_style;
	TangleStyle* referenced_style;

	compound_style = TANGLE_COMPOUND_STYLE(style);
	if (value) {
		referenced_style = g_hash_table_lookup(compound_style->priv->style_property_references, name);
		if (referenced_style) {
			g_hash_table_insert(compound_style->priv->overriden_style_property_references, g_strdup(name), g_object_ref(referenced_style));
			g_hash_table_remove(compound_style->priv->style_property_references, name);
		}
	} else {
		referenced_style = g_hash_table_lookup(compound_style->priv->overriden_style_property_references, name);
		if (referenced_style) {
			g_hash_table_insert(compound_style->priv->style_property_references, g_strdup(name), g_object_ref(referenced_style));
			g_hash_table_remove(compound_style->priv->overriden_style_property_references, name);
		}
	}
			
	TANGLE_STYLE_CLASS(tangle_compound_style_parent_class)->set_style_property(style, name, value);
}

static void call_callback(gpointer key, gpointer value, gpointer user_data) {
	const gchar* name;
	TangleStyle* style;
	TanglePointers* pointers;
	TanglePropertyCallback callback;
	gpointer data;
	GValue property_value = { 0 };
	
	name = (const gchar*)key;
	style = TANGLE_STYLE(value);
	pointers = TANGLE_POINTERS(user_data);
	tangle_pointers_get(pointers, 2, &callback, &data);
	tangle_style_get_style_property(style, name, &property_value);
	callback(name, &property_value, data);
	g_value_unset(&property_value);
}

static void tangle_compound_style_foreach_style_property(TangleStyle* style, TanglePropertyCallback callback, gpointer user_data) {
	TangleCompoundStyle* compound_style;
	TanglePointers* pointers;
	GList* list_item;

	TANGLE_STYLE_CLASS(tangle_compound_style_parent_class)->foreach_style_property(style, callback, user_data);
	
	pointers = tangle_pointers_new(2, callback, user_data);
	compound_style = TANGLE_COMPOUND_STYLE(style);
	g_hash_table_foreach(compound_style->priv->style_property_references, call_callback, pointers);
	tangle_pointers_free(pointers, 2);
}

static void tangle_compound_style_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleCompoundStyle* compound_style;
	
	compound_style = TANGLE_COMPOUND_STYLE(object);

	switch (prop_id) {
		case PROP_STYLESHEET:
			compound_style->priv->stylesheet = g_value_get_object(value);
			g_object_ref(compound_style->priv->stylesheet);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_compound_style_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleCompoundStyle* compound_style;

	compound_style = TANGLE_COMPOUND_STYLE(object);

        switch (prop_id) {
		case PROP_STYLESHEET:
			g_value_set_object(value, compound_style->priv->stylesheet);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_compound_style_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_compound_style_parent_class)->finalize(object);
}

static void tangle_compound_style_dispose(GObject* object) {
	TangleCompoundStyle* compound_style;
	
	compound_style = TANGLE_COMPOUND_STYLE(object);
	
	TANGLE_UNREF_AND_NULLIFY_OBJECT(compound_style->priv->stylesheet);
	
	if (compound_style->priv->style_property_references) {
		g_hash_table_destroy(compound_style->priv->style_property_references);
		compound_style->priv->style_property_references = NULL;
	}
	if (compound_style->priv->overriden_style_property_references) {
		g_hash_table_destroy(compound_style->priv->overriden_style_property_references);
		compound_style->priv->overriden_style_property_references = NULL;
	}

	G_OBJECT_CLASS(tangle_compound_style_parent_class)->dispose(object);
}

static void tangle_compound_style_class_init(TangleCompoundStyleClass* compound_style_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(compound_style_class);
	TangleStyleClass* style_class = TANGLE_STYLE_CLASS(compound_style_class);

	gobject_class->finalize = tangle_compound_style_finalize;
	gobject_class->dispose = tangle_compound_style_dispose;
	gobject_class->set_property = tangle_compound_style_set_property;
	gobject_class->get_property = tangle_compound_style_get_property;

	style_class->set_style_property = tangle_compound_style_set_style_property;
	style_class->get_style_property = tangle_compound_style_get_style_property;
	style_class->foreach_style_property = tangle_compound_style_foreach_style_property;

	/**
	 * TangleCompoundStyle:stylesheet:
	 */
	g_object_class_install_property(gobject_class, PROP_STYLESHEET,
	                                g_param_spec_object("stylesheet",
	                                "Stylesheet",
	                                "The stylesheet that has instantiated this style",
	                                TANGLE_TYPE_STYLESHEET,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleCompoundStylePrivate));
}

static void tangle_compound_style_init(TangleCompoundStyle* compound_style) {
	compound_style->priv = G_TYPE_INSTANCE_GET_PRIVATE(compound_style, TANGLE_TYPE_COMPOUND_STYLE, TangleCompoundStylePrivate);
	compound_style->priv->style_property_references = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);
	compound_style->priv->overriden_style_property_references = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_object_unref);
}

static void add_style_property_reference(const gchar* name, const GValue* value, gpointer user_data) {
	TanglePointers* pointers;
	TangleCompoundStyle* compound_style;
	TangleStyle* style;
	GValue property_value = { 0 };
	
	pointers = TANGLE_POINTERS(user_data);
	tangle_pointers_get(pointers, 2, &compound_style, &style);
	if (g_hash_table_lookup(compound_style->priv->overriden_style_property_references, name) ||
	    TANGLE_STYLE_CLASS(tangle_compound_style_parent_class)->get_style_property(TANGLE_STYLE(compound_style), name, &property_value)) {
		g_hash_table_insert(compound_style->priv->overriden_style_property_references, g_strdup(name), g_object_ref(style));
		g_value_unset(&property_value);
	} else {
		g_hash_table_insert(compound_style->priv->style_property_references, g_strdup(name), g_object_ref(style));
	}
}

static void remove_style_property_reference(const gchar* name, const GValue* value, gpointer user_data) {
	TanglePointers* pointers;
	TangleCompoundStyle* compound_style;
	TangleStyle* style;
	
	pointers = TANGLE_POINTERS(user_data);
	tangle_pointers_get(pointers, 2, &compound_style, &style);
	if (g_hash_table_lookup(compound_style->priv->style_property_references, name) == style) {	
		g_hash_table_remove(compound_style->priv->style_property_references, name);
	} else if (g_hash_table_lookup(compound_style->priv->overriden_style_property_references, name) == style) {	
		g_hash_table_remove(compound_style->priv->overriden_style_property_references, name);
	}
}

