/*
 * tangle-trick.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-trick.h"

G_DEFINE_ABSTRACT_TYPE(TangleTrick, tangle_trick, TANGLE_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_LAYOUT
};

struct _TangleTrickPrivate {
	TangleLayout* layout;
};

TangleLayout* tangle_trick_get_layout(TangleTrick* trick) {

	return trick->priv->layout;
}

void tangle_trick_set_layout(TangleTrick* trick, TangleLayout* layout) {
	g_return_if_fail(TANGLE_IS_TRICK(trick));
	
	if (trick->priv->layout != layout) {
		if (layout) {
			g_return_if_fail(trick->priv->layout == NULL);
			
			trick->priv->layout = layout;
			g_object_add_weak_pointer(G_OBJECT(trick->priv->layout), (gpointer*)&trick->priv->layout);
		} else {
			g_return_if_fail(trick->priv->layout != NULL);
			
			g_object_remove_weak_pointer(G_OBJECT(trick->priv->layout), (gpointer*)&trick->priv->layout);
			trick->priv->layout = NULL;
		}
		
		g_object_notify(G_OBJECT(trick), "layout");
	}
}

void tangle_trick_manipulate_preferred_width(TangleTrick* trick, TangleWidget* widget, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	TangleTrickClass* trick_class;
	
	trick_class = TANGLE_TRICK_GET_CLASS(trick);
	
	if (trick_class->manipulate_preferred_width) {
		trick_class->manipulate_preferred_width(trick, widget, for_height, min_width_p, natural_width_p);
	}
}

void tangle_trick_manipulate_preferred_height(TangleTrick* trick, TangleWidget* widget, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	TangleTrickClass* trick_class;
	
	trick_class = TANGLE_TRICK_GET_CLASS(trick);
	
	if (trick_class->manipulate_preferred_height) {
		TANGLE_TRICK_GET_CLASS(trick)->manipulate_preferred_height(trick, widget, for_width, min_height_p, natural_height_p);
	}
}

void tangle_trick_manipulate_allocation(TangleTrick* trick, TangleWidget* widget, ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleTrickClass* trick_class;
	
	trick_class = TANGLE_TRICK_GET_CLASS(trick);
	
	if (trick_class->manipulate_allocation) {
		TANGLE_TRICK_GET_CLASS(trick)->manipulate_allocation(trick, widget, box, flags);
	}
}

void tangle_trick_manipulate_child_allocation(TangleTrick* trick, TangleWidget* widget, ClutterActor* actor, ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleTrickClass* trick_class;
	
	trick_class = TANGLE_TRICK_GET_CLASS(trick);
	
	if (trick_class->manipulate_child_allocation) {
		TANGLE_TRICK_GET_CLASS(trick)->manipulate_child_allocation(trick, widget, actor, box, flags);
	}
}

void tangle_trick_layout_changed(TangleTrick* trick) {
	if (trick->priv->layout) {
		tangle_layout_changed(trick->priv->layout);
	}
}

static void tangle_trick_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTrick* trick;
	
	trick = TANGLE_TRICK(object);

	switch (prop_id) {
		case PROP_LAYOUT:
			tangle_trick_set_layout(trick, TANGLE_LAYOUT(g_value_get_object(value)));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_trick_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTrick* trick;

	trick = TANGLE_TRICK(object);

        switch (prop_id) {
		case PROP_LAYOUT:
			g_value_set_object(value, trick->priv->layout);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_trick_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_trick_parent_class)->finalize(object);
}

static void tangle_trick_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_trick_parent_class)->dispose(object);
}

static void tangle_trick_class_init(TangleTrickClass* trick_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(trick_class);

	gobject_class->finalize = tangle_trick_finalize;
	gobject_class->dispose = tangle_trick_dispose;
	gobject_class->set_property = tangle_trick_set_property;
	gobject_class->get_property = tangle_trick_get_property;

	/**
	 * TangleTrick:widget:
	 *
	 * The #TangleLayout that is manipulated with this trick.
	 */
	g_object_class_install_property(gobject_class, PROP_LAYOUT,
	                                g_param_spec_object ("layout",
	                                "Layout",
	                                "The layout that is manipulated with this trick",
	                                TANGLE_TYPE_LAYOUT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleTrickPrivate));
}

static void tangle_trick_init(TangleTrick* trick) {
	trick->priv = G_TYPE_INSTANCE_GET_PRIVATE(trick, TANGLE_TYPE_TRICK, TangleTrickPrivate);
}

