/*
 * tangle-curved-transition.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-curved-transition.h"
#include "tangle-vault.h"

/**
 * SECTION:tangle-curved-transition
 * @Short_description: A transition that moves an actor through predefined points
 * @Title: TangleCurvedTransition
 */

G_DEFINE_TYPE(TangleCurvedTransition, tangle_curved_transition, TANGLE_TYPE_EASING_TRANSITION);

enum {
	PROP_0,
	PROP_SHOW_FROM_POINT_MODE,
	PROP_SHOW_FROM_X,
	PROP_SHOW_FROM_Y,
	PROP_MOVE_VIA_POINT_MODE,
	PROP_MOVE_VIA_X,
	PROP_MOVE_VIA_Y,
	PROP_HIDE_TO_POINT_MODE,
	PROP_HIDE_TO_X,
	PROP_HIDE_TO_Y
};

struct _TangleCurvedTransitionPrivate {
	TangleCurvedTransitionPointMode show_from_point_mode;
	gfloat show_from_x;
	gfloat show_from_y;
	TangleCurvedTransitionPointMode move_via_point_mode;
	gfloat move_via_x;
	gfloat move_via_y;
	TangleCurvedTransitionPointMode hide_to_point_mode;
	gfloat hide_to_x;
	gfloat hide_to_y;
};

TangleTransition* tangle_curved_transition_new(gulong mode, guint duration) {

	return TANGLE_TRANSITION(g_object_new(TANGLE_TYPE_CURVED_TRANSITION, "mode", mode, "duration", duration, NULL));
}

void tangle_curved_transition_get_show_from(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode* show_from_point_mode_return, gfloat* show_from_x_return, gfloat* show_from_y_return) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (show_from_point_mode_return) {
		*show_from_point_mode_return = curved_transition->priv->show_from_point_mode;
	}
	if (show_from_x_return) {
		*show_from_x_return = curved_transition->priv->show_from_x;
	}
	if (show_from_y_return) {
		*show_from_y_return = curved_transition->priv->show_from_y;
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

void tangle_curved_transition_set_show_from(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode show_from_point_mode, gfloat show_from_x, gfloat show_from_y) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (curved_transition->priv->show_from_point_mode != show_from_point_mode) {
		curved_transition->priv->show_from_point_mode = show_from_point_mode;
		g_object_notify(G_OBJECT(curved_transition), "show-from-point-mode");
	}
	if (curved_transition->priv->show_from_x != show_from_x) {
		curved_transition->priv->show_from_x = show_from_x;
		g_object_notify(G_OBJECT(curved_transition), "show-from-x");
	}
	if (curved_transition->priv->show_from_y != show_from_y) {
		curved_transition->priv->show_from_y = show_from_y;
		g_object_notify(G_OBJECT(curved_transition), "show-from-y");
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

void tangle_curved_transition_get_move_via(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode* move_via_point_mode_return, gfloat* move_via_x_return, gfloat* move_via_y_return) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (move_via_point_mode_return) {
		*move_via_point_mode_return = curved_transition->priv->move_via_point_mode;
	}
	if (move_via_x_return) {
		*move_via_x_return = curved_transition->priv->move_via_x;
	}
	if (move_via_y_return) {
		*move_via_y_return = curved_transition->priv->move_via_y;
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

void tangle_curved_transition_set_move_via(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode move_via_point_mode, gfloat move_via_x, gfloat move_via_y) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (curved_transition->priv->move_via_point_mode != move_via_point_mode) {
		curved_transition->priv->move_via_point_mode = move_via_point_mode;
		g_object_notify(G_OBJECT(curved_transition), "move-via-point-mode");
	}
	if (curved_transition->priv->move_via_x != move_via_x) {
		curved_transition->priv->move_via_x = move_via_x;
		g_object_notify(G_OBJECT(curved_transition), "move-via-x");
	}
	if (curved_transition->priv->move_via_y != move_via_y) {
		curved_transition->priv->move_via_y = move_via_y;
		g_object_notify(G_OBJECT(curved_transition), "move-via-y");
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

void tangle_curved_transition_get_hide_to(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode* hide_to_point_mode_return, gfloat* hide_to_x_return, gfloat* hide_to_y_return) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (hide_to_point_mode_return) {
		*hide_to_point_mode_return = curved_transition->priv->hide_to_point_mode;
	}
	if (hide_to_x_return) {
		*hide_to_x_return = curved_transition->priv->hide_to_x;
	}
	if (hide_to_y_return) {
		*hide_to_y_return = curved_transition->priv->hide_to_y;
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

void tangle_curved_transition_set_hide_to(TangleCurvedTransition* curved_transition, TangleCurvedTransitionPointMode hide_to_point_mode, gfloat hide_to_x, gfloat hide_to_y) {
	g_object_freeze_notify(G_OBJECT(curved_transition));

	if (curved_transition->priv->hide_to_point_mode != hide_to_point_mode) {
		curved_transition->priv->hide_to_point_mode = hide_to_point_mode;
		g_object_notify(G_OBJECT(curved_transition), "hide-to-point-mode");
	}
	if (curved_transition->priv->hide_to_x != hide_to_x) {
		curved_transition->priv->hide_to_x = hide_to_x;
		g_object_notify(G_OBJECT(curved_transition), "hide-to-x");
	}
	if (curved_transition->priv->hide_to_y != hide_to_y) {
		curved_transition->priv->hide_to_y = hide_to_y;
		g_object_notify(G_OBJECT(curved_transition), "hide-to-y");
	}

	g_object_thaw_notify(G_OBJECT(curved_transition));
}

static void on_timeline_completed(ClutterTimeline* timeline, gpointer user_data) {
	g_object_unref(timeline);
	g_object_unref(user_data);
}

static void on_alpha_notify(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	ClutterAlpha* alpha;
	TangleVault* vault;
	ClutterPath* path;
	TangleActor* actor;
	ClutterKnot knot;
	
	alpha = CLUTTER_ALPHA(object);
	vault = TANGLE_VAULT(user_data);
	tangle_vault_get(vault, 2, CLUTTER_TYPE_PATH, &path, TANGLE_TYPE_ACTOR, &actor);
	
	clutter_path_get_position(path, clutter_alpha_get_alpha(alpha), &knot);
	g_object_set(actor, "transition-move-x", (gdouble)knot.x, "transition-move-y", (gdouble)knot.y, NULL);
}

static ClutterTimeline* animate_quadratic_path(TangleEasingTransition* easing_transition, TangleActor* actor, gint x1, gint y1, gint x2, gint y2, gint x3, gint y3) {
	ClutterPath* path;
	ClutterTimeline* timeline;
	ClutterAlpha* alpha;
	TangleVault* vault;
	
	path = clutter_path_new();
	clutter_path_add_move_to(path, x1, y1);
	clutter_path_add_curve_to(path,
	                          x2 * 3 / 2 + x1 / 3, y2 * 3 / 2 + y1 / 3,
	                          x2 * 3 / 2 + x3 / 3, y2 * 3 / 2 + y3 / 3,
				  x3, y3);
	
	timeline = clutter_timeline_new(tangle_easing_transition_get_duration(easing_transition));
	alpha = clutter_alpha_new_full(timeline, tangle_easing_transition_get_mode(easing_transition));
	g_signal_connect(timeline, "completed", G_CALLBACK(on_timeline_completed), alpha); /* free both */
	
	vault = tangle_vault_new(2, CLUTTER_TYPE_PATH, path, TANGLE_TYPE_ACTOR, actor);
	tangle_signal_connect_vault(alpha, "notify::alpha", G_CALLBACK(on_alpha_notify), vault); /* drive movement */
	
	clutter_timeline_start(timeline);
	
	g_object_unref(path);
	
	return timeline;
}

static void transform_point(TangleActor* actor, gfloat actor_x, gfloat actor_y, TangleCurvedTransitionPointMode point_mode, gfloat transition_x, gfloat transition_y, gfloat* x_return, gfloat* y_return) {
	ClutterActor* parent;
	gfloat stage_x;
	gfloat stage_y;
	
	switch (point_mode) {
		case TANGLE_CURVED_TRANSITION_ABSOLUTE:
			parent = clutter_actor_get_parent(CLUTTER_ACTOR(actor));
			clutter_actor_get_transformed_position(CLUTTER_ACTOR(parent), &stage_x, &stage_y);
			*x_return = transition_x - stage_x;
			*y_return = transition_y - stage_y;
			break;
		case TANGLE_CURVED_TRANSITION_RELATIVE:
			*x_return = actor_x + transition_x;
			*y_return = actor_y + transition_y;
			break;
		default:
			g_critical("Invalid value for the \"X_point_mode\" property.");
			break;
	}
}

static ClutterTimeline* tangle_curved_transition_animate_actor(TangleTransition* transition, TangleActor* actor, ClutterActorBox* current_box, ClutterActorBox* new_box) {
	ClutterTimeline* timeline = NULL;
	TangleCurvedTransition* curved_transition;
	gulong mode;
	guint duration;
	gfloat move_x, move_y;
	gdouble scale_x, scale_y;
	gfloat x1, y1, x2, y2;
	ClutterAnimation* animation;
	
	curved_transition = TANGLE_CURVED_TRANSITION(transition);

	mode = tangle_easing_transition_get_mode(TANGLE_EASING_TRANSITION(curved_transition));
	duration = tangle_easing_transition_get_duration(TANGLE_EASING_TRANSITION(curved_transition));
	
	if (mode && duration) {
		tangle_actor_get_transition_move(actor, &move_x, &move_y);
		tangle_actor_get_transition_scale(actor, &scale_x, &scale_y);

		if (current_box && new_box) {
		} else if (current_box) {
			switch (curved_transition->priv->hide_to_point_mode) {
				case TANGLE_CURVED_TRANSITION_DISABLED:
					TANGLE_TRANSITION_CLASS(tangle_curved_transition_parent_class)->animate_actor(transition, actor, current_box, new_box);
					break;
				case TANGLE_CURVED_TRANSITION_ABSOLUTE:
				case TANGLE_CURVED_TRANSITION_RELATIVE:
					transform_point(actor, current_box->x1, current_box->y1, curved_transition->priv->show_from_point_mode,
					        	curved_transition->priv->hide_to_x, curved_transition->priv->hide_to_y, &x1, &y1);
					if (curved_transition->priv->move_via_point_mode == TANGLE_CURVED_TRANSITION_DISABLED) {
						g_object_set(actor, "transition-move-x", 0.0, "transition-move-y", 0.0, NULL);
						animation = tangle_actor_animate(actor,
					                                	 tangle_easing_transition_get_mode(TANGLE_EASING_TRANSITION(curved_transition)),
										 tangle_easing_transition_get_duration(TANGLE_EASING_TRANSITION(curved_transition)),
										 "transition-move-x", x2, "transition-move-y", x2, NULL);
						timeline = clutter_animation_get_timeline(animation);
					} else {
						transform_point(actor, new_box->x1, new_box->y1, curved_transition->priv->move_via_point_mode,
					                	curved_transition->priv->move_via_x, curved_transition->priv->move_via_y, &x1, &y1);
						timeline = animate_quadratic_path(TANGLE_EASING_TRANSITION(curved_transition), actor, 0.0, 0.0, x1, y1, x2, y2);
					}
					break;			
				default:
					g_critical("Invalid value for the \"X_point_mode\" property.");
					break;
			}

		} else {
			switch (curved_transition->priv->show_from_point_mode) {
				case TANGLE_CURVED_TRANSITION_DISABLED:
					TANGLE_TRANSITION_CLASS(tangle_curved_transition_parent_class)->animate_actor(transition, actor, current_box, new_box);
					break;
				case TANGLE_CURVED_TRANSITION_ABSOLUTE:
				case TANGLE_CURVED_TRANSITION_RELATIVE:
					transform_point(actor, 0.0, 0.0, curved_transition->priv->show_from_point_mode,
					        	curved_transition->priv->show_from_x, curved_transition->priv->show_from_y, &x1, &y1);
					tangle_actor_set_transition_move(actor, move_x + x1, move_y + y1);
					if (curved_transition->priv->move_via_point_mode == TANGLE_CURVED_TRANSITION_DISABLED) {
						animation = tangle_actor_animate(actor,
					                                	 tangle_easing_transition_get_mode(TANGLE_EASING_TRANSITION(curved_transition)),
										 tangle_easing_transition_get_duration(TANGLE_EASING_TRANSITION(curved_transition)),
										 "transition-move-x", 0.0, "transition-move-y", 0.0, NULL);
						timeline = clutter_animation_get_timeline(animation);
					} else {
						transform_point(actor, x1 / 2, y1 / 2, curved_transition->priv->move_via_point_mode,
					                	curved_transition->priv->move_via_x, curved_transition->priv->move_via_y, &x2, &y2);
						timeline = animate_quadratic_path(TANGLE_EASING_TRANSITION(curved_transition), actor, x1, y1, x2, y2, 0.0, 0.0);
					}
					break;			
				default:
					g_critical("Invalid value for the \"X_point_mode\" property.");
					break;
			}	
		}
	}
	
	return timeline;
}


static void tangle_curved_transition_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleCurvedTransition* curved_transition;
	
	curved_transition = TANGLE_CURVED_TRANSITION(object);

	switch (prop_id) {
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_curved_transition_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleCurvedTransition* curved_transition;

	curved_transition = TANGLE_CURVED_TRANSITION(object);

        switch (prop_id) {
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_curved_transition_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_curved_transition_parent_class)->finalize(object);
}

static void tangle_curved_transition_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_curved_transition_parent_class)->dispose(object);
}

static void tangle_curved_transition_class_init(TangleCurvedTransitionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleTransitionClass* transition_class = TANGLE_TRANSITION_CLASS(klass);

	gobject_class->finalize = tangle_curved_transition_finalize;
	gobject_class->dispose = tangle_curved_transition_dispose;
	gobject_class->set_property = tangle_curved_transition_set_property;
	gobject_class->get_property = tangle_curved_transition_get_property;

	transition_class->animate_actor = tangle_curved_transition_animate_actor;

	g_type_class_add_private (gobject_class, sizeof (TangleCurvedTransitionPrivate));
}

static void tangle_curved_transition_init(TangleCurvedTransition* curved_transition) {
	curved_transition->priv = G_TYPE_INSTANCE_GET_PRIVATE(curved_transition, TANGLE_TYPE_CURVED_TRANSITION, TangleCurvedTransitionPrivate);
	curved_transition->priv->show_from_point_mode = TANGLE_CURVED_TRANSITION_RELATIVE;
	curved_transition->priv->show_from_x = 100.0;
	curved_transition->priv->show_from_y = 50.0;
	curved_transition->priv->move_via_point_mode = TANGLE_CURVED_TRANSITION_RELATIVE;
	curved_transition->priv->move_via_x = 0.0;
	curved_transition->priv->move_via_y = 100.0;
}

