/*
 * tangle-draggable-actor.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-clone.h"


void _clutter_actor_set_opacity_parent(ClutterActor *self, ClutterActor *parent);
void _clutter_actor_set_enable_model_view_transform(ClutterActor *self, gboolean enable);


G_DEFINE_TYPE(TangleClone, tangle_clone, CLUTTER_TYPE_CLONE);

enum {
	PROP_0,
	PROP_SOURCE_PICKABLE
};

struct _TangleClonePrivate {
	guint source_pickable : 1;
};

ClutterActor* tangle_clone_new(ClutterActor* source) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_CLONE, "source", source, NULL));
}

gboolean tangle_clone_get_source_pickable(TangleClone* clone) {

	return clone->priv->source_pickable;
}

void tangle_clone_set_source_pickable(TangleClone* clone, gboolean source_pickable) {
	if (clone->priv->source_pickable != source_pickable) {
		clone->priv->source_pickable = source_pickable;
		clutter_actor_queue_redraw(CLUTTER_ACTOR(clone));
		g_object_notify(G_OBJECT(clone), "source-pickable");
	}
}

static void tangle_clone_pick(ClutterActor* actor, const ClutterColor* color) {
	ClutterActorBox box;
		
	if (clutter_actor_should_pick_paint(actor)) {
		CLUTTER_ACTOR_CLASS(tangle_clone_parent_class)->pick(actor, color);
	} else if (TANGLE_CLONE(actor)->priv->source_pickable) {
		CLUTTER_ACTOR_CLASS(tangle_clone_parent_class)->paint(actor);
	}
}

static void tangle_clone_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleClone* clone;
	
	clone = TANGLE_CLONE(object);

	switch (prop_id) {
		case PROP_SOURCE_PICKABLE:
			tangle_clone_set_source_pickable(clone, g_value_get_boolean(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_clone_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleClone* clone;

	clone = TANGLE_CLONE(object);

        switch (prop_id) {
		case PROP_SOURCE_PICKABLE:
			g_value_set_boolean(value, clone->priv->source_pickable);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_clone_class_init(TangleCloneClass* clone_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(clone_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(clone_class);

	gobject_class->set_property = tangle_clone_set_property;
	gobject_class->get_property = tangle_clone_get_property;
	
	clutter_actor_class->pick = tangle_clone_pick;

	/**
	 * TangleClone:source-pickable:
	 *
	 * Whether the source actor can be picked.
	 */
	g_object_class_install_property(gobject_class, PROP_SOURCE_PICKABLE,
	                                g_param_spec_boolean("source-pickable",
	                                                     "Source pickable",
	                                                     "Whether the source actor can be picked",
	                                                     TRUE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof(TangleClonePrivate));
}

static void tangle_clone_init(TangleClone* clone) {
	clone->priv = G_TYPE_INSTANCE_GET_PRIVATE(clone, TANGLE_TYPE_CLONE, TangleClonePrivate);
	clone->priv->source_pickable = TRUE;
}
