/*
 * tangle-show-actor-action.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */
 
#include "tangle-show-actor-action.h"
#include "tangle-misc.h"

/**
 * SECTION:tangle-show-actor-action
 * @Short_description: An action that makes an actor visible
 * @Title: TangleShowActorAction
 */

G_DEFINE_TYPE(TangleShowActorAction, tangle_show_actor_action, TANGLE_TYPE_ACTION);

enum {
	PROP_0,
	PROP_ACTOR,
	PROP_HIDE_OTHERS
};

struct _TangleShowActorActionPrivate {
	ClutterActor* actor;
	guint hide_others : 1;
};

static void hide_actor(ClutterActor* actor, gpointer user_data);

TangleAction* tangle_show_actor_action_new(void) {

	return TANGLE_ACTION(g_object_new(TANGLE_TYPE_SHOW_ACTOR_ACTION, NULL));
}

ClutterActor* tangle_show_actor_action_get_actor(TangleShowActorAction* action) {

	return action->priv->actor;
}

void tangle_show_actor_action_set_actor(TangleShowActorAction* action, ClutterActor* actor) {
	g_return_if_fail(actor != NULL);
	
	if (action->priv->actor != actor) {
		if (action->priv->actor) {
			g_object_unref(action->priv->actor);
		}

		action->priv->actor = actor;
		g_object_ref(action->priv->actor);

		g_object_notify(G_OBJECT(action), "actor");
	}
}
	
gboolean tangle_show_actor_action_get_hide_others(TangleShowActorAction* action) {

	return action->priv->hide_others;
}

void tangle_show_actor_action_set_hide_others(TangleShowActorAction* action, gboolean hide_others) {
	if (action->priv->hide_others != hide_others) {
		action->priv->hide_others = hide_others;
		
		g_object_notify(G_OBJECT(action), "hide-others");
	}
}

static void tangle_show_actor_action_execute(TangleAction* tangle_action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	TangleShowActorAction* action;
	ClutterActor* parent;
	
	action = TANGLE_SHOW_ACTOR_ACTION(tangle_action);
	
	g_return_if_fail(action->priv->actor != NULL);
	
	if (action->priv->hide_others) {
		parent = clutter_actor_get_parent(action->priv->actor);
		g_warn_if_fail(parent != NULL);
		
		if (CLUTTER_IS_CONTAINER(parent)) {
			clutter_container_foreach(CLUTTER_CONTAINER(parent), hide_actor, action->priv->actor);
		}
	}
	
	if (TANGLE_IS_ACTOR(action->priv->actor)) {
		tangle_actor_show(TANGLE_ACTOR(action->priv->actor));
	} else {
		clutter_actor_show(action->priv->actor);
	}
}

static void tangle_show_actor_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleShowActorAction* action;
	
	action = TANGLE_SHOW_ACTOR_ACTION(object);

	switch (prop_id) {
		case PROP_ACTOR:
			tangle_show_actor_action_set_actor(action, CLUTTER_ACTOR(g_value_get_object(value)));
			break;
		case PROP_HIDE_OTHERS:
			action->priv->hide_others = g_value_get_boolean(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_show_actor_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleShowActorAction* action;

	action = TANGLE_SHOW_ACTOR_ACTION(object);

        switch (prop_id) {
		case PROP_ACTOR:
			g_value_set_object(value, action->priv->actor);
			break;
		case PROP_HIDE_OTHERS:
			g_value_set_boolean(value, action->priv->hide_others);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_show_actor_action_dispose(GObject* object) {
	TangleShowActorAction* action;
	
	action = TANGLE_SHOW_ACTOR_ACTION(object);

	TANGLE_UNREF_AND_NULLIFY_OBJECT(action->priv->actor);
	
	G_OBJECT_CLASS(tangle_show_actor_action_parent_class)->dispose(object);
}

static void tangle_show_actor_action_class_init(TangleShowActorActionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);
	TangleActionClass* action_class = TANGLE_ACTION_CLASS(klass);

	gobject_class->dispose = tangle_show_actor_action_dispose;
	gobject_class->set_property = tangle_show_actor_action_set_property;
	gobject_class->get_property = tangle_show_actor_action_get_property;

	action_class->execute = tangle_show_actor_action_execute;

	/**
	 * TangleShowActorAction:actor:
	 */
	g_object_class_install_property(gobject_class, PROP_ACTOR,
	                                g_param_spec_object("actor",
	                                                    "Actor",
	                                                    "The actor to be shown",
	                                                    CLUTTER_TYPE_ACTOR,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleShowActorAction:hide-other:
	 */
	g_object_class_install_property(gobject_class, PROP_HIDE_OTHERS,
	                                g_param_spec_boolean("hide-others",
	                                                     "Hide others",
	                                                     "Whether to hide other actors in the same container or not",
	                                                     FALSE,
	                                                     G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(TangleShowActorActionPrivate));
}

static void tangle_show_actor_action_init(TangleShowActorAction* show_actor_action) {
	show_actor_action->priv = G_TYPE_INSTANCE_GET_PRIVATE(show_actor_action, TANGLE_TYPE_SHOW_ACTOR_ACTION, TangleShowActorActionPrivate);
}

static void hide_actor(ClutterActor* actor, gpointer user_data) {
	if (actor != user_data) {
		if (TANGLE_IS_ACTOR(actor)) {
			tangle_actor_hide_animated(TANGLE_ACTOR(actor));
		} else {
			clutter_actor_hide(actor);
		}
	}
}
