/*
 * tangle-private.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-private.h"


gboolean _tangle_scriptable_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node, TangleTemplate* template, gboolean* template_applied_return) {
	static ClutterScriptable* previous_scriptable = NULL;

	gboolean retvalue = FALSE;
	gpointer type_class;
	GParamSpec* param_spec;
	const gchar* s;
	GObject* object;
	gchar* id;

	/* This is a hack to identify, when a new scriptable is being parsed and thus, when template parameters must be cleared. */
	if (scriptable != previous_scriptable) {
		previous_scriptable = scriptable;
		if (template) {
			tangle_template_clear_parameter_values(template);
		}
	}
	
	type_class = g_type_class_ref(G_OBJECT_TYPE(scriptable));
	
	if (name[0] == '$') {
		if (!template) {
			g_warning("A template parameter '%s' found, but no template for the object (id: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (tangle_template_are_parameters_complete(template)) {
			g_warning("The value of a template parameter '%s' found but the parameters are already completed for the object (id: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (tangle_template_get_parameter_value(template, name + 1)) {
			g_warning("The value of a template parameter '%s' is defined twice for the object (id: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!clutter_script_parse_node(script, value, name, node, NULL)) {
			g_warning("The value of a template parameter '%s' could not be parsed (object: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!tangle_template_set_parameter_value(template, name + 1, value)) {
			g_warning("A template parameter '%s' does not exist in a template (id: %s) for the object (id: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(template)), clutter_get_script_id(G_OBJECT(scriptable)));
		} else {
			if (tangle_template_are_parameters_complete(template)) {
				tangle_template_apply_properties(template, G_OBJECT(scriptable));
				if (template_applied_return) {
					*template_applied_return = TRUE;
				}
			}

			retvalue = TRUE;
		}
	} else if (name[0] == '@') {
		if (!clutter_script_parse_node(script, value, name, node, NULL)) {
			g_warning("The value of a signal handler property '%s' could not be parsed (object: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!G_VALUE_HOLDS_STRING(value)) {
			g_warning("The value of a signal handler property '%s' of an object (id: %s) was not string, skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else if (!tangle_signal_connect_dynamically_from_script(G_OBJECT(scriptable), name + 1, g_value_get_string(value), script)) {
			g_warning("Failed to connect signal handler property '%s' of an object (id: %s), skipped.", name + 1, clutter_get_script_id(G_OBJECT(scriptable)));
		} else {
			retvalue = TRUE;
		}
	} else if ((param_spec = g_object_class_find_property(G_OBJECT_CLASS(type_class), name)) &&
	           G_IS_PARAM_SPEC_OBJECT(param_spec) &&
		   !g_type_is_a(G_PARAM_SPEC_VALUE_TYPE(param_spec), TANGLE_TYPE_TEMPLATE) &&
	           (s = json_node_get_string(node)) &&
		   (object = clutter_script_get_object(script, s)) &&
		   TANGLE_IS_TEMPLATE(object)) {
		if (!g_type_is_a(tangle_template_get_object_type(TANGLE_TEMPLATE(object)), G_PARAM_SPEC_VALUE_TYPE(param_spec))) {
			g_warning("The type of the template (id: %s) does not match to property (name: %s), skipped.", s, name);
		} else if (!(object = tangle_template_create_object(TANGLE_TEMPLATE(object)))) {
			g_warning("Could not instantiate template (id: %s) for property (name: %s), skipped.", s, name);
		} else {
			g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec));
			g_value_set_object(value, object);
			
			retvalue = TRUE;
		}
	}

	g_type_class_unref(type_class);

	return retvalue;
}

gboolean _tangle_scriptable_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	if (name[0] != '$' && name[0] != '@') {
		g_object_set_property(G_OBJECT(scriptable), name, value);
	}
	
	return TRUE;
}

gchar* _tangle_script_lookup_filename(ClutterScript* script, const gchar* filename) {
	gchar* path = NULL;
	gchar* dir;
	
	if (g_path_is_absolute(filename)) {
		path = g_strdup(filename);
	} else if (script) {
		path = clutter_script_lookup_filename(script, filename);
	} else {
		dir = g_get_current_dir();
		path = g_build_filename(dir, filename, NULL);
		if (!g_file_test(path, G_FILE_TEST_EXISTS)) {
			g_free(path);
			path = NULL;
		}
		g_free(dir);
	}
	
	return path;
}
