/*
 * tangle-button-group.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-button-group.h"

/**
 * SECTION:tangle-button-group
 * @Short_description: 	A group of #TangleButtons managed like radio buttons
 * @Title: TangleButtonGroup
 */

G_DEFINE_TYPE(TangleButtonGroup, tangle_button_group, G_TYPE_OBJECT);

enum {
	PROP_0,
	PROP_BUTTON,
	PROP_SELECTED_BUTTON,
	PROP_MEMORY_MANAGED
};

struct _TangleButtonGroupPrivate {
	GList* buttons;
	TangleButton* selected_button;
	
	guint memory_managed : 1;
};

static TangleButtonGroup* new_valist(gboolean memory_managed, TangleButton* button, va_list button_list);
static void on_weak_notify(gpointer user_data, GObject* object);
static void on_notify_selected(GObject* object, GParamSpec* param_spec, gpointer user_data);

TangleButtonGroup* tangle_button_group_new(void) {

	return TANGLE_BUTTON_GROUP(g_object_new(TANGLE_TYPE_BUTTON_GROUP, NULL));
}

TangleButtonGroup* tangle_button_group_new_with_buttons(TangleButton* first_button, ...) {
	TangleButtonGroup* button_group;
	va_list button_list;
	
	va_start(button_list, first_button);
	button_group = new_valist(FALSE, first_button, button_list);
	va_end(button_list);

	return button_group;
}

void tangle_button_group_create(TangleButton* first_button, ...) {
	TangleButtonGroup* button_group;
	va_list button_list;
	
	va_start(button_list, first_button);
	button_group = new_valist(TRUE, first_button, button_list);
	va_end(button_list);

	g_object_unref(button_group);
}

void tangle_button_group_add_button(TangleButtonGroup* button_group, TangleButton* button) {
	if (!g_list_find(button_group->priv->buttons, button)) {
		if (tangle_button_get_selected(button)) {
			if (button_group->priv->selected_button) {
				tangle_button_set_selected(button_group->priv->selected_button, FALSE);
			}
			button_group->priv->selected_button = button;
			g_object_notify(G_OBJECT(button_group), "selected-button");
		}
		
		g_signal_connect(button, "notify::selected", G_CALLBACK(on_notify_selected), button_group);
		button_group->priv->buttons = g_list_prepend(button_group->priv->buttons, button);

		if (button_group->priv->memory_managed) {
			g_object_weak_ref(G_OBJECT(button), (GWeakNotify)g_object_unref, button_group);
			g_object_ref(button_group);
		}
	}
}

void tangle_button_group_remove_button(TangleButtonGroup* button_group, TangleButton* button) {
	GList* button_in_list;
	
	if (!(button_in_list = g_list_find(button_group->priv->buttons, button))) {
		g_warning("Could not remove a button that is not in a button group.");
	} else {
		if (button_group->priv->selected_button == button) {
			button_group->priv->selected_button = NULL;
			g_object_notify(G_OBJECT(button_group), "selected-button");
		}
		
		g_signal_handlers_disconnect_by_func(button, G_CALLBACK(on_notify_selected), button_group);
		button_group->priv->buttons = g_list_delete_link(button_group->priv->buttons, button_in_list);

		if (button_group->priv->memory_managed) {
			g_object_weak_unref(G_OBJECT(button), (GWeakNotify)g_object_unref, button_group);
			g_object_unref(button_group);
		}
	}
}

GList* tangle_button_group_get_buttons(TangleButtonGroup* button_group) {
	
	return g_list_copy(button_group->priv->buttons);
}

TangleButton* tangle_button_group_get_selected_button(TangleButtonGroup* button_group) {

	return button_group->priv->selected_button;
}


static void tangle_button_group_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleButtonGroup* button_group;
	
	button_group = TANGLE_BUTTON_GROUP(object);

	switch (prop_id) {
		case PROP_BUTTON:
			tangle_button_group_add_button(button_group, TANGLE_BUTTON(g_value_get_object(value)));
			break;
		case PROP_MEMORY_MANAGED:
			button_group->priv->memory_managed = g_value_get_boolean(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_button_group_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleButtonGroup* button_group;

	button_group = TANGLE_BUTTON_GROUP(object);

        switch (prop_id) {
		case PROP_SELECTED_BUTTON:
			g_value_set_object(value, button_group->priv->selected_button);
			break;
		case PROP_MEMORY_MANAGED:
			g_value_set_boolean(value, button_group->priv->memory_managed);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_button_group_dispose(GObject* object) {
	

	G_OBJECT_CLASS(tangle_button_group_parent_class)->dispose(object);
}

static void tangle_button_group_class_init(TangleButtonGroupClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->dispose = tangle_button_group_dispose;
	gobject_class->set_property = tangle_button_group_set_property;
	gobject_class->get_property = tangle_button_group_get_property;

	/**
	 * TangleButtonGroup:button:
	 *
	 * Adds a TangleButton to the TangleButtonGroup.
	 */
	g_object_class_install_property(gobject_class, PROP_BUTTON,
	                                g_param_spec_object("button",
	                                "Button",
	                                "Add a TangleButton to the TangleButtonGroup",
	                                TANGLE_TYPE_BUTTON,
	                                G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleButtonGroup:selected-button:
	 *
	 * Returns the selected button in the group.
	 */
	g_object_class_install_property(gobject_class, PROP_SELECTED_BUTTON,
	                                g_param_spec_object("selected-button",
	                                "Selected button",
	                                "Returns the selected button in the group",
	                                TANGLE_TYPE_BUTTON,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * TangleButtonGroup:memory-managed:
	 *
	 * Whether the TangleButtonGroup manages its references itself.
	 */
	g_object_class_install_property(gobject_class, PROP_MEMORY_MANAGED,
	                                g_param_spec_boolean("memory-managed",
	                                "Memory managed",
	                                "Whether the TangleButtonGroup manages its references itself",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleButtonGroupPrivate));
}

static void tangle_button_group_init(TangleButtonGroup* button_group) {
	button_group->priv = G_TYPE_INSTANCE_GET_PRIVATE(button_group, TANGLE_TYPE_BUTTON_GROUP, TangleButtonGroupPrivate);
}

static TangleButtonGroup* new_valist(gboolean memory_managed, TangleButton* button, va_list button_list) {
	TangleButtonGroup* button_group;
	GArray* array;
	gint i;
	
	array = g_array_new(FALSE, FALSE, sizeof(GParameter));
	
	if (memory_managed) {
		GParameter parameter = { 0 };
		parameter.name = "memory-managed";
		g_value_init(&parameter.value, G_TYPE_BOOLEAN);
		g_value_set_boolean(&parameter.value, TRUE);
		g_array_append_val(array, parameter);
	}
	
	while (button) {
		GParameter parameter = { 0 };
		parameter.name = "button";
		g_value_init(&parameter.value, TANGLE_TYPE_BUTTON);
		g_value_set_object(&parameter.value, button);
		g_array_append_val(array, parameter);

		button = va_arg(button_list, TangleButton*);
	}

	button_group = g_object_newv(TANGLE_TYPE_BUTTON_GROUP, array->len, (GParameter*)array->data);
	
	for (i = 0; i < array->len; i++) {
		g_value_unset(&g_array_index(array, GParameter, i).value);
	}
	g_array_free(array, TRUE);
	
	return button_group;	
}

static void on_notify_selected(GObject* object, GParamSpec* param_spec, gpointer user_data) {
	TangleButtonGroup* button_group;
	TangleButton* button;
	TangleButton* previous_selected_button;
	gboolean selected;
	
	button_group = TANGLE_BUTTON_GROUP(user_data);
	button = TANGLE_BUTTON(object);
	
	selected = tangle_button_get_selected(button);
	if (!selected && button_group->priv->selected_button == button) {
		button_group->priv->selected_button = NULL;
		g_object_notify(G_OBJECT(button_group), "selected-button");
	} else if (selected && button_group->priv->selected_button != button) {
		previous_selected_button = button_group->priv->selected_button;
		button_group->priv->selected_button = button;
		if (previous_selected_button) {
			tangle_button_set_selected(previous_selected_button, FALSE);
		}
		g_object_notify(G_OBJECT(button_group), "selected-button");	
	}
}
