/*
 * tangle-box-layout.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-box-layout.h"
#include "tangle-actor.h"
#include "tangle-widget.h"

/**
 * SECTION:tangle-box-layout
 * @Short_description: 	A layout manager that arranges children to a single horizontal or vertical line
 * @Title: TangleBoxLayout
 */

G_DEFINE_TYPE(TangleBoxLayout, tangle_box_layout, TANGLE_TYPE_LAYOUT);

struct _TangleBoxLayoutPrivate {
	gfloat interactive_width;
	gfloat min_width;
	gfloat natural_width;
	gfloat shrinkable_width;
	gfloat expandable_width;
	gfloat for_height;
	guint shrinkable_count;
	guint expandable_count;
	guint overflow_top_left : 1;
	guint overflow_bottom_right : 1;
};

G_DEFINE_TYPE(TangleBoxLayoutMeta, tangle_box_layout_meta, TANGLE_TYPE_LAYOUT_META);

struct _TangleBoxLayoutMetaPrivate {
	gfloat requested_width;
	gfloat allocated_width;
	guint interacting : 1;
	guint shrinkable : 1;
	guint expandable : 1;
};

/* TangleBoxLayout */


static void calculate_preferred_width(TangleBoxLayout* box_layout, TangleWidget* widget, gfloat for_height) {
	GList* child_in_list;

	if (for_height < 0 || for_height != box_layout->priv->for_height) {
		box_layout->priv->for_height = for_height;
		box_layout->priv->interactive_width = box_layout->priv->min_width = box_layout->priv->natural_width = box_layout->priv->shrinkable_width = box_layout->priv->expandable_width = 0;
		box_layout->priv->expandable_count = 0;
		for (child_in_list = tangle_widget_get_children_readonly(widget); child_in_list; child_in_list = child_in_list->next) {
			ClutterActor* actor;

			actor = CLUTTER_ACTOR(child_in_list->data);
			if (CLUTTER_ACTOR_IS_VISIBLE(actor)) {
				TangleBoxLayoutMeta* box_layout_meta;
				gfloat min_width;
				gfloat natural_width;

				box_layout_meta = TANGLE_BOX_LAYOUT_META(tangle_layout_get_meta(TANGLE_LAYOUT(box_layout), actor));
				if (TANGLE_IS_ACTOR(actor) && tangle_actor_get_interacting(TANGLE_ACTOR(actor))) {
					box_layout_meta->priv->interacting = 1;
					tangle_layout_get_preferred_actor_width(TANGLE_LAYOUT(box_layout), actor, for_height, NULL, &natural_width);
					box_layout->priv->interactive_width += natural_width;
					box_layout->priv->min_width += natural_width;
					box_layout->priv->natural_width += natural_width;
				} else {
					box_layout_meta->priv->interacting = 0;
					tangle_layout_get_preferred_actor_width(TANGLE_LAYOUT(box_layout), actor, for_height, &min_width, &natural_width);
					box_layout->priv->min_width += min_width;
					box_layout->priv->natural_width += natural_width;
					if (min_width < natural_width) {
						box_layout_meta->priv->shrinkable = 1;
						box_layout->priv->shrinkable_width += natural_width;
						box_layout->priv->shrinkable_count++;
					} else {
						box_layout_meta->priv->shrinkable = 0;
					}
				}

				box_layout_meta->priv->requested_width = natural_width;
				if (box_layout_meta->priv->expandable) {
					box_layout->priv->expandable_width += natural_width;
					box_layout->priv->expandable_count++;
				}
			}
		}
	}
}

static void allocate_width(TangleBoxLayout* box_layout, TangleWidget* widget, gfloat width) {
	GList* child_in_list;
	gfloat interacting_factor;
	gfloat shrinkable_factor;
	gfloat fixed_factor;
	gfloat expandable_factor;

	if (width < box_layout->priv->interactive_width) {
		interacting_factor = width / box_layout->priv->interactive_width;
		shrinkable_factor = 0;
		fixed_factor = 0;
		expandable_factor = 0;
	} else if (width < box_layout->priv->min_width) {
		interacting_factor = 1;
		shrinkable_factor = fixed_factor =
		                    (width - box_layout->priv->interactive_width) /
		                    (box_layout->priv->natural_width - box_layout->priv->interactive_width);
		expandable_factor = 1;
	} else if (width < box_layout->priv->natural_width) {
		if (box_layout->priv->shrinkable_count > 0) {
			interacting_factor = 1;
			shrinkable_factor = (width - (box_layout->priv->natural_width - box_layout->priv->shrinkable_width)) /
		                	    box_layout->priv->shrinkable_width;
			fixed_factor = 1;
			expandable_factor = 1;
		} else {
			interacting_factor = 1;
			shrinkable_factor = fixed_factor =
			                    (width - box_layout->priv->interactive_width) /
			                    (box_layout->priv->natural_width - box_layout->priv->interactive_width);
			expandable_factor = 1;
		}
	} else if (width > box_layout->priv->natural_width) {
		if (box_layout->priv->expandable_count > 0) {
			interacting_factor = 1;
			shrinkable_factor = 1;
			fixed_factor = 1;
			expandable_factor = (width - (box_layout->priv->natural_width - box_layout->priv->expandable_width)) /
		                	    box_layout->priv->expandable_width;
		} else {
			shrinkable_factor = interacting_factor = fixed_factor = expandable_factor =
			                     width / box_layout->priv->natural_width;
		}	
	} else {
		interacting_factor = 1;
		shrinkable_factor = 1;
		fixed_factor = 1;
		expandable_factor = 1;
	}

	for (child_in_list = tangle_widget_get_children_readonly(widget); child_in_list; child_in_list = child_in_list->next)
	{
		ClutterActor* actor;
		TangleBoxLayoutMeta* box_layout_meta;

		actor = child_in_list->data;
		box_layout_meta = TANGLE_BOX_LAYOUT_META(tangle_layout_get_meta(TANGLE_LAYOUT(box_layout), actor));
		if (box_layout_meta->priv->interacting) {
			box_layout_meta->priv->allocated_width = box_layout_meta->priv->requested_width * interacting_factor;
		} else if (!box_layout_meta->priv->shrinkable && !box_layout_meta->priv->expandable) {
			box_layout_meta->priv->allocated_width = box_layout_meta->priv->requested_width * fixed_factor;
		} else {
			if (box_layout_meta->priv->shrinkable) {
				box_layout_meta->priv->allocated_width = box_layout_meta->priv->requested_width * shrinkable_factor;
			}
			if (box_layout_meta->priv->expandable) {
				box_layout_meta->priv->allocated_width = box_layout_meta->priv->requested_width * expandable_factor;			
			}
		}
	}
}

static void get_preferred_height(TangleBoxLayout* box_layout, TangleWidget* widget, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	GList* child_in_list;
	gfloat min_height;
	gfloat natural_height;

	if (for_width > 0) {
		calculate_preferred_width(box_layout, widget, -1);
		allocate_width(box_layout, widget, for_width);
	}
	
	min_height = natural_height = 0;
	for (child_in_list = tangle_widget_get_children_readonly(widget); child_in_list; child_in_list = child_in_list->next) {
		ClutterActor* actor;
		
		actor = child_in_list->data;
		if (CLUTTER_ACTOR_IS_VISIBLE(actor)) {
			TangleBoxLayoutMeta* box_layout_meta;
			gfloat actor_for_width;
			gfloat actor_min_height;
			gfloat actor_natural_height;

			if (for_width > 0) {
				box_layout_meta = TANGLE_BOX_LAYOUT_META(tangle_layout_get_meta(TANGLE_LAYOUT(box_layout), actor));
				actor_for_width = box_layout_meta->priv->allocated_width;
			} else {
				actor_for_width = for_width;
			}
			tangle_layout_get_preferred_interactive_height(TANGLE_LAYOUT(box_layout), actor, actor_for_width, FALSE, &actor_min_height, &actor_natural_height, NULL);
			min_height = MAX(min_height, actor_min_height);
			natural_height = MAX(natural_height, actor_natural_height);
		}
	}
	if (min_height_p) {
		*min_height_p = min_height;
	}
	if (natural_height_p) {
		*natural_height_p = natural_height;
	}
}

static void tangle_box_layout_get_preferred_width(TangleLayout* layout, TangleWidget* widget, gfloat for_height, gfloat* min_width_p, gfloat* natural_width_p) {
	TangleBoxLayout* box_layout;

	box_layout = TANGLE_BOX_LAYOUT(layout);

	calculate_preferred_width(box_layout, widget, for_height);
	if (min_width_p) {
		*min_width_p = box_layout->priv->min_width;
	}
	if (natural_width_p) {
		*natural_width_p = box_layout->priv->natural_width;
	}
}

static void tangle_box_layout_get_preferred_height(TangleLayout* layout, TangleWidget* widget, gfloat for_width, gfloat* min_height_p, gfloat* natural_height_p) {
	TangleBoxLayout* box_layout;

	box_layout = TANGLE_BOX_LAYOUT(layout);

	get_preferred_height(box_layout, widget, for_width, min_height_p, natural_height_p);
}

static void tangle_box_layout_allocate(TangleLayout* layout, TangleWidget* widget, gfloat width, gfloat height, ClutterAllocationFlags flags) {
	TangleBoxLayout* box_layout;
	GList* child_in_list;
	gfloat x;
	
	box_layout = TANGLE_BOX_LAYOUT(layout);

	calculate_preferred_width(box_layout, widget, height);
	allocate_width(box_layout, widget, width);
	
	x = 0;
	for (child_in_list = tangle_widget_get_children_readonly(widget); child_in_list; child_in_list = child_in_list->next) {
		ClutterActor* actor;
		TangleBoxLayoutMeta* box_layout_meta;
		gfloat y;
		gfloat actor_width;
		gfloat actor_height;
		ClutterActorBox actor_box;
		
		actor = CLUTTER_ACTOR(child_in_list->data);
		box_layout_meta = TANGLE_BOX_LAYOUT_META(tangle_layout_get_meta(TANGLE_LAYOUT(box_layout), actor));

		y = 0;
		actor_width = box_layout_meta->priv->allocated_width;
		tangle_layout_get_preferred_actor_height(layout, actor, actor_width, NULL, &actor_height);
		if (actor_height > height) {
			if (TANGLE_IS_ACTOR(actor) && tangle_actor_get_interacting(TANGLE_ACTOR(actor))) {
				if (box_layout->priv->overflow_top_left && box_layout->priv->overflow_bottom_right) {
					y = (height - actor_height) / 2;
				} else if (box_layout->priv->overflow_top_left) {
					y = height - actor_height;
				} else if (!box_layout->priv->overflow_bottom_right) {
					actor_height = height;
				}
			} else {
				actor_height = height;
			}
		} else if (actor_height < height) {
			actor_height = height;
		}

		actor_box.x1 = x;
		actor_box.x2 = x + actor_width;
		actor_box.y1 = y;
		actor_box.y2 = y + actor_height;
		tangle_layout_allocate_actor(layout, actor, &actor_box, flags);

		x += actor_width;
	}
	
	/* Invalidate length requests (there is no queue-relayout signal in ClutterActor). */
	box_layout->priv->for_height = -1;
}

static TangleLayoutMeta* tangle_box_layout_create_layout_meta(TangleLayout* layout, ClutterActor* actor) {

	return tangle_box_layout_meta_new(layout, actor);
}

static void tangle_box_layout_finalize(GObject* object) {
	G_OBJECT_CLASS (tangle_box_layout_parent_class)->finalize(object);
}

static void tangle_box_layout_dispose(GObject* object) {
	G_OBJECT_CLASS (tangle_box_layout_parent_class)->dispose(object);
}

static void tangle_box_layout_class_init(TangleBoxLayoutClass* box_layout_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(box_layout_class);
	TangleLayoutClass* layout_class = TANGLE_LAYOUT_CLASS(box_layout_class);

	gobject_class->finalize = tangle_box_layout_finalize;
	gobject_class->dispose = tangle_box_layout_dispose;

	layout_class->get_preferred_width = tangle_box_layout_get_preferred_width;
	layout_class->get_preferred_height = tangle_box_layout_get_preferred_height;
	layout_class->allocate = tangle_box_layout_allocate;
	layout_class->create_layout_meta = tangle_box_layout_create_layout_meta;

	g_type_class_add_private (gobject_class, sizeof (TangleBoxLayoutPrivate));
}

static void tangle_box_layout_init(TangleBoxLayout* box_layout) {
	box_layout->priv = G_TYPE_INSTANCE_GET_PRIVATE(box_layout, TANGLE_TYPE_BOX_LAYOUT, TangleBoxLayoutPrivate);
	box_layout->priv->overflow_bottom_right = 1;
}

TangleLayout* tangle_box_layout_new(void) {

	return TANGLE_LAYOUT(g_object_new(TANGLE_TYPE_BOX_LAYOUT, NULL));
}


/* TangleBoxMeta */

static void tangle_box_layout_meta_class_init(TangleBoxLayoutMetaClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

	g_type_class_add_private(gobject_class, sizeof(TangleBoxLayoutMetaPrivate));
}

static void tangle_box_layout_meta_init(TangleBoxLayoutMeta* box_layout_meta) {
	box_layout_meta->priv = G_TYPE_INSTANCE_GET_PRIVATE(box_layout_meta, TANGLE_TYPE_BOX_LAYOUT_META, TangleBoxLayoutMetaPrivate);	
}

TangleLayoutMeta* tangle_box_layout_meta_new(TangleLayout* layout, ClutterActor* actor) {

	return TANGLE_LAYOUT_META(g_object_new(TANGLE_TYPE_BOX_LAYOUT_META, "layout", layout, "actor", actor, NULL));
}
