
/*
 * tangle-misc.h
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2009-2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-misc.h"
#include "tangle-stylesheet.h"
#include <gobject/gvaluecollector.h>
#include <string.h>
#include <clutter/clutter.h>
#include <gmodule.h>

/**
 * SECTION:tangle-misc
 * @Short_description: Miscellaneous utility function
 * @Title: Miscellaneous utilities
 */

gboolean tangle_init(int* argc, char*** argv) {
	gboolean success = FALSE;
	TangleStylesheet* stylesheet;
	
	if (clutter_init(argc, argv) == CLUTTER_INIT_SUCCESS) {
		success = TRUE;
	}
	
	return success;
}

void tangle_unref_and_nullify_object_pointer_impl(GObject** object_pointer) {
	if (*object_pointer) {
		g_object_unref(*object_pointer);
		*object_pointer = NULL;
	}
}

TanglePointers* tangle_pointers_new(gint n, ...) {
	TanglePointers* pointers;
	va_list args;
	int i;
	gpointer p;

	pointers = (TanglePointers*)g_slice_alloc(n * sizeof(gpointer));

	va_start(args, n);
	for (i = 0; i < n; i++) {
		p = va_arg(args, gpointer);
		pointers[i] = p;		
	}
	va_end(args);

	return pointers;
}

void tangle_pointers_get(TanglePointers* pointers, gint n, ...) {
	va_list args;
	int i;
	gpointer* p_return;

	va_start(args, n);
	for (i = 0; i < n; i++) {
		p_return = va_arg(args, gpointer*);
		*p_return = pointers[i];		
	}
	va_end(args);
}

void tangle_pointers_free(TanglePointers* pointers, gint n) {
	g_slice_free1(n * sizeof(gpointer), pointers);
}

static GValue* find_property_value(guint n_properties, GObjectConstructParam* properties, GParamSpec* param_spec) {
	GValue* value = NULL;
	
	while (n_properties--) {
		if (properties->pspec == param_spec) {
			value = properties->value;
			break;
		}
		properties++;
	}
	
	return value;
}

gboolean tangle_lookup_construct_properties(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties, const gchar* first_property_name, ...) {
	gboolean success = TRUE;
	GObjectClass* object_class;
	va_list args;
	const gchar* name;
	GParamSpec* param_spec;
	GValue* value;
	gchar* error;
	
	object_class = G_OBJECT_CLASS(g_type_class_ref(type));
	g_return_val_if_fail(object_class != NULL, FALSE);
	
	va_start(args, first_property_name);
	for (name = first_property_name; name; name = va_arg(args, const gchar*)) {
		param_spec = g_object_class_find_property(object_class, name);
		if (!param_spec) {
			g_warning("%s: No such property: %s\n", G_STRLOC, name);
			success = FALSE;
			break;
		}
		if (!(value = find_property_value(n_construct_properties, construct_properties, param_spec))) {
			success = FALSE;
		} else {
			error = NULL;
			G_VALUE_LCOPY(value, args, 0, &error);
			if (error) {
				g_warning("%s: %s", G_STRLOC, error);
				g_free(error);
				success = FALSE;
			}
		}
	}
	
	g_type_class_unref(object_class);

	return success;
}

GObject* tangle_construct_with_extra_properties(GObject* (*constructor)(GType, guint, GObjectConstructParam*), GType type, guint n_construct_properties, GObjectConstructParam* construct_properties, const gchar* first_extra_property_name, ...) {
	GObject* object = NULL;
	GList* extra_construct_properties = NULL;
	GObjectClass* object_class;
	va_list args;
	const gchar* name;
	GParamSpec* param_spec;
	GValue* value;
	gboolean existing_value;
	gchar* error;
	GObjectConstructParam* construct_param;
	guint n_all_construct_properties;
	GObjectConstructParam* all_construct_properties;
	GList* list;
	
	object_class = G_OBJECT_CLASS(g_type_class_ref(type));
	g_return_val_if_fail(object_class != NULL, NULL);
		
	va_start(args, first_extra_property_name);
	for (name = first_extra_property_name; name; name = va_arg(args, const gchar*)) {
		param_spec = g_object_class_find_property(object_class, name);
		if (!param_spec) {
			g_warning("%s: No such property: %s\n", G_STRLOC, name);
			break;
		}
		if (value = find_property_value(n_construct_properties, construct_properties, param_spec)) {
			existing_value = TRUE;
			g_value_unset(value);
		} else {
			existing_value = FALSE;
			value = g_slice_new0(GValue);
		}
		g_value_init(value, G_PARAM_SPEC_VALUE_TYPE(param_spec));			
		error = NULL;
		G_VALUE_COLLECT(value, args, 0, &error);
		if (error) {
			g_warning("%s: %s", G_STRLOC, error);
			g_free(error);
		} else if (!existing_value){
			construct_param = g_slice_new(GObjectConstructParam);
			construct_param->pspec = param_spec;
			construct_param->value = value;
			extra_construct_properties = g_list_prepend(extra_construct_properties, construct_param);
		}
	}
	va_end(args);
	
	n_all_construct_properties = n_construct_properties + g_list_length(extra_construct_properties);
	all_construct_properties = g_slice_alloc(n_all_construct_properties * sizeof(GObjectConstructParam));
	memcpy(all_construct_properties, construct_properties, n_construct_properties * sizeof(GObjectConstructParam));
	construct_param = all_construct_properties + n_construct_properties;
	for (list = extra_construct_properties; list; list = list->next) {
		*construct_param = *((GObjectConstructParam*)list->data);
		construct_param++;
	}
	object = constructor(type, n_all_construct_properties, all_construct_properties);

	g_slice_free1(n_all_construct_properties * sizeof(GObjectConstructParam), all_construct_properties);
	for (list = extra_construct_properties; list; list = list->next) {
		g_value_unset(((GObjectConstructParam*)list->data)->value);
		g_slice_free(GValue, ((GObjectConstructParam*)list->data)->value);
		g_slice_free(GObjectConstructParam, list->data);
	}	
	g_list_free(extra_construct_properties);
	g_type_class_unref(object_class);

	return object;
}

gboolean tangle_signal_accumulator_non_null_handled(GSignalInvocationHint* ihint, GValue* return_accu, const GValue* handler_return, gpointer user_data) {
	GObject* object;

	object = g_value_get_object(handler_return);
	g_value_set_object(return_accu, object);

	return !object;
}

gboolean tangle_signal_connect_dynamically(GObject* object, const gchar* signal_name, const gchar* function_name, gpointer user_data) {
	gboolean retvalue = FALSE;
	GModule* module = NULL;
	gpointer function_pointer;
	
	if (!g_signal_lookup(signal_name, G_TYPE_FROM_INSTANCE(object))) {
		g_warning("No such signal: '%s'.", signal_name);
	} else if (!g_module_supported() || !(module = g_module_open(NULL, 0))) {
		g_warning("The current platform does not support modules.");
	} else if (!g_module_symbol(module, function_name, (gpointer*)&function_pointer)) {
		g_warning("No such function: '%s'", function_name);
	} else if (!g_signal_connect(object, signal_name, G_CALLBACK(function_pointer), user_data)) {
		g_warning("Failed to connect a handler function '%s' for signal '%s'.", function_name, signal_name);
	} else {
		retvalue = TRUE;
	}
	if (module) {
		g_module_close(module);
	}
	
	return retvalue;
}
