/*
 * tangle-call-function-action.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-call-function-action.h"
#include <gmodule.h>

/**
 * SECTION:tangle-call-function-action
 * @Short_description: An action that calls a function
 * @Title: TangleCallFunctionAction
 */

G_DEFINE_TYPE(TangleCallFunctionAction, tangle_call_function_action, TANGLE_TYPE_ACTION);

enum {
	PROP_0,
	PROP_FUNCTION
};

struct _TangleCallFunctionActionPrivate {
	gchar* function_name;
	TangleActionFunction function_pointer;
	GModule* module;
};

TangleCallFunctionAction* tangle_call_function_action_new() {

	return TANGLE_CALL_FUNCTION_ACTION(g_object_new(TANGLE_TYPE_CALL_FUNCTION_ACTION, NULL));
}

const gchar* tangle_call_function_action_get_function_name(TangleCallFunctionAction* action) {

	return action->priv->function_name;
}

void tangle_call_function_action_set_function_name(TangleCallFunctionAction* action, const gchar* function_name) {
	g_free(action->priv->function_name);
	action->priv->function_name = g_strdup(function_name);
	g_module_symbol(action->priv->module, function_name, (gpointer*)&action->priv->function_pointer);
	g_object_notify(G_OBJECT(action), "function");
}

TangleActionFunction tangle_call_function_action_get_function_pointer(TangleCallFunctionAction* action) {

	return action->priv->function_pointer;
}

void tangle_call_function_action_set_function_pointer(TangleCallFunctionAction* action, TangleActionFunction function_pointer) {
	action->priv->function_pointer = function_pointer;
	g_free(action->priv->function_name);
	action->priv->function_name = NULL;
	g_object_notify(G_OBJECT(action), "function");
}

static void tangle_call_function_action_execute(TangleAction* tangle_action, GObject* source, const gchar* trigger, TangleProperties* properties) {
	TangleCallFunctionAction* action;
	
	action = TANGLE_CALL_FUNCTION_ACTION(tangle_action);
	g_return_if_fail(action->priv->function_pointer != NULL);
	
	action->priv->function_pointer(tangle_action, source, trigger, properties);
}

static void tangle_call_function_action_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleCallFunctionAction* action;
	
	action = TANGLE_CALL_FUNCTION_ACTION(object);

	switch (prop_id) {
		case PROP_FUNCTION:
			tangle_call_function_action_set_function_name(action, g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_call_function_action_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleCallFunctionAction* action;

	action = TANGLE_CALL_FUNCTION_ACTION(object);

        switch (prop_id) {
		case PROP_FUNCTION:
			g_value_set_string(value, action->priv->function_name);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_call_function_action_finalize(GObject* object) {
	G_OBJECT_CLASS(tangle_call_function_action_parent_class)->finalize(object);
}

static void tangle_call_function_action_dispose(GObject* object) {
	G_OBJECT_CLASS(tangle_call_function_action_parent_class)->dispose(object);
}

static void tangle_call_function_action_class_init(TangleCallFunctionActionClass* klass) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(klass);

	gobject_class->finalize = tangle_call_function_action_finalize;
	gobject_class->dispose = tangle_call_function_action_dispose;
	gobject_class->set_property = tangle_call_function_action_set_property;
	gobject_class->get_property = tangle_call_function_action_get_property;

	/**
	 * TangleCallFunctionAction:function:
	 */
	g_object_class_install_property(gobject_class, PROP_FUNCTION,
	                                g_param_spec_string("function",
	                                                    "Function",
	                                                    "The name of the function that is called when the action is executed",
	                                                    NULL,
	                                                    G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleCallFunctionActionPrivate));
}

static void tangle_call_function_action_init(TangleCallFunctionAction* action) {
	action->priv = G_TYPE_INSTANCE_GET_PRIVATE(action, TANGLE_TYPE_CALL_FUNCTION_ACTION, TangleCallFunctionActionPrivate);
	action->priv->module = g_module_open(NULL, 0);
	g_assert(action->priv->module != NULL);
}

