/*
 * tangle-template.c
 *
 * This file is part of Tangle Toolkit - A graphical widget library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */
#include "tangle-template.h"

/**
 * SECTION:tangle-template
 * @Short_description: A template object to create other objects
 * @Title: TangleTemplate
 */

G_DEFINE_TYPE(TangleTemplate, tangle_template, TANGLE_TYPE_PROPERTIES);

enum {
	PROP_0,
	PROP_OBJECT_TYPE
};

struct _TangleTemplatePrivate {
	GType object_type;
};

TangleTemplate* tangle_template_new(GType object_type) {

	return TANGLE_TEMPLATE(g_object_new(TANGLE_TYPE_TEMPLATE, "object-type", object_type, NULL));
}

GType tangle_template_get_object_type(TangleTemplate* template) {

	return template->priv->object_type;
}

GObject* tangle_template_create_object(TangleTemplate* template) {
	GObject* object;
	guint n;
	GParameter* parameters;
	guint i;
	TangleTemplate* t;
	
	parameters = tangle_properties_get_as_parameters(TANGLE_PROPERTIES(template), &n);

	/* Create objects from possible templates recursively. */
	for (i = 0; i < n; i++) {
		if (G_VALUE_HOLDS(&parameters[i].value, TANGLE_TYPE_TEMPLATE)) {
			t = TANGLE_TEMPLATE(g_value_get_object(&parameters[i].value));
			g_object_ref(t);
			object = tangle_template_create_object(t);
			g_value_unset(&parameters[i].value);
			g_value_init(&parameters[i].value, tangle_template_get_object_type(t));
			g_value_set_instance(&parameters[i].value, object);
			g_object_unref(t);
		}
	}

	object = g_object_newv(template->priv->object_type, n, parameters);

	for (i = 0; i < n; i++) {
		g_free((gchar*)parameters[i].name);
		g_value_unset(&parameters[i].value);
	}
	g_free(parameters);
	
	return object;
}

static gboolean tangle_template_validate_property(TangleProperties* properties, const gchar* name, const GValue* value) {
	TangleTemplate* template;
	
	template = TANGLE_TEMPLATE(properties);
	
	/* TODO */
	
	return TRUE;
}

static void tangle_template_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleTemplate* template;
	
	template = TANGLE_TEMPLATE(object);

	switch (prop_id) {
		case PROP_OBJECT_TYPE:
			template->priv->object_type = g_value_get_gtype(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_template_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleTemplate* template;

	template = TANGLE_TEMPLATE(object);

        switch (prop_id) {
		case PROP_OBJECT_TYPE:
			g_value_set_gtype(value, template->priv->object_type);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_template_class_init(TangleTemplateClass* template_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(template_class);
	TanglePropertiesClass* properties_class = TANGLE_PROPERTIES_CLASS(template_class);

	gobject_class->set_property = tangle_template_set_property;
	gobject_class->get_property = tangle_template_get_property;
	
	properties_class->validate_property = tangle_template_validate_property;

	/**
	 * TangleTemplate:object-type:
	 */
	g_object_class_install_property(gobject_class, PROP_OBJECT_TYPE,
	                                g_param_spec_gtype("object-type",
	                                "Object type",
	                                "The GType of the object to be created",
	                                G_TYPE_OBJECT,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (TangleTemplatePrivate));
}

static void tangle_template_init(TangleTemplate* template) {
	template->priv = G_TYPE_INSTANCE_GET_PRIVATE(template, TANGLE_TYPE_TEMPLATE, TangleTemplatePrivate);
}
