/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <unistd.h>
#include <cstdio>
#include <sstream>
#include <sys/types.h>
#include <PeerHood.h>
#include <netinet/in.h>
#include <stdlib.h>

using namespace std;

// callback interface plugin
class ClientCallback : public CBasicCallback
{
 public:
  void Notify(TEventType aEvent, const string& aAddress);
  void NewConnection(const int aPort, MAbstractConnection* aConnection, int aConnectionId);  
};


void ClientCallback::Notify(TEventType aEvent, const string& aAddress)
{
  switch (aEvent) {
    case PH_DEVICE_LOST:
      cout << "The device " << aAddress << " is lost!" << endl;
      break;

    case PH_DEVICE_FOUND:
      cout << "The device " << aAddress << " is newly founded!" << endl;
      break;

  case PH_WEAK_LINK:
    cout << "Should change connection type from " << aAddress << endl;
    break;

    default:
      cerr << "Unknown event type!" << endl;
  }
}


void ClientCallback::NewConnection(const int aPort, MAbstractConnection* aConnection, int aConnectionId)
{
  cerr << "Someone tried to connect, target port is \"" << aPort << "\"" << endl;
  delete aConnection;
}


void menu() 
{
  cout << endl << "Make your guess:" << endl;
  cout << "  i - init" << endl;
  cout << "  n - list all nearby devices" << endl;
  cout << "  a - add devices to device database" << endl;
  cout << "  r - register a new service" << endl;
  cout << "  s - list services on all devices" << endl;
  cout << "  u - unregister a service" << endl;
  cout << "  c - chat (direct data connection)" << endl;
  cout << "  l - list locally registered services" << endl;
  cout << "  m - monitor a device" << endl;
  cout << "  k - signal monitoring" << endl;
  cout << "  p - connect to a local service" << endl;
  cout << "  e - change connection on the fly" << endl;
  cout << "  z - number crunching" << endl;
  cout << "  q - quit" << endl;
}


void ListNearbyDevices(MPeerHood* aPeerHood)
{
  TDeviceList* list;

  cout << "Following devices have been found:" << endl;
  
  list = aPeerHood->GetDeviceListL();
  
  for (TDeviceIterator i = list->Begin();i != list->End();++i) {
    cout << "  " << (*i)->GetAddress() << " proto: " << (*i)->GetPrototype();
    if ((*i)->HasPeerHood()) {
      cout << " PeerHood: yes" << endl;
    }
    else {
      cout << " PeerHood: no" << endl;
    }
  }
  
  delete list;
}

void AddDevice(MPeerHood* aPeerHood)
{
  string Address;

  cout << "Give the new device address" << endl;
  cout << "> ";
  cin >> Address;
 
  // Format address into correct form .. do some format check

  // Add the address into database
  cout << "Sending address " << Address << "to DeviceStorage" << endl;

}

//
// Change of this to allow the use of port JP 6.6.06
//
void RegisterService(MPeerHood* aPeerHood)
{
  string buffer;
  string attributes;
  string port;
  bool success=false;

  cout << "  Service to be inserted: ";
  cin >> ws;
  getline(cin, buffer);
  cout << "  Attribute string: ";
  getline(cin, attributes);
  cout << "  Service port: ";
  getline(cin, port);

  cout << buffer << attributes << port << endl;

  success = aPeerHood->RegisterService(buffer, attributes, port);

  if (success) {
    cout << "Successfull operation" << endl;
  }
  else {
    cout << "Unsuccessfull operation" << endl;
  }
}


void UnregisterService(MPeerHood* aPeerHood)
{
  string buffer;
  string port;
  bool success=false;

  cout << "  Service to be removed: ";
  cout << "  Give the service name: ";
  cin >> buffer;
  cout << "  Give the service port, if 0 then all with the given name are destroyed: ";
  cin >> port;

  success = aPeerHood->UnregisterService(buffer, port);

  if (success) {
    cout << "Successfull operation" << endl;
  }
  else {
    cout << "Unsuccessfull operation" << endl;
  }
}


void ListAllServices(MPeerHood* aPeerHood)
{
  TDeviceList* list;
  TServiceList* services;

  cout << "Following devices & services have been found:" << endl;

  list = aPeerHood->GetDeviceListL();

  for (TDeviceIterator i = list->Begin();i != list->End();++i) {
    cout << "  " << (*i)->GetAddress() << " proto: " << (*i)->GetPrototype();
    if ((*i)->HasPeerHood()) {
      cout << " PeerHood: yes" << endl;
      cout << " device name :" << (*i)->GetName() << endl;
      cout << " checksum :" << (*i)->GetChecksum() << endl;
      services = (*i)->GetServiceListL();
      
      for (TServiceIterator j = services->Begin(); j != services->End();++j) {
	cout << "    service detected: \"" << (*j)->GetName() << "\"" << endl;
	cout << "    attribute string: \"" << (*j)->GetAttributeString() << "\"" << endl;
	cout << "    port: \"" << (*j)->GetPort() << "\"" << endl;
      }
      
      delete services;
    }
    else {
      cout << " PeerHood: no" << endl;
    }
  }
  
  delete list;
}


void ListLocalServices(MPeerHood* aPeerHood)
{
  TServiceList* services;

  cout << "Following local services are registered into daemon:" << endl;
  services = aPeerHood->GetLocalServiceListL();
  
  for (TServiceIterator i = services->Begin();i != services->End();++i) {
    cout << "  service: " << (*i)->GetName() << endl;
    cout << "       attribute string: " << (*i)->GetAttributeString() << endl;
    cout << "       pid: \"" << (*i)->GetPid() << "\"" << endl;
    cout << "       port: \"" << (*i)->GetPort() << "\"" << endl;
  }

  delete services;
}


void Chat(MPeerHood* aPeerHood)
{
  int id;
  int selection;
  TDeviceList* list;
  int length;
  string buffer;
  //  int packets = 0;
  MAbstractConnection* connection;
  //  char *dataBuffer;

  list = aPeerHood->GetDeviceListL();
  id = 1;
	  
  if (list->Size() > 0) {
    cout << "Please select a device" << endl;
    
    for (TDeviceIterator i = list->Begin();i != list->End();++i, ++id) {
      cout << "  #" << id << " " << (*i)->GetAddress() << (*i)->GetPrototype() << endl;
    }
    
    selection = -1;
    cout << "> ";
    cin >> selection;
    
    if (selection > 0 && selection < id) {
      TDeviceIterator i = list->Begin();
      for (int l = 0;l < selection - 1;l++, i++);
      
      connection = aPeerHood->Connect(i, "chat");
    }
    
    else {
       delete list;
       return;
    }
  }
  else {
    cout << "Sorry, there are no nearby devices" << endl;
    delete list;
    return;
  }
  
  delete list;
  
  if (!connection) {
    cout << "Failed to create a connection" << endl;
    return;
  }
  
  while (1) {
    cout << "> ";
    cin >> buffer;
    
    if (buffer.compare("q") == 0) {
      connection->Disconnect();
      delete connection;
      return;
    }
    
    length = buffer.length();
    length = htonl(length);
    connection->Write(&length, sizeof(length));
    connection->Write(buffer.c_str(), buffer.length());
  }
}

void Numbers(MPeerHood* aPeerHood)
{
  int id;
  int selection;
  TDeviceList* list;
  int length;
  string buffer;
  //  int packets = 0;
  MAbstractConnection* connection;
  //  char *dataBuffer;

  list = aPeerHood->GetDeviceListL(&std::string ("numbers"));
  id = 1;
	  
  if (list->Size() > 0) {
    cout << "Please select a device" << endl;
    
    for (TDeviceIterator i = list->Begin();i != list->End();++i, ++id) {
      cout << "  #" << id << " " << (*i)->GetAddress() << (*i)->GetPrototype() << endl;
    }
    
    selection = -1;
    cout << "> ";
    cin >> selection;
    
    if (selection > 0 && selection < id) {
      TDeviceIterator i = list->Begin();
      for (int l = 0;l < selection - 1;l++, i++);
      
      connection = aPeerHood->Connect(i, "numbers");
    }
    
    else {
       delete list;
       return;
    }
  }
  else {
    cout << "Sorry, there are no nearby devices" << endl;
    delete list;
    return;
  }
  
  delete list;
  
  if (!connection) {
    cout << "Failed to create a connection" << endl;
    return;
  }
  
  int num;
  int limit;
//int outer_num;
//int inner;

  cout << "limit: " << endl;

  cin >> limit;

  cout << "Remote address: " << connection->GetRemoteAddress() << endl;
// for (outer_num=0;outer_num<10;outer_num++) {
     for(num=0;num<limit;num++) {
     
       stringstream strs;
       strs << num;
       buffer = strs.str();
       //sprintf((char*)buffer,"%i",num);    

       length = buffer.length();
       length = htonl(length);
       connection->Write(&length, sizeof(length));
       connection->Write(buffer.c_str(), buffer.length());
       usleep(1000);
     }
     //cout << connection->GetRemoteAddress() << endl;
     //} 
 
  connection->Disconnect();
  delete connection;
  return;
}

void MonitorDevice(MPeerHood* aPeerHood)
{
  int id;
  int selection;
  TDeviceList* list;

  list = aPeerHood->GetDeviceListL();
  id = 1;
	  
  if (list->Size() > 0) {
    cout << "Please select a device or give the address wanted" << endl;

    for (TDeviceIterator i = list->Begin();i != list->End();++i, ++id) {
      cout << "  #" << id << " " << (*i)->GetAddress() << endl;
    }
	    
    selection = -1;
    cout << "> ";
    cin >> selection;

    if (selection > 0 && selection < id) {
      cout << "Monitoring, press enter to end..." << endl;
      TDeviceIterator it = (*list)[selection - 1];
      aPeerHood->MonitorDevice(it);
      cin.get();cin.get();
      cout << "Stopping..." << endl;
      aPeerHood->UnmonitorDevice(it);
    }
    else {
      if (selection > 0) {
	// Address given .. check if valid
	cout << "Monitoring the given address" << selection << "press enter to end..." << endl;

	// Implement here monitoring for non given device (add + monitor)


	cout << "Stopping..." << endl;
	// aPeerHood->UnmonitorDevice(it);
      }
      else {
	// Selection not given .. not even address
	delete list;
	return;
      }
    }
  }
  else {
    cout << "Sorry, there are no nearby devices" << endl;
  }
  
  delete list;
}

void SignalMonitorDevice(MPeerHood* aPeerHood)
{
  int id;
  int selection;
  TDeviceList* list;

  list = aPeerHood->GetDeviceListL();
  id = 1;
	  
  if (list->Size() > 0) {
    cout << "Please select a device" << endl;

    for (TDeviceIterator i = list->Begin();i != list->End();++i, ++id) {
      cout << "  #" << id << " " << (*i)->GetAddress() << endl;
    }
	    
    selection = -1;
    cout << "> ";
    cin >> selection;

    if (selection > 0 && selection < id) {
      cout << "Monitoring, press enter to end..." << endl;
      TDeviceIterator it = (*list)[selection - 1];
      aPeerHood->SignalMonitorDevice(it);
      cin.get();cin.get();
      cout << "Stopping..." << endl;
      aPeerHood->SignalUnmonitorDevice();
    }
    else {
      delete list;
      return;
    }
  }
  else {
    cout << "Sorry, there are no nearby devices" << endl;
  }
  
  delete list;
}

void ConnectLocalService(MPeerHood* aPeerHood)
{
  
  int selection = -1;
  TServiceList* services;
  int num;
  string buffer;
  int length;

  services = aPeerHood->GetLocalServiceListL();

  if (services->Size() > 0) {
    cout << "Please select a service:" << endl;
  
    for (int i = 0;i < services->Size();i++) {
      cout << "  #" << i + 1 << " " << (*(*services)[i])->GetName() << endl;
    }
    
    cout << "> ";
    cin >> selection;
    
    if (selection > 0 && selection <= services->Size()) {
      TServiceIterator si = services->Begin();
      for (int l = 0;l < selection - 1;l++, si++);
      
      MAbstractConnection* connection = aPeerHood->Connect(si);
        
      if (connection) {

	for(num=0;num<11;num++) {
	  
	  stringstream strs;
	  strs << num;
	  buffer = strs.str();
	  //sprintf((char*)buffer,"%i",num);    
	  
	  length = buffer.length();
	  length = htonl(length);
	  connection->Write(&length, sizeof(length));
	  connection->Write(buffer.c_str(), buffer.length());
	  usleep(1000);
	}
	cout << "Disconnecting" << endl;
	connection->Disconnect();
	delete connection;
      }
      else {
	cout << " failed!" << endl;
      }
    }
  }
  else {
    cout << "Sorry, there are no local services" << endl;
  }

  delete services;
  
}


int main(int argc, char** argv)
{
  MPeerHood* peerHood;
  ClientCallback* callback = new ClientCallback;
  bool end = false;
  fd_set selectSet;
  char cmd;

  peerHood = MPeerHood::GetInstance(callback);

  if (!peerHood->Init(argc, argv)) {
    cerr << "PeerHood initialization failed!" << endl;
    exit(EXIT_FAILURE);
  }

  while (!end) {
    menu();
    FD_ZERO(&selectSet);
    FD_SET(fileno(stdin), &selectSet);
    select(fileno(stdin) + 1, &selectSet, NULL, NULL, NULL);

    if (FD_ISSET(fileno(stdin), &selectSet)) {
      cin >> cmd;

      switch (cmd) {
        case 'q':
          end = true;
          break;

        case 'i':
	  peerHood->Init(argc, argv);
          break;

        case 'n':
	  ListNearbyDevices(peerHood);
	  break;

        case 'a':
	  AddDevice(peerHood);
	  break;

	case 'r':
	  RegisterService(peerHood);
	  break;

	case 'u':
	  UnregisterService(peerHood);
	  break;

	case 'l':
	  ListLocalServices(peerHood);
	  break;

	case 's':
	  ListAllServices(peerHood);
	  break;

      case 'c':
	  Chat(peerHood);
	  break;	

      case 'z':
	  Numbers(peerHood);
	  break; 
 
      case 'm':
	MonitorDevice(peerHood);
	  break;
	  
      case 'k':
	SignalMonitorDevice(peerHood);
	break;

        case 'p':
	  ConnectLocalService(peerHood);
	  break;
 
      }
    }
    usleep(1);
  }

  delete peerHood;

  return EXIT_SUCCESS;
}
