/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name LocalConnection.cc
 * @memo Local (UNIX) socket implementation of the MAbstractConnection interface.
 *
 * @version 0.2
 * date     23.04.2003
 * change   28.04.2010
 */

#include <cassert>
#include <sstream>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/types.h>
#include <arpa/inet.h>

#include "LocalConnection.h"

using namespace std;

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial variables to their default values
 * and creates a new local socket.
 *
 * @return none
 */
CLocalConnection::CLocalConnection()
{
  iConnected = false;
  iIsListening = false;

  iSocket = socket(PF_UNIX, SOCK_STREAM, 0);
  assert(iSocket != -1);
}

/**
 * @memo Destructor.
 * @doc Destructor, removes the local socket if listening. Similarly, the
 * possibly open connection is closed.
 *
 * @return none
 */
CLocalConnection::~CLocalConnection()
{
  if (iIsListening) unlink(iAddress.c_str());

  if (iConnected) Close();
}


/**
 * @memo Connects to a remote device.
 * @doc Connects to a remote device. The connection will be tried to PSM 21
 * which is PeerHood's default PSM.
 *
 * @param aAddress The destination address.
 * @param aPort The destination port.
 *
 * @return true if the connection could be established
 */
bool CLocalConnection::Connect(const string& aAddress, int aPort)
{
  struct sockaddr_un address;

  if (iConnected) {
    return false;
  }

  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, aAddress.c_str());

  if (connect(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    return false;
  }

  iRemoteAddress = std::string("LOCAL:") + aAddress;

  iConnected = true;

  return true;
}


/**
 * @memo Kills the existing connection.
 * @doc Kills the existing connection. If there's no connection or it could not
 * be closed in a controlled way then the error code is returned. The existing
 * connection will be closed regardless of the return value.
 *
 * @return true if the connection was dropped in a controlled manner
 */
bool CLocalConnection::Disconnect()
{
  if (!iConnected) {
    return false;
  }
  
  close(iSocket);
  iConnected = false;
  
  return true;
}


/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CLocalConnection::Close()
{
  if (iIsListening) unlink(iAddress.c_str());
  close(iSocket);
}


/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. Note that this
 * function is endianess-safe since all data transmission is done locally.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CLocalConnection::Write(const void* aOutBuf, int aLength)
{
  int written = write(iSocket, (char *)aOutBuf, aLength);
  return written;
}


/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CLocalConnection::Read(void* aInBuf, int aLength)
{
  int readData = read(iSocket, (char *)aInBuf, aLength);
  return readData;
}


/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const string& CLocalConnection::GetRemoteAddress()
{  
  return iRemoteAddress;
}


/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CLocalConnection::GetFd()
{
#warning "changed"
  //if (!iConnected && !iIsListening) return -1;

  return iSocket;
}


/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted. This function will always listen the connection type's default
 * port. If some other port is needed then the <code>Listen(int aPort)</code>
 * should be used instead.
 *
 * @return true if successfull
 */
bool CLocalConnection::Listen()
{
  assert("CLocalConnection::Listen() called - this should NEVER happen!");

  return false;
}


/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted.
 *
 * @param aPort The port that should be listened.
 *
 * @return true if successfull
 */
bool CLocalConnection::Listen(int aPort)
{
  ostringstream addr;
  struct sockaddr_un address;

  addr << "/tmp/ph." << aPort;

  assert(aPort != -1);
  unlink(addr.str().c_str());
  
  DBG("CLocalConnection::Listen : listening %s", addr.str().c_str());
  
  iAddress = addr.str();

  if ((iSocket = socket(PF_UNIX, SOCK_STREAM, 0)) == -1) {
    return false;
  }
  
  memset(&address,0,sizeof(struct sockaddr_un));

  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, addr.str().c_str());
  
  if (bind(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    return false;
  }

  if (listen(iSocket, 16) == -1) {
    return false;
  }
  
  DBG(" CLocalConnection::Listen : listening on %s", addr.str().c_str());
  iIsListening = true;

  return true;
}


/**
 * @memo Accepts the incoming connection and returns a new connection object.
 * @doc Accepts the incoming connection and returns a new connection object. If
 * accepting fails then NULL is returned.
 *
 * @return new connection object or NULL in the case of an error
 */
MAbstractConnection* CLocalConnection::AcceptL()
{
  int newSocket;
  int addressLength;
  struct sockaddr_un address;
  CLocalConnection* retval;  
  char tempBuf[INET_ADDRSTRLEN];

  addressLength = sizeof(address);

  if ((newSocket = accept(iSocket, (struct sockaddr *)&address, (socklen_t *)&addressLength)) == -1) {
    return NULL;
  }

  retval = new CLocalConnection;
  if (retval->iSocket < 0)
    {
      ERR("CLocalConnection::AcceptL : socket failed");
      return NULL;
    }
  
  close(retval->iSocket);
  retval->iSocket = newSocket;
  retval->iConnected = true;
  
  retval->iRemoteAddress = std::string("LOCAL:") + std::string(inet_ntop(AF_INET, &address, tempBuf, sizeof(tempBuf)));
  iConnected = true;
  return retval;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CLocalConnection::IsListening()
{
  return iIsListening;
}


/**
 * @memo Tells if the connection is established.
 * @doc Tells if the connection is established or not.
 *
 * @return true if the connection is established
 */
bool CLocalConnection::IsConnected()
{
  return iConnected;
}

/**
 * @memo Tells if the connection has data waiting
 * @doc Checks if the connection has data to read.
 *
 * @return true if the data is ready
 */
bool CLocalConnection::HasData()
{
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
  fd_set set;
  FD_ZERO(&set);
  FD_SET(GetFd(), &set);

  switch (select(GetFd() + 1, &set, NULL, NULL, &timeout)) {
  case -1:
  	ERR("CLocalConnection::HasData(): Select error");
  	break;
    
  case 0:
    break;

  default:
    if (FD_ISSET(GetFd(), &set)) {
      return true;
    }
    else {
      return false;
    }
  }
  return false;

}

unsigned int CLocalConnection::GetDeviceChecksum()
{
	return 0;
}

void CLocalConnection::SetDeviceChecksum(unsigned int aChecksum)
{
	return; // Local connection, no need for checksum
}
