/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name PHList.h
 * @memo PeerHood list template.
 *
 * @version 0.1
 * date     17.04.2003
 * change   17.04.2003
 */

#ifndef __PHLIST_H__
#define __PHLIST_H__

#include "PHIterator.h"

#ifdef __cplusplus

/**
 * @memo PeerHood double-linked list template class.
 * @memo PeerHood double-linked list template class. The functionality of this
 * template is almost the same as the STL versions. The most fundamental 
 * differences are somewhat reduced method set and lower memory consumption.
 * Other considerable differences are that a list item is deleted automatically
 * when it's erased from the list or the entire list is deleted. Also the list
 * type can be only a pointer, not a defined data type like an integer.
 * This template is used internally by the PeerHood system when delivering 
 * neighborhood info via the PeerHood library. Users should not use this class 
 * directly!
 *
 * @see CPHIterator
 * @see CPHListItem
 */
template <class T>
class CPHList
{
 public:
  CPHList();
  ~CPHList();
  void Add(const T& aItem);
  void Erase(CPHIterator<T>& aItem);
  void Clear();
  bool Empty();
  int Size();
  CPHIterator<T> Begin(); 
  CPHIterator<T> End();
  CPHIterator<T> operator[](int position);

 private:
  CPHListItem<T>* iHead;
  int iSize;
};


/**
 * @memo Default constructor.
 * @doc Default constructor. This constructor creates an empty list of the given type.
 *
 * @return none
 */
template <class T>
CPHList<T>::CPHList()
{
  iHead = NULL;
  iSize = 0;
}


/**
 * @memo Destructor, deletes the list and all associated elements.
 * @doc Destructor, deletes the list and all associated elements. Element
 * destruction is the most fundamental difference compared to the STL list
 * template.
 *
 * @return none
 */
template <class T>
CPHList<T>::~CPHList()
{
  CPHListItem<T>* executer;

  while (iHead) {
    executer = iHead;
    iHead = iHead->iNext;
    delete executer;
  }
}


/**
 * @memo Returns an iterator pointing to the first element.
 * @doc Returns an interator pointing to the first element of the list.
 *
 * @return an iterator pointing to the first element of the list or a NULL
 * iterator if the list is empty.
 */
template <class T>
CPHIterator<T> CPHList<T>::Begin()
{
  return CPHIterator<T>(iHead);
}


/**
 * @memo Returns the last (terminating) element of the list.
 * @doc Returns the last (terminating) element of the list. Since the list is
 * NULL terminated the returned value is always NULL.
 *
 * @return NULL iterator
 */
template <class T>
CPHIterator<T> CPHList<T>::End()
{
  return NULL;
}


/**
 * @memo Adds a new element to the list.
 * @doc Adds a new element to the list. The element is always added to the 
 * beginning of the list.
 * 
 * @param aItem The item to be added.
 *
 * @return none
 */
template <class T>
void CPHList<T>::Add(const T& aItem)
{
  CPHListItem<T>* temp = new CPHListItem<T>(aItem);
  temp->iPrev = NULL;
  temp->iNext = iHead;
  if (temp->iNext) temp->iNext->iPrev = temp;
  iHead = temp;  
  iSize++;
}


/**
 * @memo Erases an element.
 * @doc Erases an element pointed by the given iterator. The iterator itself is
 * invalidated so that it points to NULL. Note that the element is also
 * destroyed unlike in the STL list!
 *
 * @param aItem Iterator pointing to the element to be deleted.
 *
 * @return none
 */
template <class T>
void CPHList<T>::Erase(CPHIterator<T>& aItem)
{
  if (!iHead) return;

  if (aItem.iCurrent == iHead) {
    if (aItem.iCurrent->iNext) aItem.iCurrent->iNext->iPrev = NULL;
    iHead = aItem.iCurrent->iNext;
  }

  else {
    aItem.iCurrent->iPrev->iNext = aItem.iCurrent->iNext;
    if (aItem.iCurrent->iNext) aItem.iCurrent->iNext->iPrev = aItem.iCurrent->iPrev;
  }

  delete aItem.iCurrent;
  aItem.iCurrent = NULL;
  iSize--;
}


/**
 * @memo Clears the list.
 * @doc Clears the list. In addition, all elements in the list are destroyed.
 *
 * @return none
 */
template <class T>
void CPHList<T>::Clear()
{
  CPHListItem<T>* executer;

  while (iHead) {
    executer = iHead;
    iHead = iHead->iNext;
    delete executer;
  }

  iSize = 0;
}


/**
 * @memo Tests if the list is empty.
 * @doc Tests if the list is empty.
 *
 * @return true if the list empty or otherwise false
 */
template <class T>
bool CPHList<T>::Empty()
{
  return iHead == NULL;
}


/**
 * @memo Returns the number of elements on the list.
 * @doc Returns the number of elements on the list.
 *
 * @return the number of elements on the list
 */
template <class T>
int CPHList<T>::Size()
{
  return iSize;
}


/**
 * @memo The '[]' operator.
 * @doc The '[]' operator. This operator returns an iterator that points to the
 * element in the given location on the list. The index starts from zero i.e.
 * the first element is in place 0. If the index is greater than list's size
 * then a NULL iterator is returned.
 *
 * @param aPosition The index of the element to be returned.
 *
 * @return iterator pointing to the requested element or NULL in the case of a
 * boundary error
 */
template <class T>
CPHIterator<T> CPHList<T>::operator[](int aPosition)
{
  CPHListItem<T>* temp = iHead;

  if (!iHead) return CPHIterator<T>(iHead);

  while (temp) {
    if (aPosition-- == 0) {
      return CPHIterator<T>(temp);
    }

    temp = temp->iNext;
  }
  
  return CPHIterator<T>(NULL);
}

#else

#endif

#endif
