/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name BaseDevice.cc
 * @memo Basic implementation of the MAbstractDevice. This version includes
 * only minimum implementation of the interface.
 *
 * @version 0.1
 * date     23.04.2003
 * change   23.04.2003
 */

#include "BaseDevice.h"

#warning "Temporary debug"
//temp
#include <syslog.h>

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG( A... )
#endif
//temp

/**
 * @memo Destructor, deletes all associated service objects.
 * @doc Destructor, deletes all associated service objects.
 *
 * @return none
 */
CBaseDevice::~CBaseDevice()
{
  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    delete *i;
  }
}


/**
 * @memo Adds a new service to the device.
 * @doc Adds a new service to the device. No checking to the validity of the
 * given service entry is performed, except the NULL check. If the given 
 * service is NULL then an assertion will take place.
 *
 * @return none
 */
void CBaseDevice::AddService(CService* aService)
{
  assert(aService != NULL);

  //  DBG("AddService: Pushing back name %s", aService->GetName().c_str());
  //  DBG("AddService: Pushing back attr %s", aService->GetAttributeString().c_str());
  //  DBG("AddService: Pushing back port %d", aService->GetPort());
  //  DBG("AddService: Pushing back pid %d", aService->GetPid());

  iServiceList.push_back(aService);
}


/**
 * @memo Returns the address of the device.
 * @doc Returns the address of the device. This address should be the real
 * address used by the device and recognized by the MAbstractConnection that
 * has the same prototype as the device.
 *
 * @return the address of the device
 */
const std::string& CBaseDevice::GetAddress()
{
  return iAddress;
}


/**
 * @memo Returns the name of the device.
 * @doc Returns the name of the device. The name is actually an id that is
 * unique for the device. It can be e.g. device's hardware address etc.
 *
 * @return the name of the device
 */
const std::string& CBaseDevice::GetName()
{
  return iName;
}

/**
 * @memo Tells whether the device has PeerHood enabled or not.
 * @doc Tells whether the device has PeerHood enabled or not. If there's no
 * PeerHood then there are no services either. This function is somewhat
 * unnecessary since devices without PeerHood should be of no interest.
 * Anyway this function can be used to "sniff" the environment.
 *
 * @return true if the remote device has PeerHood enabled
 */
bool CBaseDevice::HasPeerHood()
{
  return iHasPeerHood;
}

/**
 * @memo Returns device's prototype id.
 * @doc Returns device's prototype id. This id is unique for each network
 * technology.
 *
 * @return the prototype id of the device
 */
const std::string& CBaseDevice::GetPrototype()
{
  return iPrototype;
}

/**
 * @memo Used to query if the device has some particular service.
 * @doc Used to query if the device has some particular service present. The
 * cheking is done by iterating through the device's service list. This
 * function executes in linear time.
 *
 * @return true if the device has the given service present
 */
bool CBaseDevice::HasService(const std::string& aServiceName)
{
  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    if ((*i)->GetName().compare(aServiceName) == 0) return true;
  }

  return false;
}

/**
 * @memo Returns all services available on the device.
 * @doc Returns all services available on the device. The returned variable
 * is a list that can be iterated via a <code>TServiceIterator</code> 
 * iterator. Note that caller is responsible of freeing the memory allocated
 * for the returned list.
 *
 * @return list of all available services on the device
 */
TServiceList* CBaseDevice::GetServiceListL()
{
  CService* temp;
  TServiceList* retval = new TServiceList;

  for (std::list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    temp = new CService(*(*i));
    retval->Add(temp);
  }

  return retval;
}


/**
 * @memo Returns the checksum
 * @doc Returns the checksum. This checksum identifies
 * the device if two devices contain same user-defined devicename
 * Checksum is calculated using Daemon's process ID -number. NOTE:
 * this is only temporary solution, try to find unique number!
 *
 * @return the checksum of the device.
 */
unsigned int CBaseDevice::GetChecksum()
{
  return iChecksum;
}

int CBaseDevice::GetServiceListSize()
{
  return iServiceList.size();
}

int CBaseDevice::GetProtoListSize()
{
  return iSupportedPrototypes.size();
}

bool CBaseDevice::CheckService(unsigned short aPort)
{
  bool foundService = false;
  
  for (std::list<CService*>::iterator k = iServiceList.begin();k != iServiceList.end();++k) {
    if ((*k)->GetPort() == aPort) {
      foundService = true;
      break;
    }
  }

  return foundService;
}

bool CBaseDevice::CheckPrototype(char* aProto) 
{
  bool foundProto = false;
  
  for (std::list<char*>::iterator k = iSupportedPrototypes.begin();k != iSupportedPrototypes.end();++k) {
    if (strcmp((*k), aProto) == 0) {
      foundProto = true;
      break;
    }
  }

  return foundProto;

}
