/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name AbstractCreator.h
 * @memo Definition of a pure virtual class used by the connection factory to
 * address the concrete creators.
 *
 * @version 0.1
 * date     02.04.2003
 * change   02.04.2003
 */

#ifndef __ABSTRACTCREATOR_H__
#define __ABSTRACTCREATOR_H__

#include <AbstractConnection.h>
#include <AbstractPinger.h>
#include <AbstractMonitor.h>

/**
 * @memo A pure virtual class that defines an abstract connection creator used
 * by the PeerHood factory. Through this class connections can be created
 * transparently without knowledge of the exact transport medium.
 * @doc This pure virtual class defines a common interface for all real
 * connection creators. This is one of the base classes of the PeerHood factory
 * that in turn enables transparent use of many concrete transport medias. All
 * real connection creators must inherit this interface. Both this class and 
 * also the corresponding connection objects should be compiled into plugin that
 * can be loaded by the PeerHood library. The environment variable PH_PLUGIN_DIR
 * should point to the library's directory. The plugin must be named as
 * <code>XXpeerhood.so</code> so that the PeerHood library can find it.
 *
 * @see Factory
 * @see AbstractConnection
 */
class MAbstractCreator
{
 public:
  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MAbstractCreator() {};

  /** 
   * @memo Used to transparently create a new abstract connection object.
   * @doc Creates a new connection object of the desired type if the given
   * medium prototype is recognized by this creator. This function should
   * not be called directly; use the PeerHood factory instead.
   *
   * @param aProto Prototype id of the transport medium type.
   *
   * @return The created connection or NULL if the medium prototype was not
   * recognized by this creator.
   */
  virtual MAbstractConnection* CreateConnectionL(const std::string& aProto) = 0;


  /** 
   * @memo Used to transparently create a new abstract pinger object.
   * @doc Creates a new pinger object of the desired type if the given
   * medium prototype is recognized by this creator. This function should
   * not be called directly; use the PeerHood factory instead.
   *
   * @param aProto Prototype id of the transport medium type.
   * @param aAddress Remote device's address.
   *
   * @return The created pinger object or NULL if the medium prototype was not
   * recognized by this creator.
   */
  virtual MAbstractPinger* CreatePingerL(const std::string& aProto, const std::string& aAddress) = 0;

  /** 
   * @memo Used to transparently create a new abstract monitor object.
   * @doc Creates a new monitor object of the desired type if the given
   * medium prototype is recognized by this creator. This function should
   * not be called directly; use the PeerHood factory instead.
   *
   * @param aProto Prototype id of the transport medium type.
   * @param aAddress Remote device's address.
   *
   * @return The created monitor object or NULL if the medium prototype was not
   * recognized by this creator.
   */
  virtual MAbstractMonitor* CreateMonitorL(const std::string& aProto, const std::string& aAddress) = 0;

  /** 
   * @memo Returns creator's unique id string.
   * @doc Returns creator's unique id string. Every creator must have a <b>unique
   * </b> id or disasters will follow. Most likely creator's id is the same as 
   * corresponding plugin's id.
   *
   * @return creator's unique id string
   */
  virtual const std::string& GetPrototype() = 0;
};

#endif // __ABSTRACTCREATOR_H__

