/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name PeerHoodImpl.cc
 * @memo Implementation of the CPeerHoodImpl class.
 *
 * @version 0.3
 * date     19.05.2003
 * change   28.04.2010
 */

#include <stdio.h>
#include <cassert>
#include <sstream>
#include <unistd.h>
#include <dlfcn.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <iostream>
#include <stdlib.h>
#include <cstdlib>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "PeerHoodImpl.h"
#include "Logger.h"
#include "Factory.h"
#include "pdu.h"
#include "LibDevice.h"
#include "VirtualConnection.h"

using namespace std;

const char* LOCAL_SOCKET_NAME = "/tmp/phd.local";

CPeerHoodImpl* CPeerHoodImpl::iInstance = NULL;

/**
 * @memo Getter of the <code>CPeerHoodImpl</code> singleton.
 * @doc Getter of the <code>CPeerHoodImpl</code> singleton. This getter ensures
 * that only one instance of the class is created at any given time.
 *
 * @param aCallback Callback interface that is used to receive notifications
 * about different PeerHood events.
 *
 * @return pointer to the currently active instance of the class
 */
MPeerHood* CPeerHoodImpl::GetInstance(CBasicCallback* aCallback)
{
  if (!iInstance) {
    iInstance = new CPeerHoodImpl(aCallback);
  }

  return iInstance;
}


/**
 * @memo Constructor.
 * @doc Default constructor of the class. This constructor is protected and can
 * be called only via the <code>GetInstance</code> method.
 *
 * @param aCallback Callback interface that is used to receive notifications
 * about different PeerHood events. This method gets also iPid which is used
 * for identifying devices with user-specified device name.
 *
 * @return none
 */
CPeerHoodImpl::CPeerHoodImpl(CBasicCallback* aCallback)
{
  iPreferedPlugin = string("PreferedPlugin");
  iCallback = aCallback;
  iConnected = false;
  iCopying = false;
  iRoaming = false;

  if (!LoadParameter(iPreferedPlugin)) {
    ERR("CPeerHoodImpl::CPeerHoodImpl : could not read the parameter \"PreferedPlugin\" from the configuration file, aborting...");
    exit(EXIT_FAILURE);
  }

  if (!LoadParameter("MonitoringInterval", &iMonitoringInterval)) {
    ERR("CPeerHoodImpl::CPeerHoodImpl : could not read the parameter \"MonitoringInterval\" from the configuration file, aborting...");
    exit(EXIT_FAILURE);
  }

  pthread_mutex_init(&iLock, NULL);

  iEngine = CEngine::GetInstance(aCallback);

  iPid = getpid();
  
  /* Dirty fix to small pid problem when running on mobile devices, if the pid is too small
   * user is not allowed to bind socket to address. Usually in linux the limit for pid's is
   * 32768 and therefore this digit can be added to actual pid (if small) and wouldn't cause
   * any confusion hence this is above the limit set by OS. The actual id in this case is iPid
   * - 32768 and the pseudopid used for PeerHood operations is still unique since the existence
   * of the file will be checked and duplicate will not be used.
   */
  if(iPid < 5000)
  {
  	iPid = iPid + 32768;
  	while(CheckPidfile())	iPid++;
  }

  iPinging = false;
  iMonitoring = false;
  iRoaming = false;
  LoadPlugins();
  iEngine->LoadConnections(iPid);
  iEngine->StartListening();
}


/**
 * @memo Destructor.
 * @doc Destructor. During the destruction process the local socket is closed,
 * registered services are removed and all active pingers are stopped and
 * deleted.
 *
 * @return none
 */
CPeerHoodImpl::~CPeerHoodImpl()
{
  delete iEngine; // stops the listening thread on engine!

  if (iConnected) close(iDaemonSocket);

  for (list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
    delete *i;
  }

  if (iPinging) {
    Lock();

    iPinging = false;
    pthread_join(iThread, NULL);

    for (list<MAbstractPinger*>::iterator i = iPingerList.begin();i != iPingerList.end();++i) {
      delete *i;
    }

    Unlock();
  }

  if(iMonitoring)
    {
      iMonitoring = false;
      pthread_join(iSignalThread, NULL);
      delete iSignalMonitor;
    }

  if(iRoaming)
    StopRoamingThread();

  iInstance = NULL;
}


/**
 * @memo Initializes the PeerHood library.
 * @doc Initialized the PeerHood library. The initialization process includes
 * connection establishment to the PeerHood daemon. If the daemon cannot be
 * contacted then initialization has failed.
 *
 * @param aArgc The number of arguments.
 * @param aArgv An array containing all parameters. Currently the only
 * recognized parameter is <code>'-DEBUG='</code> that sets the logging output.
 *
 * @return true if the PeerHood daemon could be contacted
 */
bool CPeerHoodImpl::Init(int aArgc, char** aArgv)
{
  TCommand pdu;
  struct sockaddr_un address;

  if (iConnected) {
    ERR("CPeerHood::Init : already connected");
    return false;
  }

  for (int i = 0;i < aArgc;i++) {
    string arg = string(aArgv[i]);
    if (arg.find("-DEBUG=", 0) == 0) {
      Logger::SetTarget((aArgv[i] + 7));
    }
  }

  // connect to the daemon
  if ((iDaemonSocket = socket(PF_UNIX, SOCK_STREAM, 0)) == -1) {
    ERR("socket");
    return false;
  }

  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, LOCAL_SOCKET_NAME);

  if (connect(iDaemonSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CPeerHoodImpl::Init : connect failed");
    return false;
  }

  pdu.iCommand = PH_INIT;
  pdu.iPid = iPid;

  if (!Write(&pdu, sizeof(pdu))) {
    ERR("CPeerHoodImpl::Init : writing the PH_INIT command failed");
    close(iDaemonSocket);
    return false;
  }

  iConnected = true;

  return true;
}


/**
 * @memo Loads all available plugins.
 * @doc Loads all available plugins from the directory pointed by the
 * PH_PLUGIN_DIR environment variable. If the variable is not set then no
 * plugins will be loaded. All plugins have a filename *peerhood.so.
 *
 * @return none
 */
void CPeerHoodImpl::LoadPlugins()
{
  void* plugin = NULL;
  string pluginName = "";
  string pluginDir = "";
  char in[1024];
  FILE* ls = NULL;
  char* pluginEnv = NULL;

  DBG("CPeerHoodImpl::LoadPlugins : loading plugins");
  memset(in, 0, sizeof(in));

  if (!(pluginEnv = getenv("PH_PLUGIN_DIR"))) {
    struct stat str_stat;
  	if(stat("/usr/lib/peerhood",&str_stat) == 0)
  	{
  		DBG("PH_PLUGIN_DIR not set, using /usr/lib/peerhood as plugin directory.");
  		pluginEnv = "/usr/lib/peerhood";
  	}
  	else
  	{
		  ERR("PH_PLUGIN_DIR not set, no plugins will be loaded, /usr/lib/peerhood doesn't exist.");
		  return;
    }
  }

  pluginDir = string(pluginEnv);

  pluginName = string("ls ");
  pluginName.append(pluginDir);
  pluginName.append("/libpeerhood*plugin.so"); // */ (fool the stupid doc++)

  if (!(ls = popen(pluginName.c_str(), "r"))) {
    ERR("CPeerHoodImpl::LoadPlugins : unable to search for plugins");
    return;
  }

  while (fgets(in, sizeof(in), ls)) {
    pluginName = string(in);
    pluginName.erase(pluginName.size() -1);
    DBG("CPeerHoodImpl::LoadPlugins : loading plugin");
    if (!(plugin = dlopen(pluginName.c_str(), RTLD_LAZY))) {
       fprintf (stderr, "%s\n", dlerror());

      ERR("CPeerHoodImpl::LoadPlugins : plugin failed to load");

    }
    else {
      DBG("CPeerHoodImpl::LoadPlugins : plugin loaded");
    }
  }

  pclose(ls);

  DBG("CPeerHoodImpl::LoadPlugins : plugins loaded");
}


/**
 * @memo Returns a list of all detected devices.
 * @doc Returns a list containing all detected devices. It's caller's
 * responsibility to free the returned list.
 *
 * @return list containing all detected devices
 */
TDeviceList* CPeerHoodImpl::GetDeviceListL()
{
  return GetDeviceListL(NULL);
}


/**
 * @memo Returns a list of devices hosting the given service.
 * @doc Returns a list of devices hosting the given service. The returned list
 * is reserved by this function and the caller should take care of its
 * deletion.
 *
 * @param aServiceName The service to look for.
 *
 * @return list containing all detected devices that host the given service
 */
TDeviceList* CPeerHoodImpl::GetDeviceListL(const string* aServiceName)
{
  TCommand pdu;
  int numberOfDevices;
  TDeviceList* retval;
  int length;
  char *buffer = NULL;
  CLibDevice* temp;


  if (!iConnected) {
    ERR("CPeerHoodImpl::GetDeviceListL : not connected!");
    return NULL;
  }

  pdu.iCommand = PH_GET_DEVICELIST;
  pdu.iPid = iPid;

  if (!Write(&pdu, sizeof(pdu))) {
    ERR("CPeerHoodImpl::GetDeviceList : write failed");
    return NULL;
  }

  if (!Read(&numberOfDevices, sizeof(numberOfDevices))) {
    ERR("CPeerHoodImpl::GetDeviceList : read failed");
    return NULL;
  }

  retval = new TDeviceList;

  // read all devices sent by the daemon
  for (int i = 0;i < numberOfDevices;i++) {
    if (!Read(&length, sizeof(length))) {
      ERR("CPeerHoodImpl::GetDeviceList : read failed");
      delete retval;
      return NULL;
    }

    buffer = new char[length];

    if (!Read(buffer, length)) {
      ERR("CPeerHoodImpl::GetDeviceList : read failed");
      delete[] buffer;
      delete retval;
      return NULL;
    }

    // This buffer needs to contain all info (including protos)
    // First get the services working
    temp = new CLibDevice(buffer);

    // Check if we want just certain services
    if (aServiceName) {
      if (temp->HasService(*aServiceName)) {
	retval->Add(temp);
      }
      else {
	delete temp;
      }
    }
    else {
      retval->Add(temp);
    }

    delete[] buffer;
  }

  return retval;
}


/**
 * @memo Returns a list of local services.
 * @doc Returns a list of local services. The list contains all the services
 * that are registered to the local device.
 *
 * @return list containing all locally registered devices
 */
TServiceList* CPeerHoodImpl::GetLocalServiceListL()
{
  TCommand pdu;
  int numberOfServices;
  int length;
  char *buffer = NULL;
  CService* temp;
  TServiceList* retval;

  if (!iConnected) {
    ERR("CPeerHoodImpl::GetLocalServiceListL : not connected!");
    return NULL;
  }

  pdu.iCommand = PH_GET_LOCAL_SERVICELIST;
  pdu.iPid = iPid;

  if (!Write(&pdu, sizeof(pdu))) {
    ERR("CPeerHoodImpl::GetLocalServiceListL : write failed");
    return NULL;
  }

  if (!Read(&numberOfServices, sizeof(numberOfServices))) {
    ERR("CPeerHoodImpl::GetLocalServiceListL : read failed");
    return NULL;
  }

  retval = new TServiceList;

  // read all services
  for (int i = 0;i < numberOfServices;i++) {
    if (!Read(&length, sizeof(length))) {
      ERR("CPeerHoodImpl::GetLocalServiceListL : read failed");
      delete retval;
      return NULL;
    }

    buffer = new char[length];

    if (!Read(buffer, length)) {
      ERR("CPeerHoodImpl::GetLocalServiceListL : read failed");
      delete buffer;
      delete retval;
      return NULL;
    }

    temp = new CService(buffer, "local");
    retval->Add(temp);

    delete buffer;
  }

  return retval;
}

/**
 * @memo Creates connection to a service.
 * @doc Creates connection to a service. Caller must free and, if necessary,
 * close the returned connection object.
 *
 * @param aDevice The remote device to connect to.
 * @param aServiceName The destination service.
 *
 * @return connection object that is connected to a remote device or NULL if
 * the connection establishmnent failed
 */
MAbstractConnection* CPeerHoodImpl::Connect(TDeviceIterator& aDevice, const string aServiceName)
{
  TServiceList* services;
  char tempPortChar[50];

  services = (*aDevice)->GetServiceListL();

  for (TServiceIterator j = services->Begin(); j != services->End();++j) {
    if(aServiceName.compare((*j)->GetName()) == 0)
      {
	std::string tempProto((*aDevice)->GetPrototype());
	std::string tempAddress((*aDevice)->GetAddress());
	int tempPort = (*j)->GetPid();
	int servicePort = (*j)->GetPort();
	sprintf(tempPortChar, "Port to connect: %d (service port: %d)", tempPort, servicePort);
	DBG(tempPortChar);
	std::string tempName((*aDevice)->GetName());
	unsigned int tempChecksum = (*aDevice)->GetChecksum();
	delete services;
	return Connect(tempProto, tempAddress, tempPort, servicePort, aServiceName, tempName, tempChecksum);
      }
  }
  ERR("CPeerHoodImpl::Connect : can't find requested service");
  delete services;
  return NULL;

}

/**
 * @memo Creates connection to a service.
 * @doc Creates connection to a service. Caller must free and, if necessary,
 * close the returned connection object.
 *
 * @param aDevice The remote device to connect to.
 * @param aServiceName The destination service.
 *
 * @return connection object that is connected to a remote device or NULL if
 * the connection establishmnent failed
 */
MAbstractConnection* CPeerHoodImpl::Connect(MAbstractDevice& aDevice, const string& aServiceName)
{
  TServiceList* services;
  char tempPortChar[50];

  services = aDevice.GetServiceListL();

  for (TServiceIterator j = services->Begin(); j != services->End();++j) {
    if(aServiceName.compare((*j)->GetName()) == 0)
    {
			std::string tempProto(aDevice.GetPrototype());
			std::string tempAddress(aDevice.GetAddress());
			int tempPort = (*j)->GetPid();
			int servicePort = (*j)->GetPort();
			sprintf(tempPortChar, "Port to connect: %d (service port: %d)", tempPort, servicePort);
			DBG(tempPortChar);
			std::string tempName(aDevice.GetName());
			unsigned int tempChecksum = aDevice.GetChecksum();
			delete services;
			char printtemp[300];
			sprintf(printtemp,"Connect(%s,%s,%d,%d,%s,%s,%d)",tempProto.c_str(), tempAddress.c_str(), tempPort, servicePort, aServiceName.c_str(), tempName.c_str(), tempChecksum);
			DBG(printtemp);
			return Connect(tempProto, tempAddress, tempPort, servicePort, aServiceName, tempName, tempChecksum);
			//return Connect(aDevice.GetPrototype(), aDevice.GetAddress(), (*j)->GetPid(), (*j)->GetPort(), aServiceName, aDevice.GetName(), aDevice.GetChecksum()); 
    }
  }
  ERR("CPeerHoodImpl::Connect : can't find requested service");
  delete services;
  return NULL;

}


/**
 * @memo Creates connection to a service.
 * @doc Creates connection to a service. Caller must free and, if necessary,
 * close the returned connection object.
 *
 * @param aService The destination service.
 *
 * @return connection object that is connected to a remote device or NULL if
 * the connection establishmnent failed
 */
MAbstractConnection* CPeerHoodImpl::Connect(TServiceIterator& aService)
{
  ostringstream addr;
  addr << "/tmp/ph." << (*aService)->GetPid();

  return Connect(std::string("local"), addr.str() , (*aService)->GetPid(), (*aService)->GetPort(), (*aService)->GetName(), std::string("localDevice"), 0);
}

/**
 * @memo Creates connection to a service.
 * @doc Creates connection to a service. Caller must free and, if necessary,
 * close the returned connection object. Functionality is exactly the same
 * as in CPeerHoodImpl::Connect(TServiceIterator& aService) but with direct
 * reference to CService-object. This was implemented that the library could
 * be used via C-wrapper functions.
 *
 * @param aService The destination service.
 *
 * @return connection object that is connected to a remote device or NULL if
 * the connection establishmnent failed
 */
MAbstractConnection* CPeerHoodImpl::Connect(CService& aService)
{
  ostringstream addr;
  addr << "/tmp/ph." << aService.GetPid();

  return Connect(std::string("local"), addr.str() , aService.GetPid(), aService.GetPort(), aService.GetName(), std::string("localDevice"), 0);
}

/**
 * @memo Creates connection to a service.
 * @doc Creates connection to a service. Caller must free and, if necessary,
 * close the returned connection object. This method provides a virtul "handle"
 * for the application. This virtual connection is then mapped to some specific
 * technology. If roaming -option is enabled a RoamingThread is created. This thread
 * provides Seamless Connectivity for application.
 *
 * @param aPrototype Targets prototype
 * @param aAddress Target service's address.
 * @param aPort Target service's port (or equivalent).
 * @param aServiceName The destination service.
 * @param aDevice Targets device name
 * @param aChecksum Targets checksum number
 *
 * @return connection object that is connected to a service or NULL if
 * the connection establishmnent failed
 */
MAbstractConnection* CPeerHoodImpl::Connect(const string& aPrototype, const string& aAddress,
					    const int aPort, const int aPid,
					    const string& aServiceName, const string& aDeviceName, unsigned int aChecksum)
{

  u_int8_t command = PH_DIRECT_DATA;
  assert(aServiceName.length() < 256);

  iConnectionId = 0;

  //this thread is a container for passing information to RoamingThread
  CThreadInfo *newThreadInfo = new CThreadInfo;
  newThreadInfo->iInstance = this;
  newThreadInfo->iDeviceName = aDeviceName;
  newThreadInfo->iServiceName = aServiceName;
  newThreadInfo->iPrototype = aPrototype;
  newThreadInfo->iPort = aPort;
  newThreadInfo->iAddress = aAddress;
  newThreadInfo->iConnectionId = iConnectionId;
  newThreadInfo->iChecksum = aChecksum;
  newThreadInfo->iLowCount = 0;
  iVirtualConnection = new CVirtualConnection(aPrototype, newThreadInfo);


  //this follows the peerhood connection protocol:
  //connect, send command and port number (NOTE! not service name anymore)
  //read reply
  //send connection ID
  // ------- EXTENSION TO PROTOCOL----------
  //read information (checksum) request
  //request checksum from daemon
  //receive response from daemon
  //send checksum
  if(!iVirtualConnection->Connect(aAddress, aPort))
  {
    ERR("CPeerHoodImpl::Connect : failed to connect");
    delete iVirtualConnection;
    return NULL;
  }

  if (iVirtualConnection->Write(&command, sizeof(command)) == -1) {
    ERR("CPeerHoodImpl::Connect : failed to create a new direct data connection");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }

  if (iVirtualConnection->Write(&aPid, sizeof(aPid)) == -1) {
    ERR("CPeerHoodImpl::Connect : failed to connect to a service");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }

  if (iVirtualConnection->Read(&command, sizeof(command)) == -1) {
    ERR("CPeerHoodImpl::Connect : failed to read service connection response");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }

  if (command != PH_OK) {
    ERR("CPeerHoodImpl::Connect : service connection request rejected");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }

  iConnectionId = htons(iConnectionId);
  if (iVirtualConnection->Write(&iConnectionId, sizeof(iConnectionId)) == -1) {
    ERR("CPeerHoodImpl::Connect : failed to send a connection id");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }
  
  // Read the extended info request
  if(iVirtualConnection->Read(&command, sizeof(command)) == -1) {
  	ERR("CPeerHoodImpl::Connect : failed to read extended service connection response");
  	iVirtualConnection->Close();
  	delete iVirtualConnection;
  	return NULL;
  }
  
  if(command != PH_GET_PEER_INFO)
  {
  	ERR("CPeerHoodImpl::Connect : extended service connection request rejected");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
  }

	unsigned int checksum = 0;
	TCommand checksumPdu;
	
	checksumPdu.iCommand = PH_GET_CHECKSUM;
	checksumPdu.iPid = iPid;
	
	// Ask device checksum from daemon
	if(!Write(&checksumPdu,sizeof(checksumPdu)))
	{
		ERR("CPeerHoodImpl::Connect : cannot send checksum request to daemon");
		iVirtualConnection->Close();
		delete iVirtualConnection;
		return NULL;
	}
	
	// Read checksum, checksum in network byte order when received from daemon
	if(!Read(&checksum, sizeof(checksum)))
	{
		ERR("CPeerHoodImpl::Connect : cannot read checksum from daemon socket");
		iVirtualConnection->Close();
		delete iVirtualConnection;
		return NULL;
	}
	
	// Send device info to peer
	if(iVirtualConnection->Write(&checksum, sizeof(checksum)) == -1)
	{
		ERR("CPeerHoodImpl::Connect : cannot send extended service connection info");
    iVirtualConnection->Close();
    delete iVirtualConnection;
    return NULL;
	}

  iConnectionId++;

  if(!iRoaming)
    {
      std::string roaming("Roaming");
      bool roam = false;
      if(!LoadParameter(roaming))
	ERR("CPeerHoodImpl::Connect : Can't find roaming option. Disabling roaming");
      else
	{
	  if(roaming.compare("yes") == 0)
	    {
	      if(!StartRoamingThread())
		{
		  ERR("CPeerHoodImpl::Connect : failed to start roaming thread");
		  iVirtualConnection->Close();
		  delete iVirtualConnection;
		  return NULL;
		}
	    }
	}
    }

  return iVirtualConnection;

}

/**
 * @memo Sends data to the PeerHood daemon.
 * @doc Sends data to the PeerHood daemon.
 *
 * @param aBuffer Buffer containg the data to be sent.
 * @param aLength Source data's length.
 *
 * @return true if all data was sent without errors
 */
bool CPeerHoodImpl::Write(const void *aBuffer, int aLength)
{
  if (write(iDaemonSocket, aBuffer, aLength) != aLength) {
    ERR("CPeerHoodImpl::Write : write failed or size mismatch");
    close(iDaemonSocket);
    iConnected = false;

    return false;
  }

  return true;
}


/**
 * @memo Reads data from the PeerHood daemon.
 * @doc Reads data from the PeerHood daemon. The destination buffer must be
 * large enough to hold the requested data.
 *
 * @param aBuffer Destination buffer.
 * @param aLength The amount of bytes to be read.
 *
 * @return true if the requested number of bytes could be read
 */
bool CPeerHoodImpl::Read(void* aBuffer, int aLength)
{
  if (read(iDaemonSocket, aBuffer, aLength) != aLength) {
    ERR("CPeerHoodImpl::Read : read failed or size mishmatch");
    close(iDaemonSocket);
    iConnected = false;

    return false;
  }

  return true;
}

/**
 * @memo Registers a service with predefined port
 * @doc Registers a service to the PeerHood library and the daemon with predefined port.
 * This kind of service could be e.g. a one that is contacted with the direct
 * data connection mode.
 *
 * @param aName Service's name.
 * @param aAttributes Service's attributes.
 * @param aPort Service's port
 *
 * @return port number
 */
unsigned short CPeerHoodImpl::RegisterService(const string& aName, const string& aAttributes, const string& aPort)
{
  int length;
  unsigned short Port = 0;
  unsigned char response;
  char* buffer;
  char* data;
  TCommand* pdu;
  TCommand PortPdu;

  if (!iConnected) {
    ERR("CPeerHoodImpl::RegisterService : Not connected!");
    return 0;
  }

  if (!iCallback) {
    ERR("CPeerHoodImpl::RegisterService : No callback given, can't register any services!");
    return 0;
  }

  //This short part here is added to make a reservation for free port for the service
  Port = (unsigned short) std::atoi(aPort.c_str());

  PortPdu.iCommand = PH_GET_FREE_PORT;
  PortPdu.iPid = iPid;

  if (!Write(&PortPdu, sizeof(PortPdu))) {
    ERR("CPeerHoodImpl::RegisterService: write of command PH_GET_FREE_PORT failed");
    return 0;
  }

  if (!Write(&Port, sizeof(Port))) {
    ERR("CPeerHoodImpl::RegisterService: write of Port failed");
    return 0;
  }

  if (!Read(&Port, sizeof(Port))) {
    ERR("CPeerHoodImpl::RegisterService : failed to read the final port");
    return 0;
  }

  //After knowing that we have a suitable port this new service can be created
  CService* temp = new CService(aName, aAttributes, iPid, Port);

  // Could be done like all other communications .. send command and then data
  data = temp->MarshallL(&length);
  length += sizeof(struct TCommand);
  buffer = new char[length];
  pdu = (TCommand *)buffer;
  pdu->iCommand = PH_INSERT_SERVICE;
  pdu->iPid = iPid;
  memcpy(buffer + sizeof(struct TCommand), data, length - sizeof(struct TCommand));
  delete[] data;

  if (!Write(buffer, length)) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed");
    delete[] buffer;
    delete temp;

    return 0;
  }

  delete[] buffer;

  if (!Read(&response, sizeof(response))) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed");
    delete temp;

    return 0;
  }

  // Not actually needed any more as there might be many services of the same name
  // This place could in future be used to check other error possibilities
  if (response == PH_ALREADY_REGISTERED) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed: the name already exists");
    delete temp;
    return 0;
  }

  iServiceList.push_back(temp);

  return Port;
}


/**
 * @memo Registers a service.
 * @doc Registers a service to the PeerHood library and the daemon.
 * This kind of service could be e.g. a one that is contacted with the direct
 * data connection mode.
 *
 * @param aName Service's name.
 * @param aAttributes Service's attributes.
 *
 * @return port number, if the service was registered succesfully - 0 otherwise
 */
unsigned short CPeerHoodImpl::RegisterService(const string& aName, const string& aAttributes)
{
  int length;
  unsigned char response;
  unsigned short Port = 0;
  char* buffer;
  char* data;
  TCommand* pdu;
  TCommand  PortPdu;

  if (!iConnected) {
    ERR("CPeerHoodImpl::RegisterService : Not connected!");
    return 0;
  }

  if (!iCallback) {
    ERR("CPeerHoodImpl::RegisterService : No callback given, can't register any services!");
    return 0;
  }

  //Here we need to get a suitable port for the service
  PortPdu.iCommand = PH_GET_FREE_PORT;
  PortPdu.iPid = iPid;

  if (!Write(&PortPdu, sizeof(PortPdu))) {
    ERR("CPeerHoodImpl::RegisterService: write of command PH_GET_FREE_PORT failed");
    return 0;
  }

  if (!Write(&Port, sizeof(Port))) {
    ERR("CPeerHoodImpl::RegisterService: write of Port failed");
    return 0;
  }

  if (!Read(&Port, sizeof(Port))) {
    ERR("CPeerHoodImpl::RegisterService : failed to read the final port");
    return 0;
  }

  // And then it is possible to use this with port
  CService* temp = new CService(aName, aAttributes, iPid, Port);

  data = temp->MarshallL(&length);
  length += sizeof(struct TCommand);
  buffer = new char[length];
  pdu = (TCommand *)buffer;
  pdu->iCommand = PH_INSERT_SERVICE;
  pdu->iPid = iPid;
  memcpy(buffer + sizeof(struct TCommand), data, length - sizeof(struct TCommand));
  delete[] data;

  if (!Write(buffer, length)) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed");
    delete[] buffer;
    delete temp;

    return 0;
  }

  delete[] buffer;

  if (!Read(&response, sizeof(response))) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed");
    delete temp;

    return 0;
  }

  // Not actually needed any more as there could be of many services with the same name
  if (response == PH_ALREADY_REGISTERED) {
    ERR("CPeerHoodImpl::RegisterService : inserting a service to the daemon failed: the name already exists");
    delete temp;
    return 0;
  }

  iServiceList.push_back(temp);

  return Port;
}

/**
 * @memo Unregisters a method of the given name. Notice that this destroys ALL services of the given name
 * @doc Unregisters a method. The requested service is removed from the
 * PeerHood library and the daemon. Note that the actual service object is not
 * deleted - it's the application's responsibility to take care of that.
 *
 * @param aService Service's name.
 *
 * @return true if the service was succesfully unregistered
 */
bool CPeerHoodImpl::UnregisterService(const string& aService)
{
  unsigned short Port=0;
  TCommand pdu;
  bool Deleted = false;

  DBG("CPeerHoodImpl::UnregisterService : removing all services with given name");

  list<CService*>::iterator i = iServiceList.begin();

  while(i != iServiceList.end()) {
    if ((*i)->GetName().compare(aService) == 0) {

      DBG("CPeerHoodImpl::UnregisterService : Name matches");

      pdu.iCommand = PH_REMOVE_SERVICE;
      pdu.iPid = iPid;

      if (!Write(&pdu, sizeof(pdu))) {
	ERR("CPeerHoodImpl::UnregisterService: write of command PH_REMOVE_SERVICE failed");
	return false;
      }

      Port = (*i)->GetPort();

      if (!Write(&Port, sizeof(Port))) {
	ERR("CPeerHoodImpl::RegisterService: write of Port failed");
	return false;
      }

      delete *i;
      i = iServiceList.erase(i);

      Deleted = true;
    }
    else {
      DBG("CPeerHoodImpl::UnregisterService : Name does not match");
      ++i;
    }
  }

  return Deleted;
}

/**
 * @memo Unregisters a service.
 * @doc Unregisters a method. The requested service is removed from the
 * PeerHood library and the daemon. Note that the actual service object is not
 * deleted - it's the application's responsibility to take care of that.
 *
 * @param aService Service's name.
 *
 * @return true if the service was succesfully unregistered
 */
bool CPeerHoodImpl::UnregisterService(const string& aService, const string& aPort)
{
  unsigned short Port=0;
  TCommand pdu;

  // Convert the port
  Port = (unsigned short) std::atoi(aPort.c_str());

  if (Port) {
    for (list<CService*>::iterator i = iServiceList.begin();i != iServiceList.end();++i) {
      if ((*i)->GetPort() == Port) {

	pdu.iCommand = PH_REMOVE_SERVICE;
	pdu.iPid = iPid;

	if (!Write(&pdu, sizeof(pdu))) {
	  ERR("CPeerHoodImpl::UnregisterService: write of command PH_REMOVE_SERVICE failed");
	  return false;
	}

	if (!Write(&Port, sizeof(Port))) {
	  ERR("CPeerHoodImpl::RegisterService: write of Port failed");
	  return false;
	}

	// free memory of the list element in i and then remove list element
	delete *i;
	iServiceList.erase(i);

	return true;
      }
    }
  }

  if(!UnregisterService(aService)) {
    ERR("CPeerHoodImpl::UnregisterService : failed with port and name");
    return false;
  }

  return true;
}

/**
 * @memo Sets the prefered plugin.
 * @doc Sets the prefered networking plugin. This function can be used to
 * override the setting 'PreferedPlugin' from the configuration file. Changes
 * will take effect only from the moment of call, i.e. running services are not
 * affected.
 *
 * @param aPluginName New default plugin's name.
 *
 * @return none
 */
void CPeerHoodImpl::SetPreferedPlugin(const char* aPluginName)
{
  iPreferedPlugin = string(aPluginName);
}


/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CPeerHoodImpl::LoadParameter(string& aParameter)
{
  char* home;
  string line;
	string path;

  if ((home = getenv("HOME")) == NULL) {
  	path = "/etc/default/phconfig";
    //return false;
  }
  else
  {
  	path = string(home);
  	path.append("/.phconfig");
  }
  
  struct stat st;
  
  if(stat(path.c_str(),&st) == -1)
  {
  	path.clear();
  	path = std::string("/etc/default/phconfig");
  }
  
  ifstream in(path.c_str());

  if (!in.is_open()) {
    ERR("CPeerHoodImpl::LoadParameter : Failed to open the configuration file");
    in.close();
    return false;
  }

  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    string::size_type position = line.find_first_of(':');

    if (position != string::npos) {
      string parameter = line.substr(0, position);
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}


/**
 * @memo Loads an integral parameter from the configuration file.
 * @doc Loads an integral parameter from the configuration file. If the given
 * paramter is found then the argument <code>aParameter</code> is replaced with
 * the parameter's value.
 *
 * @param aName Parameter's name.
 * @param aParameter Will contain asked parameter's value on return. This
 * argument cannot be NULL.
 *
 * @return true if the requested parameter could be loaded
 */
bool CPeerHoodImpl::LoadParameter(char* aName, int* aParameter)
{
  string name(aName);

  assert(aParameter != NULL);

  if (!LoadParameter(name)) return false;

  *aParameter = atoi(name.c_str());

  return true;
}


/**
 * @memo Monitors a remote device.
 * @doc Monitors a remote device. If a change (connection loss, return to
 * range etc.) happens then the registered callback interface is notified.
 *
 * @param aDevice Iterator pointing to the target device.
 *
 * @return true if the monitoring could be started
 */
bool CPeerHoodImpl::MonitorDevice(TDeviceIterator& aDevice)
{
  // CRITICAL SECTION STARTS
  Lock();

  for (list<MAbstractPinger*>::iterator i = iPingerList.begin();i != iPingerList.end();++i) {
    if ((*i)->GetAddress().compare((*aDevice)->GetAddress()) == 0) {
      ERR("CPeerHoodImpl::MonitorDevice : device already under monitoring");
      Unlock();
      return false;
    }
  }

  MAbstractPinger* pinger = Factory::CreatePingerL((*aDevice)->GetPrototype(), (*aDevice)->GetAddress());
  iPingerList.push_back(pinger);

  if (!iPinging) {
    if (pthread_create(&iThread, NULL, *CPeerHoodImpl::ThreadStarter, this) != 0) {
      ERR("CPeerHoodImpl::MonitorDevice : failed to spawn the pinger thread!");
      delete *iPingerList.end();
      iPingerList.erase(iPingerList.end());
      Unlock();
      return false;
    }

    iPinging = true;
  }

  // CRITICAL SECTION ENDS
  Unlock();

  return true;
}


/**
 * @memo Ceases device monitoring.
 * @doc Ceases device monitoring.
 *
 * @param aDevice Target device.
 *
 * @return true if the device was under monitoring, otherwise false
 */
bool CPeerHoodImpl::UnmonitorDevice(TDeviceIterator& aDevice)
{
  // CRIICAL SECTION STARTS
  Lock();

  for (list<MAbstractPinger*>::iterator i = iPingerList.begin();i != iPingerList.end();++i) {
    if ((*i)->GetAddress().compare((*aDevice)->GetAddress()) == 0) {
      DBG("CPeerHoodImpl::UnmonitorDevice : deleting a pinger");
      delete *i;
      iPingerList.erase(i);

      // Unlock is duplicated to prevent the pinger thread from deadlocking
      if (iPingerList.size() == 0) {
	Unlock();
	iPinging = false;
	assert(pthread_join(iThread, NULL) == 0);
      }
      else {
	Unlock();
      }

      return true;
    }
  }

  // CRITICAL SECTION ENDS
  Unlock();

  ERR("CPeerHoodImpl::UnmonitorDevice : tried to delete a non-existing pinger!");

  return false;
}

/**
 * @memo Monitors a remote device.
 * @doc Monitors a remote device.
 *
 * @param aDevice Iterator pointing to the target device.
 *
 * @return true if the monitoring could be started
 */
bool CPeerHoodImpl::SignalMonitorDevice(TDeviceIterator& aDevice)
{

  if(!iMonitoring)
    {
      iSignalMonitor = Factory::CreateMonitorL((*aDevice)->GetPrototype(), (*aDevice)->GetAddress());
      iMonitoring = true;
      if (pthread_create(&iSignalThread, NULL, *CPeerHoodImpl::SignalThreadStarter, this) != 0) {
	ERR("CPeerHoodImpl::SignalMonitorDevice : failed to spawn the monitor thread!");
	return false;
      }
    }
  else
    {
      ERR("CPeerHoodImpl::SignalMonitorDevice : already monitoring!");
      return false;
    }
  return true;
}


/**
 * @memo Ceases device monitoring.
 * @doc Ceases device monitoring.
 *
 * @param aDevice Target device.
 *
 * @return true if the device was under monitoring, otherwise false
 */
bool CPeerHoodImpl::SignalUnmonitorDevice()
{
  if(iMonitoring)
    {
      iMonitoring = false;
      DBG("CPeerHoodImpl::SignalUnmonitorDevice : stopping monitoring thread...");
      if(pthread_join(iSignalThread, NULL) != 0) {
	ERR("CPeerHoodImpl::SignalUnmonitorDevice : failed to join monitor thread");
	delete iSignalMonitor;
	return false;
      }
    }
  else
    {
      ERR("CPeerHoodImpl::SignalMonitorDevice : not monitoring!");
      return false;
    }
  delete iSignalMonitor;
  return true;
}


/**
 * @memo Starts the monitoring thread.
 * @doc Starts the monitoring thread. This is function is used because the
 * <code>pthreads</code> library doesn't allow to start a member function as a
 * thread.
 *
 * @param aArguments Pointer to the currently running instance of the class.
 *
 * @return always NULL
 */
void* CPeerHoodImpl::ThreadStarter(void* aArguments)
{
  CPeerHoodImpl* fake = (CPeerHoodImpl *)aArguments;

  fake->PingerThread();

  return NULL;
}


/**
 * @memo Starts the signal monitoring thread.
 * @doc Starts the signal monitoring thread. This is function is used because the
 * <code>pthreads</code> library doesn't allow to start a member function as a
 * thread.
 *
 * @param aArguments Pointer to the currently running instance of the class.
 *
 * @return always NULL
 */
void* CPeerHoodImpl::SignalThreadStarter(void* aArguments)
{
  CPeerHoodImpl* fake = (CPeerHoodImpl *)aArguments;

  fake->MonitorThread();

  return NULL;
}

/**
 * @memo Starts the roaming thread.
 * @doc Starts the roaming thread. This is function is used because the
 * <code>pthreads</code> library doesn't allow to start a member function as a
 * thread.
 *
 * @param aArguments Pointer to the currently running instance of the class.
 *
 * @return always NULL
 */
void* CPeerHoodImpl::RoamingThreadStarter(void* aArguments)
{
  CThreadInfo *tempThreadInfo = (CThreadInfo *)aArguments;
  CPeerHoodImpl* fake = (CPeerHoodImpl *)aArguments;
  fake->RoamingThread();
  return NULL;
}

/**
 * @memo Stops the roaming thread.
 * @doc Stops the roaming thread.
 *
 * @param aArguments Pointer to the currently running instance of the class.
 *
 * @return boolean for errors
 */
bool CPeerHoodImpl::StopRoamingThread()
{
  if(!iRoaming)
    return false;

  iRoaming = false;
  if(pthread_join(iRoamingThread, NULL) != 0) {
    ERR(" CPeerHoodImpl::StopRoamingThread : failed to join roaming thread");
    return false;
  }
  return true;
}


/**
 * @memo Creates a new thread for Roaming Thread
 * @param Creates a new thread for Roaming Thread
 *
 * @return true if the thread could be created
 */
bool CPeerHoodImpl::StartRoamingThread()
{
  if(iRoaming)
    return false;

  iRoaming = true;
  if (pthread_create(&iRoamingThread, NULL, *CPeerHoodImpl::RoamingThreadStarter, this) != 0) {
    ERR("CPeerHoodImpl::StartRoamingThread : failed to spawn the roaming thread!");
    return false;
  }

  return true;
}


/**
 * @memo Monitor thread's function.
 * @doc Function that the dedicated monitoring thread is running on. Devices
 * under monitoring are checked one at a time so that the interval one device
 * is checked is approximately the value 'MonitoringInterval' from the
 * configuration file. If a device is lost or found again then the registered
 * callback interface is notified.
 *
 * @return none
 */
void CPeerHoodImpl::PingerThread()
{
  int sleepTime;
  unsigned int currentPinger = 0;
  bool inRange;
  bool ping;
  list<MAbstractPinger*>::iterator iterator;

  for (;;) {
    if ((sleepTime = iMonitoringInterval / iPingerList.size()) < 1) sleepTime = 1;

    for (int i = 0;i < sleepTime;i++) {
      ERR("sleep");
      if (!iPinging) return;
      sleep(1);
    }

    // CRITICAL SECTION STARTS
    Lock();

    if (!iPinging) {
      Unlock();
      return;
    }

    if (++currentPinger > iPingerList.size()) currentPinger = 1;

    iterator = iPingerList.begin();

    for (unsigned int i = 1;i < currentPinger;i++) iterator++;

    inRange = (*iterator)->InRange();
    ping = (*iterator)->Ping();

    // first unsuccesfull ping
    if (inRange && !ping) {
      if (iCallback) iCallback->Notify(PH_DEVICE_LOST, string((*iterator)->GetAddress()));
    }

    // first succesfull ping after connection lost
    else if (!inRange && ping) {
      if (iCallback) iCallback->Notify(PH_DEVICE_FOUND, string((*iterator)->GetAddress()));
    }

    // CRITICAL SECTION ENDS
    Unlock();
  }
}

/**
 * @memo Signal monitor thread's function.
 * @doc Function that the dedicated monitoring thread is running on.
 * Method notifies callback PH_WEAK_LINK if link level goes down
 *
 * @return none
 */
void CPeerHoodImpl::MonitorThread()
{
  int signalValue = 0;
  for(;;)
    {
      if(!iMonitoring)
	return;

      signalValue = iSignalMonitor->Monitor();

      if (iSignalMonitor->GetAddress().find(":") < iSignalMonitor->GetAddress().size())
	{
	  if(signalValue < 230)
	    {
	      if (iCallback) iCallback->Notify(PH_WEAK_LINK, iSignalMonitor->GetAddress());

	    }
	}
      else
	{
	  if(signalValue < 50)
	    {
	      if (iCallback) iCallback->Notify(PH_WEAK_LINK, iSignalMonitor->GetAddress());
	    }
	}

      sleep(1);
    }

}

/**
 * @memo Roaming Thread Implementation
 * @doc Roaming Thread Implementation. This thread monitores current connections
 * and tries to replace weak connections using alternative networking technology.
 * iThreadList is used by RoamingThread and VirtualConnection class for to contain
 * all the estblished connections. If these two
 * threads try to change or read this list at the sametime problems arise. PeerHoodImpl
 * class uses iCopying boolean member variable which is used for "locking" the iThreadList.
 * RoamingThread first locks the iThreadList (virtualconnection can't change it)
 * and then makes a local copy of it and releases the "lock". More specific description
 * in Thesis.
 *
 * @return none
 */
void CPeerHoodImpl::RoamingThread()
{
  int state = 0, signalValue = 0, lowCount = 0;
  std::string newAddress = "";
  std::string newProto = "";
  int newPort = 0;
  TServiceList* services;
  bool serviceFound = false;
  MAbstractConnection* retval;
  int daemonSocket;
  struct sockaddr_un address;
  int servicePort = 0;


  // connect to the daemon
  if ((daemonSocket = socket(PF_UNIX, SOCK_STREAM, 0)) == -1) {
    ERR("CPeerHoodImpl::RoamingThread : socket");
    iRoaming = false;
    return;
  }

  address.sun_family = PF_UNIX;
  strcpy(address.sun_path, "/tmp/phd.local");

  if (connect(daemonSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    ERR("CPeerHoodImpl::RoamingThread : connect");
    iRoaming = false;
    return;
  }

  std::list<CThreadInfo> iTempThreadList;
  while(iRoaming)
    {
      //clear local list and make copy of orignal list
      iTempThreadList.clear();
      iTempThreadList.resize(0);

      iCopying = true;

      sleep(1);
      std::cerr << "list size is " << iThreadList.size() << std::endl;
      for (list<CThreadInfo*>::iterator i = iThreadList.begin();i != iThreadList.end();++i)
	{
	  CThreadInfo *tempThreadList = new CThreadInfo;
	  tempThreadList->iDeviceName = (*i)->iDeviceName;
	  tempThreadList->iServiceName = (*i)->iServiceName;
	  tempThreadList->iAddress = (*i)->iAddress;
	  tempThreadList->iPrototype = (*i)->iPrototype;
	  tempThreadList->iPort = (*i)->iPort;
	  tempThreadList->iChecksum = (*i)->iChecksum;
	  tempThreadList->iLowCount = (*i)->iLowCount;
	  tempThreadList->iConnectionId = (*i)->iConnectionId;
	  iTempThreadList.push_back((*tempThreadList));
	  delete tempThreadList;
	}

      iCopying = false;

      for (list<CThreadInfo>::iterator i = iTempThreadList.begin();i != iTempThreadList.end();++i)
	{

	  switch(state)
	    {
	    case 0:
	      {
		serviceFound = false;

		TCommand pdu;
		TDeviceList* retval;
		char *buffer = NULL;
		CLibDevice* temp;

		pdu.iCommand = PH_INIT;
		pdu.iPid = 0;

		if (!write(daemonSocket, &pdu, sizeof(pdu))) {
		  ERR("CPeerHoodImpl::RoamingThread : writing the PH_INIT command");
		  close(daemonSocket);
		  iRoaming = false;
		  break;
		}

		pdu.iCommand = PH_GET_DEVICELIST;
		pdu.iPid = 0;

		if (!write(daemonSocket, &pdu, sizeof(pdu))) {
		  ERR("CPeerHoodImpl::RoamingThread writing PH_GET_DEVICELIST failed");
		  close(daemonSocket);
		  iRoaming = false;
		  break;
		}

		int numberOfDevices = 0;
		if (!read(daemonSocket, &numberOfDevices, sizeof(numberOfDevices))) {
		  ERR("CPeerHoodImpl::RoamingThread : reading numberofDevices failed");
		  close(daemonSocket);
		  iRoaming = false;
		  break;
		}

		retval = new TDeviceList;
		int length = 0;
		// read all devices
		for (int j = 0;j < numberOfDevices;j++) {
		  if (!read(daemonSocket, &length, sizeof(length))) {
		    ERR("CPeerHoodImpl::RoamingThread : reading length of device failed");
		    delete retval;
		    iRoaming = false;
		    break;
		  }

		  buffer = new char[length];

		  if (!read(daemonSocket, buffer, length)) {
		    ERR("CPeerHoodImpl::RoamingThread : reading device failed");
		    delete[] buffer;
		    delete retval;
		    iRoaming = false;
		    break;
		  }

		  temp = new CLibDevice(buffer);
		  retval->Add(temp);
		  delete[] buffer;
		}

		for (TDeviceIterator j = retval->Begin();j != retval->End();++j) {
		  if(((*j)->GetName().compare(i->iDeviceName) == 0) && ((*j)->GetChecksum() == i->iChecksum) && ((*j)->GetPrototype().compare(i->iPrototype) != 0))
		    {
		      services = (*j)->GetServiceListL();
		      for (TServiceIterator k = services->Begin(); k != services->End();++k) {
			if((*k)->GetName().compare(i->iServiceName) == 0)
			  {
			    newAddress = std::string((*j)->GetAddress());
			    newProto = std::string ((*j)->GetPrototype());
// 			    newPort = (*k)->GetPort();
			    newPort = (*k)->GetPid();
			    servicePort = (*k)->GetPort();
			    serviceFound = true;
			  }
		      }
		      delete services;
		    }
		}
		// delete temp;
		delete retval;
		state = 1;
		break;
	      }//case 0:

	    case 1:
	      {
		MAbstractMonitor *signalMonitor = Factory::CreateMonitorL(i->iPrototype, i->iAddress);

		signalValue = signalMonitor->Monitor();
		std::cerr << "Current connection signal = " << signalValue << std::endl;
		delete signalMonitor;


		bool signalLow = false;
		if(i->iPrototype.compare("bt-base") == 0)
		  {
		    if(signalValue < 230)
		      {
			;
			// if (iCallback) iCallback->Notify(PH_WEAK_LINK, iSignalMonitor->GetAddress());
			signalLow = true;
		      }
		  }

		if(i->iPrototype.compare("wlan-base") == 0)
		  {
		    if(signalValue < 50)
		      {
			  ;
			  //if (iCallback) iCallback->Notify(PH_WEAK_LINK, iSignalMonitor->GetAddress());
			  signalLow = true;
		      }
		  }
		if(i->iPrototype.compare("gprs-base") == 0)
		  {
		      ;
		  }

		iCopying = true;
		for (list<CThreadInfo*>::iterator j = iThreadList.begin();j != iThreadList.end();++j)
		  {
		    if(memcmp(&i, (*j), sizeof(i)))
		      {
			if(signalLow)
			  {
			    (*j)->iLowCount++;
			    i->iLowCount = (*j)->iLowCount;
			    if( (*j)->iLowCount > 3)
			      (*j)->iLowCount = 0;
			  }
			break;
		      }
		  }
		iCopying = false;

		if(i->iLowCount > 3)
		  {
		    if(serviceFound)
		      state = 2;
		    else
		      {
			state = 0;
			i->iLowCount = 0;
		      }
		  }
		else
		  state = 0;

		break;
	      }

	    case 2:
	      {
		u_int8_t command = PH_REESTABLISH;
		  assert(i->iServiceName.length() < 256);
		  state = 0;
		  std::cerr << "CPeerHoodImpl::RoamingThread : trying to connect " << newAddress << " on " << newPort << std::endl;
		  retval = Factory::CreateConnectionL(newProto);
		  if (!retval)
		    {
		      iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		      break;
		    }

		  if (!retval->Connect(newAddress, newPort)) {
		    ERR("CPeerHoodImpl::RoamingThread : failed to connect");
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }

		  if (retval->Write(&command, sizeof(command)) == -1) {
		    ERR("CPeerHoodImpl::RoamingThread : failed to write command");
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }

		  if (retval->Write(&servicePort, sizeof(servicePort)) == -1) {
		    ERR("CPeerHoodImpl::RoamingThread : failed to write service Port");
		    retval->Close();
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }


// 		  if (retval->Write(i->iServiceName.c_str(), i->iServiceName.length()) == -1) {
// 		    ERR("CPeerHoodImpl::RoamingThread : failed to write servicename");
// 		    retval->Close();
// 		    delete retval;
// 		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
// 		    break;
// 		  }

		  if (retval->Read(&command, sizeof(command)) == -1) {
		    ERR("CPeerHoodImpl::RoamingThread : failed to read command");
		    retval->Close();
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }

		  if (command != PH_OK) {
		    ERR("CPeerHoodImpl::RoamingThread : service rejected connection");
		    retval->Close();
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }

		  u_int8_t connectionId = i->iConnectionId;
		  connectionId = htons(connectionId);

		  if (retval->Write(&connectionId, sizeof(connectionId)) == -1) {
		    ERR("CPeerHoodImpl::RoamingThread : failed to send a connection id");
		    retval->Close();
		    delete retval;
		    iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		    break;
		  }

		  iVirtualConnection->SetConnectionType(retval, newProto, newAddress, newPort);
		  break;
	      }

	    }//switch(state)

	  //timer determines the looping speed
	  //of going through the connections from the list
	  for(int j=0;j<5;j++)
	    {
	      if(!iRoaming)
		break;
	      sleep(1);
	    }

	}//for (list<CThreadInfo*>::iterator i = iThreadList.begin();i != iThreadList.end();++i)

    }//for(;;)


  //  for (list<CThreadInfo>::iterator i = iTempThreadList.begin();i != iTempThreadList.end();++i)
    //    delete i;
#warning "CPeerHoodImpl::RoamingThread : possible memory leak"
  iTempThreadList.clear();
  iTempThreadList.resize(0);

  return;
}

/**
 * @memo Sets library's internal lock.
 * @doc Sets library's internal lock. This lock is used to prevent multiple
 * threads from accessing the same monitoring data at the same time.
 *
 * @return none
 */
void CPeerHoodImpl::Lock()
{
  pthread_mutex_lock(&iLock);
}


/**
 * @memo Unsets the internal lock.
 * @doc Unsets library's internal locking mutex. This lock is used to prevent
 * multiple threads to access the same monitoring data at the same time.
 *
 * @return none
 */
void CPeerHoodImpl::Unlock()
{
  pthread_mutex_unlock(&iLock);
}

/*
 * Check if pid file exists in /tmp/
 *
 * @return true if exists, false if not.
 */
bool CPeerHoodImpl::CheckPidfile()
{
	struct stat str_stat;
	std::string path = "/tmp/ph.";
	
	// Create path for pid file
	std::stringstream strs;
	strs << iPid;
	path.append(strs.str());

	if(stat(path.c_str(),&str_stat) == 0)
	{
		DBG("CPeerHoodImpl::CheckPidFile: pseudo pid in use:");
		DBG(path.c_str());
		return true;
	}
	else
	{
		DBG("CPeerHoodImpl::CheckPidFile: pseudo pid not used:");
		DBG(path.c_str());
		return false;
	}
}

