/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2008 LUT. .
 *
 * @name MaemoSystemListener.cc
 * @memo Implementation of the MaemoSystemListener.
 *
 * @version 0.18
 * date     25.08.2008
 * change   17.12.2008
 */

#include <cstdlib>
//#include <time.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>
#include "MaemoSystemListener.h"

using namespace std;

// Definitions for battery signals - no header for these (?)
#define BME_SIGNAL_IF "com.nokia.bme.signal"
#define BME_SIGNAL_PATH "/com/nokia/bme/signal"
#define BME_REQUEST_IF "com.nokia.bme.request"
#define BME_REQUEST_PATH "/com/nokia/bme/request"
#define BME_TIMELEFT_REQ "timeleft_info_req"
#define BME_STATUS_REQ "status_info_req"
#define BME_BATTERY_LOW "battery_low"
#define BME_BATTERY_TIMELEFT "battery_timeleft"
#define BME_CHARGER_CONNECTED "charger_connected"
#define BME_CHARGER_DICONNECTED "charger_disconnected"
#define BME_CHARGING_ON "charger_charging_on"
#define BME_CHARGING_OFF "charger_charging_off"

/**
 * @memo Default constructor - not used
 * @doc Cannot be used.
 */
CMaemoSystemListener::CMaemoSystemListener()
{
	// Get PeerHood daemon and register
	iConnection = NULL;
	iName = std::string("MaemoSystemListener");
	iConverter = NULL;
}

/**
 * @memo Constructor, initializes necessary variables.
 * @doc Constructor sets the name (hardcoded..) of the listener and registers
 * itself with given component.
 * 
 * @param aConverter Component that will own this listener, cannot be NULL!
 * 
 */
CMaemoSystemListener::CMaemoSystemListener(MAbstractStateConverter* aConverter)
{
	iConverter = aConverter;
	iConnection = NULL;
	iName = std::string("MaemoSystemListener");
	iBatterylow = false;
	if(iConverter != NULL) iConverter->RegisterListener(this);
}
/**
 * @memo Destructor, closes connection to DBUS
 * @doc Destructor only closes the connection to DBUS if it seems to be active.
 * If iConnection isn't NULL it will be closed - closing cannot be checked.
 */
CMaemoSystemListener::~CMaemoSystemListener()
{
	if(iConnection != NULL)
	{
		// Returns void - cannot be checked
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

/**
 * @memo Connects listener into defined DBUS-interface
 * @doc A facade for conneting the listener to DBUS-interface, actually calls
 * internal functions to set up the connection and to register listener.
 * 
 * @return bool, true if success
 */
bool CMaemoSystemListener::Connect()
{
	if(!SetupConnection())	return false;
	
	if(!RegisterSignals()) return false;
	
	return true;
}

/**
 * @memo Disconnects the listener from DBUS
 * @doc Disconnects the connection to DBUS (required by private DBUS connection)
 * if the connection wasn't closed earlier.
 */
void CMaemoSystemListener::Disconnect()
{
	if(iConnection != NULL)
	{
		dbus_connection_close(iConnection);
		iConnection = NULL;
	}
}

/**
 * @memo Checks the system state at startup
 * @doc Checks the Maemo environment state at startup by sending a pending call
 * (request) into the interface of mce requesting the current state of the system. 
 * Function then waits for the answer and sets the state of the daemon corresponding
 * to the state returned by DBUS. Possible states are NORMAL and FLIGHT modes,
 * if the mode is unknown daemon is set to offline/passive mode. Doesn't check the 
 * battery state.
 * 
 */
void CMaemoSystemListener::CheckInitialState()
{
	DBusMessage* msg = NULL;
	DBusPendingCall* statecall = NULL;
	DBusMessageIter iter;
	const char* mdata = NULL; 
	int msgtype;
	
	// Connection lost
	if(iConnection == NULL) return;
	
	// Create a method call
	msg = dbus_message_new_method_call(MCE_SERVICE,MCE_REQUEST_PATH,MCE_REQUEST_IF,MCE_DEVICE_MODE_GET);
	
	if(msg == NULL)
	{
		ERR("MaemoSystemListener::CheckInitialState: Cannot create message.");
		return;
	}
	
	// Reply cannot be sent?
	if(!dbus_connection_send_with_reply(iConnection,msg,&statecall,-1))
	{
		ERR("MaemoSystemListener::CheckInitialState: Cannot send state request.");
		return;
	}
	
	// Cannot do a pending call
	if(statecall == NULL)
	{
		ERR("MaemoSystemListener::CheckInitialState: Pending call failed.");
		return;
	}
	
	// Make sure that data is sent
	dbus_connection_flush(iConnection);
	
	// Don't need the message anymore - unref (free) it
	dbus_message_unref(msg);
	msg = NULL;
	
	// Block until reply received
	dbus_pending_call_block(statecall);
	
	// Check the reply - no reply, do nothing and return
	if((msg = dbus_pending_call_steal_reply(statecall)) == NULL)
	{
		ERR("MaemoSystemListener::CheckInitialState: no reply received.");
		return;
	}
	
	// Iterate all fields
	if(!dbus_message_iter_init(msg,&iter)) return;
	
	// Iterate until the end, DBUS_TYPE_INVALID ends the message
	while ((msgtype = dbus_message_iter_get_arg_type(&iter)) != DBUS_TYPE_INVALID)
	{
		// Device modes are represented as strings
		if (msgtype == DBUS_TYPE_STRING)
		{
			// Get the string
			dbus_message_iter_get_basic(&iter, &mdata);
			
			// Check mode
			if (strcasecmp(mdata, MCE_NORMAL_MODE) == 0)
			{
				// Online - active
				iConverter->SetState(true);
			}
			else if (strcasecmp(mdata, MCE_FLIGHT_MODE) == 0)
			{
				// Offline - passive
				iConverter->SetState(false);
			}
			
			/* Otherwise let's assume that the device is in offline state - can not be sure
			 * so set daemon to passive mode.
			 * Mce should always return one of the previous modes, if this is reached it
			 * probably is an error. */
			else
			{
				DBG("CMaemoSystemListener::CheckInitialState: unknown state: %s", mdata);
				iConverter->SetState(false);
			}
		}

		if (!dbus_message_iter_next(&iter)) break;
	}
	
	// Free pending call
	dbus_pending_call_unref(statecall);
	statecall = NULL;
	
	// Free message
	dbus_message_unref(msg);
	msg = NULL;
}

/**
 * @memo Checks the system state
 * @doc Called periodically by daemon. This function pops the first message from the
 * queue and checks the message for state changes, the queue isn't going to be long
 * because listener has a private connection to DBUS and listens only one interface 
 * that sends only messages regarding the system state change. HandleMessage()-
 * function is called to check the message if there is a message in the queue, otherwise
 * nothing is done.
 */
void CMaemoSystemListener::CheckState()
{
	DBusMessage* message = NULL;
	
	// Check connection
	if(iConnection == NULL) return;
	
	// Allows messages to be read and marshalled from the wire on non-blocking connection
	// Second parameter is blocking time in milliseconds
	dbus_connection_read_write(iConnection,0);
	
	// Pop the first message
	if((message = dbus_connection_pop_message(iConnection)) != NULL)
	{
		HandleMessage(message);
		dbus_message_unref(message);
		message = NULL;
	}
	else return;
}

/**
 * @memo Returns the name of this listener
 * @doc Returns the name that was set in constructor.
 */
const std::string& CMaemoSystemListener::GetName()
{
	return iName;
}

/**
 * @memo Sets up the connection to DBUS
 * @doc Sets up the private connection to DBUS.
 * 
 * @return true when success, false in case of connection error
 */
bool CMaemoSystemListener::SetupConnection()
{
	DBusError error;

	// Initialize error
	dbus_error_init(&error);

	// Get system bus, private connection
	iConnection = dbus_bus_get_private(DBUS_BUS_SYSTEM, &error);

	// Error
	if (iConnection == NULL)
	{
		if (dbus_error_is_set(&error))
		{
			ERR("MaemoSystemListener::SetupConnection: Connection error: %s", error.message);
			
			// Free error
			dbus_error_free(&error);
		}
		return false;
	}
	else return true;
}

/**
 * @memo Registers listener to signal(s) and interface(s)
 * @doc Registers this listener to com.nokia.mce.signal and com.nokia.bme.signal
 *  interfaces for listening system and battery state change messages.
 * 
 * @return true if success, false in case of error
 */
bool CMaemoSystemListener::RegisterSignals()
{
	DBusError error;
	std::string signal_and_if;

	// Check connection
	if(iConnection == NULL) return false;
	
	// Create signal that is listened
	// form: type='signal',interface='com.nokia.mce.signal'
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += MCE_SIGNAL_IF;
	signal_and_if += "\'";

	// Initialize error variable
	dbus_error_init(&error);
	
	// Connect to com.nokia.mce.signal interface listening for 'signal'
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	// Make sure it is sent
	dbus_connection_flush(iConnection);
	
	// If cannot listen 
	if (dbus_error_is_set(&error))
	{
		ERR("CMaemoSystemListener::RegisterSignals: Cannot add listening to signal: %s, reason: %s", signal_and_if.c_str(), error.message);
		dbus_error_free(&error);
		return false;
	}
	
	signal_and_if.clear();
	
	// Connect to com.nokia.bme.signal
	signal_and_if = "type=\'";
	signal_and_if += dbus_message_type_to_string(DBUS_MESSAGE_TYPE_SIGNAL);
	signal_and_if += "\',interface=\'";
	signal_and_if += BME_SIGNAL_IF;
	signal_and_if += "\'";
	
	dbus_error_init(&error);
	dbus_bus_add_match(iConnection, signal_and_if.c_str(), &error);
	dbus_connection_flush(iConnection);
	
	if (dbus_error_is_set(&error))
	{
		ERR("CMaemoSystemListener::RegisterSignals: Cannot add listening to signal: %s, reason: %s", signal_and_if.c_str(), error.message);
		dbus_error_free(&error);
		return false;
	}
	
	signal_and_if.clear();
	
	return true;
}
/**
 * @memo Handles the message (signal) from D-Bus
 * @doc Handles the given message, reacts to signals from com.nokia.mce.signal
 * and com.nokia.bme.signal interfaces. From mce FLIGHT and NORMAL signals are
 * recognized and the state of the connected component is changed accordingly.
 * From bme signals related to battery state changes (battery low, charging) are
 * recognized and reacted. When the battery is on low charge connected component
 * (PeerHood daemon) is set to passive mode.
 * 
 * @param message The message to be handled.
 */
void CMaemoSystemListener::HandleMessage(DBusMessage* message)
{
	DBusMessageIter msg_iter;
	int msgtype;
	// NOTE: no need to free, is a part of the DBusMessage
	const char* mdata = NULL;

	// Message iterator
	dbus_message_iter_init(message, &msg_iter);
	
	// Device state change
	if (dbus_message_is_signal(message, MCE_SIGNAL_IF, MCE_DEVICE_MODE_SIG)) {
		
		// Go through all fields until the end is reached (DBUS_TYPE_INVALID)
		while ((msgtype = dbus_message_iter_get_arg_type(&msg_iter)) != DBUS_TYPE_INVALID)
		{
			// State is represented as string
			if (msgtype == DBUS_TYPE_STRING)
			{
				// Get the string
				dbus_message_iter_get_basic(&msg_iter, &mdata);
				
				// Check mode
				if (strcasecmp(mdata, MCE_NORMAL_MODE) == 0)
				{
					DBG("MaemoSystemListener::HandleMessage: state: online");
					// Online - active
					iConverter->SetState(true);
				}
				else if (strcasecmp(mdata, MCE_FLIGHT_MODE) == 0)
				{
					DBG("MaemoSystemListener::HandleMessage: state: offline");
					// Offline - passive
					iConverter->SetState(false);
				}
			}
			if (!dbus_message_iter_next(&msg_iter)) break;
		}
	}

/* MCE_SHUTDOWN_SIG and MCE_THERMAL_SHUTDOWN_SIG are missing from Maemo Fremantle
 * headers - when built for that device these signals will not be used. */
#ifndef __PH_MAEMO_N900
	// Stopping
	else if (dbus_message_is_signal(message, MCE_SIGNAL_IF, MCE_SHUTDOWN_SIG)) {
		DBG("MaemoSystemListener::HandleMessage: System set to shutdown, stopping.");
		// Call closing
		iConverter->TriggerShutdown();
	}
	
	// Overheating - device closes
	else if (dbus_message_is_signal(message, MCE_SIGNAL_IF, MCE_THERMAL_SHUTDOWN_SIG)) {
		DBG("MaemoSystemListener::HandleMessage: Overheating, stopping.");
		// Call closing
		iConverter->TriggerShutdown();
	}
#endif
	// Battery low, set state to passive to reduce power consumption
	else if(dbus_message_is_signal(message,BME_SIGNAL_IF,BME_BATTERY_LOW)) {
		DBG("MaemoSystemListener::HandleMessage: Battery low.");
		if(!iBatterylow)
		{
			iConverter->SetState(false);
			iBatterylow = true;
		}
	}
	
	// Charger connected, if battery was on low charge, change the state value
	else if(dbus_message_is_signal(message,BME_SIGNAL_IF,BME_CHARGER_CONNECTED))
	{
		DBG("MaemoSystemListener::HandleMessage: Charger connected.");
		if(iBatterylow) iBatterylow = false;
	}
	
	// Charging, set the state to active
	else if(dbus_message_is_signal(message,BME_SIGNAL_IF,BME_CHARGING_ON))
	{
		DBG("MaemoSystemListener::HandleMessage: Battery charging.");
		if(!iBatterylow)
		{
			iConverter->SetState(true);
		}
	}
}
