/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name BTConnection.cc
 * @memo Bluetooth implementation of the MAbstractConnection interface.
 *
 * @version 0.2
 * date     23.04.2003
 * change   28.04.2010
 */

#include <iostream>
#include <cassert>
#include <unistd.h>
#include <sys/socket.h>
#include <bluetooth/bluetooth.h>
#include <bluetooth/l2cap.h>
#include "BTConnection.h"
#include <stdlib.h>
#include <sys/types.h>
#include <arpa/inet.h>


const int PEERHOOD_PSM = 22;
const int L2CAP_MTU = 673;

/**
 * @memo Default constructor.
 * @doc Default constructor, sets the initial vatiables to their default values
 * and creates a new Bluetooth socket.
 *
 * @return none
 */
CBTConnection::CBTConnection()
{
  iConnected = false;
  iIsListening = false;

  iSocket = socket(PF_BLUETOOTH, SOCK_SEQPACKET, BTPROTO_L2CAP);
  assert(iSocket != -1);
  
  iChecksum = 0;
}


/**
 * @memo Connects to a remote device.
 * @doc Connects to a remote device. The connection will be tried to PSM 21
 * which is PeerHood's default PSM.
 *
 * @param aAddress The destination address.
 * @param aPort The destination port.
 *
 * @return true if the connection could be established
 */
bool CBTConnection::Connect(const std::string& aAddress, int aPort)
{
  struct sockaddr_l2 address;

  if (iConnected) {
    return false;
  }

  address.l2_family = AF_BLUETOOTH;
  bdaddr_t *bdaddr;
  bdaddr = strtoba(const_cast<char*>(aAddress.c_str()));

  baswap(&address.l2_bdaddr, bdaddr);
  address.l2_psm = htobs(aPort);
  
  if (connect(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
    free(bdaddr);
    return false;
  }
  
  iConnected = true;
  iRemoteAddress = std::string("BT:") + aAddress;

  free(bdaddr);
  return true;
}


/**
 * @memo Kills the existing connection.
 * @doc Kills the existing connection. If there's no connection or it could not
 * be closed in a controlled way then the error code is returned. The existing
 * connection will be closed regardless of the return value.
 *
 * @return true if the connection was dropped in a controlled manner
 */
bool CBTConnection::Disconnect()
{
  if (!iConnected) {
    return false;
  }
  
  close(iSocket);
  iConnected = false;
  
  return true;
}


/**
 * @memo Explicitly closes the socket.
 * @doc Explicitly closes the socket without any additional state checking,
 * i.e. this function will execute even if there's no existing connection.
 * This function is used by the PeerHood engine to ensure that no file
 * descriptors are leaked.
 *
 * @return none
 */
void CBTConnection::Close()
{
  close(iSocket);
}


/**
 * @memo Sends data to the other end of the connection.
 * @doc Sends data to the other end of the connection. If data is longer than
 * L2CAP's default MTU then it is segmented and sent in multiple chunks. The
 * receiving entity is responsible of reassembling the chunks. Note that this
 * function is not endianess-safe.
 *
 * @param aOutBuf Buffer containing the data that should be written.
 * @param aLength The number of bytes to be written.
 *
 * @return the number of bytes written or -1 in the case of an error
 */
int CBTConnection::Write(const void* aOutBuf, int aLength)
{
  int written = 0;

  if (!iConnected) {
    return -1;
  }
  while (aLength > 0) {
    if (aLength > L2CAP_MTU) {
      written += write(iSocket, (char *)aOutBuf + written, L2CAP_MTU);
      aLength -= L2CAP_MTU;
    }
    else {
      written += write(iSocket, (char *)aOutBuf + written, aLength);
      aLength = 0;
    }
  }

  return written;
}


/**
 * @memo Reads data from an open connection.
 * @doc Reads data from an open connection. If the requested data is longer
 * than L2CAP's maximum MTU then multiple chunks are assumed. Just like 
 * <code>Read</code> this function is not endianess-safe so it should be
 * treated more like a standard Posix sequential socket.
 *
 * @param aInBuf Buffer where the data is read to.
 * @param aLength The number of bytes that should be read.
 *
 * @return the number of bytes read or -1 in the case of an error
 */
int CBTConnection::Read(void* aInBuf, int aLength)
{
  int retval = 0;

  if (!iConnected) {
    return -1;
  }

  while (aLength > 0) {
    if (aLength > L2CAP_MTU) {
      retval += read(iSocket, (char *)aInBuf + retval, L2CAP_MTU);
      aLength -= L2CAP_MTU;
    }
    else {
      retval += read(iSocket, (char *)aInBuf + retval, aLength);
      aLength = 0;
    }
  }

  return retval;
}


/**
 * @memo Returns connection's remote address.
 * @doc Returns connection's remote address. Note that the existence of the
 * address doesn't mean that there's a connection established.
 *
 * @return connection's remote address
 */
const std::string& CBTConnection::GetRemoteAddress()
{  
  return iRemoteAddress;
}


/**
 * @memo Returns connection's file descriptor.
 * @doc Returns connection's file descriptor. Other classes can have a direct
 * access to the connection's socket (or similar entity) via this function's
 * return value.
 *
 * @return connection's file descriptor or -1 if one isn't available
 */
int CBTConnection::GetFd()
{
  if (!iConnected && !iIsListening) return -1;

  return iSocket;
}


/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted. This function will always listen the connection type's default
 * port. If some other port is needed then the <code>Listen(int aPort)</code>
 * should be used instead.
 *
 * @return true if successfull
 */
bool CBTConnection::Listen()
{
  return Listen(PEERHOOD_PSM);
}


/**
 * @memo Sets the connection to the listening state.
 * @doc Sets the connection to the listening state. This function must be
 * called before any incoming connections targeted to this object can be
 * accepted.
 *
 * @param aPort The port that should be listened.
 *
 * @return true if successfull
 */
bool CBTConnection::Listen(int aPort)
{
  bdaddr_t bdaddr;
  struct sockaddr_l2 address;

  baswap(&bdaddr, BDADDR_ANY);

	memset(&address,0,sizeof(struct sockaddr_l2));

  address.l2_psm = htobs(aPort);
  address.l2_family = AF_BLUETOOTH;
  address.l2_bdaddr = bdaddr;
  if (bind(iSocket, (struct sockaddr *)&address, sizeof(address)) == -1) {
  	std::cerr << "CBTConnection::Listen Bind failed, socket: " << iSocket << ", port: " << aPort << std::endl;
    return false;
  }

  if (listen(iSocket, 16) == -1) {
    return false;
  }

  iIsListening = true;

  return true;
}


/**
 * @memo Accepts the incoming connection and returns a new connection object.
 * @doc Accepts the incoming connection and returns a new connection object. If
 * accepting fails then NULL is returned.
 *
 * @return new connection object or NULL in the case of an error
 */
MAbstractConnection* CBTConnection::AcceptL()
{
  int newSocket;
  int addressLength;
  struct sockaddr_l2 address;
  CBTConnection* retval;  
  char tempBuf[18];
  bdaddr_t tempBdaddr;

  addressLength = sizeof(address);

  if ((newSocket = accept(iSocket, (struct sockaddr *)&address, (socklen_t *)&addressLength)) == -1) {
    return NULL;
  }

  retval = new CBTConnection;
  if (retval->iSocket < 0)
      return NULL;
  
  close(retval->iSocket);
  retval->iSocket = newSocket;
  retval->iConnected = true;
  
  memset(&tempBdaddr, 0, sizeof(tempBdaddr));
  baswap(&tempBdaddr, &((struct sockaddr_l2 *)&address)->l2_bdaddr);

  retval->iRemoteAddress =  std::string("BT:") + std::string(batostr(&tempBdaddr));
  
// #warning "CBTConnection::AcceptL : missing remoteaddress"
//   retval->iRemoteAddress = std::string("BT:") + std::string("ei:to:te:ut:et:tu");

  iConnected = true;
  return retval;
}


/**
 * @memo Tells if the connection is listening.
 * @doc Tells if the connection is in the listening state or not.
 *
 * @return true if the connection is in the listening state
 */
bool CBTConnection::IsListening()
{
  return iIsListening;
}


/**
 * @memo Tells if the connection is established.
 * @doc Tells if the connection is established or not.
 *
 * @return true if the connection is established
 */
bool CBTConnection::IsConnected()
{
  return iConnected;
}


/**
 * @memo Tells if the connection has data waiting
 * @doc Checks if the connection has data to read.
 *
 * @return true if the data is ready
 */
bool CBTConnection::HasData()
{
	struct timeval timeout;
	timeout.tv_sec = 0;
	timeout.tv_usec = 0;
  fd_set set;
  FD_ZERO(&set);
  FD_SET(GetFd(), &set);

  switch (select(GetFd() + 1, &set, NULL, NULL, &timeout)) {
  case -1:
  	break;
    
  case 0:
    break;

  default:
    if (FD_ISSET(GetFd(), &set)) {
      return true;
    }
    else {
      return false;
    }
  }

  return false;
}

unsigned int CBTConnection::GetDeviceChecksum()
{
	return iChecksum;
}

void CBTConnection::SetDeviceChecksum(unsigned int aChecksum)
{
	iChecksum = aChecksum;
}

