/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name Logger.cc
 * @memo Logger class used in the PeerHood system.
 *
 * @version 0.1
 * date     03.04.2003
 * change   03.04.2003
 */

#include <iostream>
#include "Logger.h"

Logger* Logger::iInstance = NULL;
std::ostream* Logger::iOut = NULL;


/**
 * @memo Sets the logging output target.
 * @doc Sets the logging output target. By default, all logging goes to stderr.
 * This function can be used to override the default target. A target can be
 * e.g. a file. The target can be changed on the fly.
 *
 * @param aTarget The new logging target.
 *
 * @return none
 */
void Logger::SetTarget(const char* aTarget)
{
  if (!iInstance) {
    iInstance = new Logger;
  }

  if ((!iInstance->iOut) && (iInstance->iOut != &std::cerr)) {
    delete iInstance->iOut;
    iInstance->iOut = NULL;
  }

  if (!aTarget) {
    iInstance->iOut = &std::cerr;
  }
  else {
    iInstance->iOut = new std::ofstream(aTarget, std::ios::app);
  }
}


/**
 * @memo Writes a log message.
 * @doc Writes a log message to the current logging target.
 *
 * @param aType The type of the message. Possible types are:<ll>
 * <li>DEBUG: normal debug output</li>
 * <li>ERROR: abnormal error condition</li></ll>
 *
 * @param aMessage The message.
 *
 * @return none
 */
void Logger::Log(LogType aType, const char* aMessage)
{
  if (!iInstance) {
    SetTarget(NULL);
  }
  
  switch (aType) {
    case DEBUG:
      *iOut << "DEBUG: " << aMessage << std::endl;
      break;
      
    case ERROR:
      *iOut << "ERROR: " << aMessage << std::endl;
      break;

    default:
      *iOut << "FATAL: Unknown log level" << std::endl;
      break;
  }
}
