/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Copyright 2003 LUT. .
 *
 * @name GPRSPlugin.cc
 * @memo GPRS implementation of PeerHood's plugin interface.
 *
 * @version 0.1
 * date     30.06.2003
 * change   14.11.2003
 */

#include <stdio.h>
#include <cstdlib>
#include <ctime>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <iostream>
#include <fstream>
#include <sstream>
#include <csignal>
#include <sys/ioctl.h>
#include <errno.h>

#include "DeviceStorage.h"
#include "pdu.h"
#include "GPRSPlugin.h"
#include "Daemon.h"
#include "ListenerFactory.h"
#include "AbstractStateConverter.h"
#include "PluginStorage.h"

const char* PLUGIN_NAME = "gprs-base";
/** @TODO Add this to config - what IF/dev is needed with GPRS? */
const char* PLUGIN_DEVICE = " ";
const char* LOCAL_SOCKET_NAME = "/tmp/phd.local";

//GPRS Gateway signalling port (data signalling port is GATEWAY + 1)
const int GATEWAY = 9000;

//inquiry variation (not used)
const float INQUIRY_VARIATION = 2.0;

//timer for giving up and trying to reconnect the gateway
const int DISCONNECT_TIMEOUT = 50;

//counter for keeping devices on device-list
const int KEEP_ALIVE_STAMP = 10;

void PipeSignalHandler(int aSignum);

/** 
 * @memo Catches SIGPIPE signal.
 * @doc Catches SIGPIPE signal.
 *
 * @return none
 */
void PipeSignalHandler(int aSignum)
{
}

/** 
 * @memo Default constructor, sets the internal variables to their default
 * values.
 * @doc Default constructor, sets the internal variables to their default
 * values.
 * This method loads gateway address, monitoring and service checking interval variables from
 * configuration file.
 * @return none
 */
CGPRSPlugin::CGPRSPlugin()
{
  iStarted = false;
  iInSdp = false;
  iAdvertFailed = false;
  iInquiryFailed = false;
  iLocalFailed = false;

  iId = 0;
  iName = std::string(PLUGIN_NAME);
  iNDevice = std::string(PLUGIN_DEVICE);
#ifdef __PH_OLD_PLUGIN_STYLE__
	pluginMap[PLUGIN_NAME] = this;
#else
	PluginStorage::GetInstance()->AddPlugin(PLUGIN_NAME, this);
#endif
  srand(time(NULL));
  signal(SIGPIPE, PipeSignalHandler);

  iGateway = "gateway";
  if(!LoadParameter(iGateway))
    {
      ERR("CGPRSPlugin::CGPRSPlugin : Gateway entry not found from .phconfig!");
      return;
    }
    
  std::string stringDelay = "GPRSMonitoringInterval";
  if(!LoadParameter(stringDelay))
    {
      ERR("CGPRSPlugin::InquiryThread : Can't load parameters, using defaults");
      iInquiryInterval = 10;
    }

  iInquiryInterval = atoi(stringDelay.c_str()) + 1 + (int)(INQUIRY_VARIATION * rand() / (RAND_MAX + 1.0)); 

  stringDelay = "GPRSServiceCheckInterval";
  if(!LoadParameter(stringDelay))
    {
      ERR("CGPRSPlugin::InquiryThread : Can't load parameters, using defaults");
      iServiceCheckInterval = 5;
    }
  iServiceCheckInterval = atoi(stringDelay.c_str());
}

/**
 * @memo Loads a parameter from the configuration file.
 * @doc Loads a parameter from the configuration file. If the requested
 * parameter is found then the argument <code>aParameter</code> is replaced
 * with parameter's value.
 *
 * @param aParameter Parameter's name.
 *
 * @return true if the requested parameter was found
 */
bool CGPRSPlugin::LoadParameter(std::string& aParameter)
{
  char* home;
  std::string line;
  
  if ((home = getenv("HOME")) == NULL) {
    return false;
  }
  
  std::string path(home);
  path.append("/.phconfig");
  
  std::ifstream in(path.c_str());
  
  if (!in.is_open()) {
    ERR("Failed to open the configuration file $HOME/.phconfig");
    in.close();
    return false;
  }
  
  while (!in.eof()) {
    getline(in, line);
    if (line[0] == '#') continue;
    std::string::size_type position = line.find_first_of(':');
    
    if (position != std::string::npos) {
      std::string parameter = line.substr(0, position);
      if (parameter.compare(aParameter) == 0) {
        aParameter = line.substr(++position);
        in.close();
        return true;
      }
    }
  }

  in.close();
  return false;
}

/** 
 * @memo Starts the plugin's inquiry routine.
 * @doc Starts the plugin's inquiry routine. After this method is called the
 * plugin starts to monitor its environment and tries to locate nearby 
 * PeerHood-capable devices. Whenever a change in the neighborhood status is
 * detected the plugin will update the device & service database . This plugin
 * uses protocol defined by virtual peerhood for GPRS devices to locate
 * PeerHood-capable devices in the neighborhood.
 *
 * @return boolean telling if the plugin could be started
 */
bool CGPRSPlugin::Start()
{
  if(iStarted)
    {
      ERR("CGPRSPlugin::Start : inquiry thread already started");
      return false;
    }
  
  iStarted = true;

  if (pthread_create(&iInquiryThread, NULL, *CGPRSPlugin::ThreadStarter, this) != 0) {
    ERR("CGPRSPlugin::Start : failed to create inquiry thread");
    iStarted = false;
    return false;
  }
  
  return true;
}


/** 
 * @memo Stops the plugin's inquiry routine.
 * @doc Stops the plugin's inquiry routine. After this call is handled the 
 * plugin will stop monitoring the environment. This means that daemon's 
 * device & service database will be deprecated as the time passes.
 *
 * @return none
 */
void CGPRSPlugin::Stop()
{
  if (!iStarted)
    {
      DBG("CGPRSPlugin::Stop : inquiry thread not running");
      return;
    }
  
  iStarted = false;

  if (pthread_join(iInquiryThread, NULL) != 0) {
    ERR("CGPRSPlugin::Stop : failed to join the inquiring thread");
    return;
  }

}

/** 
 * @memo Starts advertazing the PeerHood service by providing a lookup service.
 * @doc Starts advertazing the PeerHood service by providing a lookup service.Other GPRS-
 * enabled PeerHood devices will look PeerHood service from this well-known lookup service
 * when they try to locate the surrounding PeerHood devices. If this function fails other devices are
 * not capable of detecting PeerHood on this device.
 *
 * @return true if the advertising was started successfully
 */
bool CGPRSPlugin::Advert()
{  
  if (iInSdp) {
    ERR("CGPRSPlugin::Advert : advert thread already started");
    return false;
  }
  iInSdp = true;

  if (pthread_create(&iAdvertThread, NULL, *CGPRSPlugin::AdvertStarter, this) != 0) {
    ERR("CGPRSPlugin::Advert() : failed to create the advertising thread");
    Unadvert();
    return false;
  }
  return true;
}

/** 
 * @memo Removes the PeerHood tag from the lookup service.
 * @doc Removes the PeerHood tag from the lookup service. After this
 * function is invoked, other devices are not able to detect the existence of
 * the PeerHood service on this machine.
 *
 * @return true if the advertising was ended succesfully
 */
bool CGPRSPlugin::Unadvert()
{
  if (!iInSdp) {
    ERR("CGPRSPlugin::UnAdvert : advert thread not running");
    return false;
  }

  iInSdp = false;

   if (pthread_join(iAdvertThread, NULL) != 0) {
    ERR("CGPRSPlugin::Unadvert : failed to join the advertising thread");
    return false;
  }

  return true;
}

/** 
 * @memo Returns plugin's unique id string.
 * @doc Returns plugin's unique id string. The id of this plugin is 'gprs-base'.
 *
 * @return plugin's unique id string ('gprs-base')
 */
const std::string& CGPRSPlugin::GetName()
{
  return iName;
}

void CGPRSPlugin::UpdateState()
{
	DBG("CGPRSPlugin::UpdateState");
	
	// Check states of the listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		(*iter)->CheckState();
	}
}

void CGPRSPlugin::RegisterListener(MAbstractListener* aListener)
{
	if(aListener != NULL)
	{
		iListenerList.push_back(aListener);
		DBG("CGPRSPlugin::RegisterListener: new listener %s.", aListener->GetName().c_str());
	}
}

void CGPRSPlugin::SetState(bool aActive)
{
	DBG("CGPRSPlugin::SetState: %s", aActive ? "active" : "passive");
}

void CGPRSPlugin::TriggerShutdown()
{
	DBG("CGPRSPlugin::TriggerShutdown");
}

void CGPRSPlugin::LoadListeners()
{
	int listeners = 0;
	std::string listenerName = "gprs";
	listeners = ListenerFactory::GetInstance()->CreateListeners(listenerName, (MAbstractStateConverter*)this);
	if(listeners == 0) DBG("CGPRSPlugin: No listeners for type \"%s\" found.", listenerName.c_str());

	// Connect listeners and check the initial state of the system, go through all listeners
	std::list<MAbstractListener*>::iterator iter;
	for(iter = iListenerList.begin(); iter != iListenerList.end(); ++iter)
	{
		if((*iter)->Connect())
		{
			(*iter)->CheckInitialState();
		}
	}
	DBG("CGPRSPlugin: Listeners started");
}
/** 
 * @memo Starts plugin's main (inquiry) thread.
 * @doc Starts plugin's main (inquiry) thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real inquiry function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CGPRSPlugin::ThreadStarter(void *aArguments)
{
  CGPRSPlugin *fake = (CGPRSPlugin *)aArguments;
  fake->InquiryThread();
  return NULL;
}


/** 
 * @memo Starts plugin's advertising thread. 
 * @doc Starts plugin's advertising thread. This dummy function is required
 * because C++ doesn't allow non-static member functions to be started as a
 * thread. This function just calls plugin instance's real advertising
 * function.
 *
 * @param aArguments Pointer to the currently running instance.
 *
 * @return always NULL
 */
void *CGPRSPlugin::AdvertStarter(void *aArguments)
{
  CGPRSPlugin *fake = (CGPRSPlugin *)aArguments;
  fake->AdvertThread();
  return NULL;
}

/** 
 * @memo Tests TCP connection for incoming data.
 * @doc  Tests TCP connection using ioctl. This method checks how manyt
 * bytes there are available for reading. This checking is necessary because
 * MSG_WAITALL flag is used for reading.
 *
 * @param aFd Socket filedescriptor which is checked
 * @param aBytes How many bytes are expected to be available
 *
 * @return false if there is not enough bytes pending
 */
bool CGPRSPlugin::TestConnection(unsigned int aFd, unsigned int aBytes)
{
  int nPeek = 0;
  int testBuffer = 0;
  nPeek = recv(aFd, &testBuffer, sizeof(testBuffer),  MSG_PEEK);

  unsigned long pending = 0;
  int errvalue = 0;
  if(ioctl(aFd, FIONREAD, &pending) == -1)
    {
      std::cerr << "CGPRSPlugin::TestConnection : ioctl failed " << std::endl;
      return false;
    }
      
  if(pending < aBytes)
    {
      std::cout << "CGPRSPlugin::TestConnection : there is only " << pending << " bytes pending while expecting " << aBytes << std::endl;
      return false;
    }

  return true;
}


/** 
 * @memo Plugin's main thread that performs inquiry at certain intervals.
 * @doc Plugin's main thread that performs inquiry at certain intervals.
 * This inquiry creates TCP-connection to gateway which tries first to acquire
 * unique id number. After id has been received, inquiry starts to inquiry
 * for number of devices and services. Found devices are stored to the DeviceStorage
 * shared with all plugins and the PeerHood Daemon. This whole process is performed
 * continuosly on user-defined interval. If the connection breaks thread tries to
 * reconnect instantly and returns back to it's initial state.
 *
 * @param none
 *
 * @return none
 */
void CGPRSPlugin::InquiryThread()
{
  int state = 0, numberOfDevices = 0, numberOfDevicesReceived = 0, aliveCount = 0, inquiryCount = 0;
  bool firstRun = true;
  fd_set masterSet;
  struct timeval timeVal;
  CTCPConnection signalConnection("NULL", std::string("GPRS"));
  CGPRSSignal tempSignal;
   
  while(iStarted)
    {
      if((iInquiryFailed) || (iAdvertFailed))
	state = 0;
      
      if(state > 0)
     	{
	  FD_ZERO(&masterSet);
	  FD_SET(signalConnection.GetFd(), &masterSet);

	  timeVal.tv_sec = 1;
	  timeVal.tv_usec = 0;
	  
	  if(select(signalConnection.GetFd() + 1, &masterSet, NULL, NULL, &timeVal) ==-1){
	    ERR("CGPRSPlugin::InquiryThread() : select failed");
	    iInquiryFailed = true;
	  }
	  
	  if (FD_ISSET(signalConnection.GetFd(), &masterSet))
	    {
	      if(!DecapsulateSignal(signalConnection.GetFd(), tempSignal))
		{
		  ERR("CGPRSPlugin::InquiryThread() : Decapsulate failed");
		  iInquiryFailed = true;
		}
	      
	      switch(tempSignal.iMessageType)
		{
		case GETSERV:
		  {
		    CGPRSSignal sendSignal;
		    sendSignal.iMessageType = GETSERVREP;
		    sendSignal.iOriginalId = iId;
		    sendSignal.iData = 0;
		    sendSignal.iData2 = 0;
		    sendSignal.iServiceName = "";
		    
		    if(!EncapsulateSignal(signalConnection.GetFd(), sendSignal))
		      {
			ERR("CGPRSPlugin::InquiryThread() : GETSERVREP Encapsulate failed");
			iInquiryFailed = true;
			break;
		      }
		    
		    if(!CDaemon::GetInstance()->SendServiceList(&signalConnection))
		      {
			ERR("CGPRSPlugin::InquiryThread : SendServiceList failed");
			iInquiryFailed = true;
			break;
		      }

		    aliveCount = 0;
		    break;
		  }
		  
		case PROBE:
		  {
		    CGPRSSignal sendSignal;
		    sendSignal.iMessageType = REPPROBE;
		    sendSignal.iOriginalId = iId;
		    sendSignal.iData = 0;
		    sendSignal.iData2 = 0;
		    sendSignal.iServiceName = "";
		    
		    if(!EncapsulateSignal(signalConnection.GetFd(), sendSignal))
		      {
			ERR("CGPRSPlugin::InquiryThread() : REPPROBE Encapsulate failed");
			iInquiryFailed = true;
			break;
		      }
		    aliveCount = 0;
		    break;
		  }
		  
		case REPID:
		  {
		    iId = tempSignal.iOriginalId;
		    state = 5;
		    aliveCount = 0;
		    break;
		  }

		case REQ:
		  {
		    //start inquiry if inquiryCount has reached
		    if(inquiryCount > iInquiryInterval)
		      {
			state = 2;
			inquiryCount = 0;
		      }
		    else
		      //else just release request
		      state = 4;
		    
		    aliveCount = 0;
		    break;
		  }
		  
		case REPDEV:
		  {
		    numberOfDevicesReceived = 0;
		    numberOfDevices = 0;
		    iInquiryList.clear();
		    numberOfDevices = tempSignal.iData;

		    if(numberOfDevices == 0)
		      {
			//finish inquiry
			DBG("CGPRSPlugin::InquiryThread() : there are no devices");
			state = 4;
		      }
		    else
		      //wait for devices, inquiry is still running
			state = 5;
		    aliveCount = 0;
		    break;
		  }
		  
		case DEVICE:
		  {
		    numberOfDevicesReceived++;
		    CGPRSInquiry *tempInquiry = new CGPRSInquiry;
		    tempInquiry->iDeviceId = tempSignal.iData;
		    tempInquiry->iServices = 0;
		    tempInquiry->iServicesReceived = 0;
		    tempInquiry->iShouldCheck = false;
		    tempInquiry->iDeviceChecked = false;
		    iInquiryList.push_back(tempInquiry);
		    
		    if(numberOfDevicesReceived == numberOfDevices)
		      {
			DBG("CGPRSPlugin::InquiryThread() : All devices received");
			//check services
			state = 3;
		      }
		    else
		      //wait for more devices
		      state = 5;
		    
		    //in both cases, inquiry is still running
		    if(AddDevice(tempSignal.iData))
		      DBG("CGPRSPlugin::InquiryThread() : added new device");
		    aliveCount = 0;
		    break;
		  }
		  
		case DEVICEINFO:
		  {
		    std::ostringstream tempName;
		    tempName << "GPRS" << tempSignal.iOriginalId;
		    std::list<CDaemonDevice *>::iterator j;
		    for (j = iDeviceList.begin();j != iDeviceList.end();++j) {
		      if ((*j)->GetAddress().compare(tempName.str()) == 0)
			{
			  (*j)->SetDeviceName(tempSignal.iServiceName);
			  (*j)->SetChecksum(ntohl(tempSignal.iData));
			  break;
			}
		    } 
		    
		    //wait for number of services inquiry is still running
		    state = 5;
		    aliveCount = 0;
		    break;
		  }

		case REPSERV:
		  {
		    numberOfDevicesReceived = 0;
		    numberOfDevices = 0;

		    for (std::list<CGPRSInquiry *>::iterator j = iInquiryList.begin();j != iInquiryList.end();++j) {
		      if((*j)->iDeviceId == tempSignal.iOriginalId)
			{
			  (*j)->iServices = tempSignal.iData;
			  if(tempSignal.iData == 0)
			    {
			      DBG("CGPRSPlugin::InquiryThread() : this device has no services");
			      //no more services for this device, query next device for services
			      (*j)->iDeviceChecked = true;
			      state = 3;
			    }
			  else
			    {
			      DBG("CGPRSPlugin::InquiryThread() : this device has services");
			      //wait for services
			      state = 5;
			    }
			  break;
			}
		    }
		    //in both cases, inquiry is still running
		    aliveCount = 0;
		    break;
		  }

		case SERVICE:
		  {
		    int testBuffer = 0;
		    int nPeek = 0;
		   
		    if(!TestConnection(signalConnection.GetFd(), tempSignal.iData))
		      {
			ERR("CGPRSPlugin::InquiryThread() : TestConnection failed");
			iInquiryFailed = true;
			break;
		      }
		    
		    char *buffer = new char[tempSignal.iData];
		    memset(buffer, 0, tempSignal.iData);
		    
		    if(recv(signalConnection.GetFd(), buffer, tempSignal.iData, MSG_WAITALL) <= 0) {
		      ERR("CGPRSPlugin::InquiryThread() : failed to read service info");
		      delete[] buffer;
		      iInquiryFailed = true;
		      break;
		    }
		    
		    CService* tempService;
		    tempService = new CService(buffer);
		    DBG("  the service is : %s", tempService->GetName().c_str());
		    DBG("  attributes are : %s", tempService->GetAttributeString().c_str());
		    delete[] buffer;
		    
		    CGPRSPlugin::AddService(tempSignal.iOriginalId, tempService);
		   		    
		    for (std::list<CGPRSInquiry *>::iterator j = iInquiryList.begin();j != iInquiryList.end();++j) {
		      if((*j)->iDeviceId == tempSignal.iOriginalId)
			{
			  (*j)->iServicesReceived++;
			  if((*j)->iServicesReceived == (*j)->iServices)
			    {
			      DBG("CGPRSPlugin::InquiryThread() : all the services received");
			      (*j)->iDeviceChecked = true;
			    }
			  break;
			}
		    }
		    //query next device for services, inquiry is still runnning
		    state = 3;
		    aliveCount = 0;
		    break;
		  }
		  
		}// switch(tempSignal.iMessageType)
	    }//  if (FD_ISSET(signalConnection.GetFd(), &masterSet))
	}

      switch(state) {
	
	//connection broken and initial state
      case 0:
	{
	  aliveCount = 0;
	   if(signalConnection.IsConnected())
	     signalConnection.Disconnect();

	  if(!firstRun)
	    {
	      iId = 0;
	      
	      if(iInSdp)
		Unadvert();

	      CTCPConnection signalConnection("NULL", std::string("GPRS"));
	    }
	  else
	    firstRun = false;

	  //create a new connection
	  if (!signalConnection.Connect(iGateway, GATEWAY)){
	    ERR(" CGPRSPlugin::InquiryThread :failed to connect");
	    iInquiryFailed = true;
	    sleep(1);
	    break;
	  }
	  
	  //start advertThread if not running
	  if(!iInSdp)
	    Advert();
	  
	  iAdvertFailed = false;
	  iInquiryFailed = false;

	  FD_ZERO(&masterSet);
	  FD_SET(signalConnection.GetFd(), &masterSet);
	   
	  state = 1;
	  break;
	}
	
	//id request state
      case 1:
	{
	  tempSignal.iMessageType = GETID;
	  tempSignal.iOriginalId = 0;
	  tempSignal.iData = 0;
	  tempSignal.iData2 = 0;
	  tempSignal.iServiceName = "";
	  
	  if(!EncapsulateSignal(signalConnection.GetFd(), tempSignal))
	    {
	      ERR("CGPRSPlugin::InquiryThread() : GETID Encapsulate failed");
	      iInquiryFailed = true;
	      break;
	    }
	  //go to idle state
	  state = 5;
	  break;
	}
	
	//device request state
      case 2:
	{
	  tempSignal.iMessageType = GETDEV;
	  tempSignal.iOriginalId = iId;
	  tempSignal.iData = 0;
	  tempSignal.iData2 = 0;
	  tempSignal.iServiceName = "";
	  
	  if(!EncapsulateSignal(signalConnection.GetFd(), tempSignal))
	    {
	      ERR("CGPRSPlugin::InquiryThread() : GETDEV Encapsulate failed");
	      iInquiryFailed = true;
	      break;
	    }
	  //go to idle state
	  state = 5;
	  break;
	}

	//service request state
      case 3:
	{
	  bool found = false;
	  for (std::list<CGPRSInquiry *>::iterator j = iInquiryList.begin();j != iInquiryList.end();++j) {
	    if(((*j)->iShouldCheck) && (!(*j)->iDeviceChecked))
	      {
		found = true;
		tempSignal.iMessageType = GETSERV;
		tempSignal.iOriginalId = iId;
		tempSignal.iData = (*j)->iDeviceId;
		tempSignal.iData2 = 0;
		tempSignal.iServiceName = "";
		
		if(!EncapsulateSignal(signalConnection.GetFd(), tempSignal))
		  {
		    ERR("CGPRSPlugin::InquiryThread() : GETDEV Encapsulate failed");
		    iInquiryFailed = true;
		    break;
		  }
		//go to idle state
		state = 5;
		break;
	      }
	  }
	  
	  if(!found)
	    {
	      DBG("CGPRSPlugin::InquiryThread() : all the devices and services are now checked");
	      //finish inquiry
	      state = 4;
	      CDeviceStorage::GetInstance()->Update(PLUGIN_NAME, iDeviceList);
	    }
	  break;
	}
	
	//request complete
      case 4:
	{
	  tempSignal.iMessageType = REQCOMPLETE;
	  tempSignal.iOriginalId = iId;
	  tempSignal.iData = 0;
	  tempSignal.iData2 = 0;
	  tempSignal.iServiceName = "";
	  
	  if(!EncapsulateSignal(signalConnection.GetFd(), tempSignal))
	    {
	      ERR("CGPRSPlugin::InquiryThread() : REQCOMPLETE Encapsulate failed");
	      iInquiryFailed = true;
	      break;
	    }

	  //go to idle state
	  CGPRSPlugin::MakeOlder();
	  state = 5;
	  break;
	}

	//idle state
      case 5:
	{
	  if(aliveCount > DISCONNECT_TIMEOUT)
	    {
	      DBG("CGPRSPlugin::InquiryThread() : Gateway has disconnect. Trying to reconnect...");
	      iInquiryFailed = true;
	    }
	  else
	    aliveCount++;

	  inquiryCount++;
	  
	  break;
	}
	
      }//switch(state)
    }//while (iStarted)
  DBG("CGPRSPlugin::InquiryThread() : loop exited");
}
/** 
 * @memo Checks if the found device is new (i.e. not detected before).
 * @doc Checks if the found device is new (i.e. not detected before). If this
 * is the case then it is added to plugin's internal device list and the common
 * device & service database should be updated later on.
 * There two kinds of counters for device: timestamp is used for to mark "old" devices
 * and lastchecked is for performing service checking on specific interval.
 * timestamp is zerod every time when device is found. If timestamp reaches specific value
 * device is erased. If device is seen, lastchecked value is decreased.  If lastchecked
 * reaches zero, device services are checked.
 * 
 * @param aId The ID of the device to be added.
 *
 * @return true if the checked device is new
 */
bool CGPRSPlugin::AddDevice(unsigned int aId)
{
   std::string deviceName;
   char ctemp[100];
   sprintf(ctemp, "GPRS%d", aId);
   deviceName = ctemp;

   for (std::list<CDaemonDevice *>::iterator j = iDeviceList.begin();j != iDeviceList.end();++j) {
     if ((*j)->GetAddress().compare(deviceName) == 0) {
       (*j)->SetTimestamp(0);
       DBG("Timestamp is zerod");
       
       for (std::list<CServiceCheck *>::iterator k = iServiceCheckList.begin();k != iServiceCheckList.end();++k) {
	 if(deviceName.compare((*k)->iDeviceName) == 0)
	   {
	     if((*k)->iLastChecked > iServiceCheckInterval)
	       (*k)->iLastChecked = iServiceCheckInterval;
	     
	     if((*k)->iLastChecked > 0)
	       (*k)->iLastChecked--;
	     
	     if((*k)->iLastChecked == 0)
	       {
		 for (std::list<CGPRSInquiry *>::iterator j = iInquiryList.begin();j != iInquiryList.end();++j) {
		   if((*j)->iDeviceId == aId)
		     {
		       DBG("Marked to be checked now");
		       (*j)->iShouldCheck = true;
		       break;
		     }
		 }
		 (*k)->iLastChecked = iServiceCheckInterval;
	       }
	     return false;
	   }
       }
     }
   }
   
   
   CDaemonDevice* temp = new CDaemonDevice(deviceName);
   temp->SetPrototype(PLUGIN_NAME);
   temp->SetPeerHood(true);
   iDeviceList.push_back(temp);
   for (std::list<CGPRSInquiry *>::iterator j = iInquiryList.begin();j != iInquiryList.end();++j) {
     if((*j)->iDeviceId == aId)
       {
	  DBG("Marked to be checked now");
	  (*j)->iShouldCheck = true;
	  break;
       }
   }
     
#warning "memory leak?"
   CServiceCheck *tempCheck = new CServiceCheck;
   tempCheck->iDeviceName = std::string(deviceName);
   tempCheck->iLastChecked = iServiceCheckInterval;
   iServiceCheckList.push_back(tempCheck);
   
   return true;
}

/** 
 * @memo Increases each device's timestamp by one and checks is some of them
 * exceeds the limit.
 * @doc Increases each device's timestamp value by one and checks if some of
 * them exceeds the limit defined byt KEEP_ALIVE_STAMP. In that case the "old"
 * device is removed from
 * the internal device list so the common device & service database should be
 * updated.
4 *
 * @return true if some device was removed from the internal device list
 */
bool CGPRSPlugin::MakeOlder()
{
  bool retval = false;
  std::string deviceName;

  for (std::list<CDaemonDevice *>::iterator i = iDeviceList.begin();i != iDeviceList.end();) {
    if ((*i)->GetTimestamp() > KEEP_ALIVE_STAMP ) {
      deviceName = (*i)->GetAddress();
      DBG("removing a device from the internal database");

      std::list<CDaemonDevice *>::iterator sentenced = i++;
      delete *sentenced;
      iDeviceList.erase(sentenced);
      retval = true;
      
      for (std::list<CServiceCheck *>::iterator j = iServiceCheckList.begin();j != iServiceCheckList.end();++j) {
	if(deviceName.compare((*j)->iDeviceName) == 0)
	  {
	    DBG("removing device from service check database");
	    std::list<CServiceCheck *>::iterator sentenced = j++;
	    delete *sentenced;
	    iServiceCheckList.erase(sentenced);
	    
	    break;
	  }
      }
    }
    else {
      (*i)->IncreaseTimestamp();
      i++;
    }
    
  }

  return retval;
}

/** 
 * @memo Adds services to internal devicelist
 * @doc Adds services to internal devicelist. Method updates devicelist
 * by adding services for specific device.
 *
 * @param aId Device ID
 * @param aService Service which is added
 * @return true if some device was removed from the internal device list
 */
bool CGPRSPlugin::AddService(unsigned int aId, CService* aService)
{
  std::string deviceName;
  char ctemp[100];
  
  sprintf(ctemp, "GPRS%d", aId);
  deviceName = std::string(ctemp);
 
  std::list<CDaemonDevice *>::iterator i;
  for (i = iDeviceList.begin();i != iDeviceList.end();++i) {
    if ((*i)->GetAddress().compare(deviceName) == 0)
      {
	if(!((*i)->HasService(aService->GetName())))
	  {
	    (*i)->AddService(aService);
	    return true;
	  }
	else
	  DBG("CGPRSPlugin::AddService : service %s is already listed", aService->GetName().c_str());

      }
  }
  return false;
}

/** 
 * @memo Listens for incoming service list requests and routes data to service.
 * @doc Listens for incoming service list requests and routes data to service.
 * Function serves for incoming service list requests by sending
 * all the local registered services. Function also provides routing
 * for data between local services and GPRS-gateway.
 *
 * @return none
 */
void CGPRSPlugin::AdvertThread()
{
  fd_set masterSet, tempSet;
  std::string serviceName;
  unsigned int maxFd = 0;
  struct timeval timeVal;

  iAdvertFailed = false;
  
  //stay on this loop until inquiryThread gets proper id
  while(iId <= 0)
    {
      sleep(1);
      
      //return if unadvert want's to end thread
      if(!iInSdp)
	return;
      
    }
 
  CTCPConnection *dataConnection = new CTCPConnection("NULL", "GPRS");
  CGPRSConnection *localConnection = new CGPRSConnection;
  
  //listen for local-connections
  if(!localConnection->Listen(0)) {
    ERR("CGPRSPlugin::AdvertThread() : local connection listening failed");
    iAdvertFailed = true;
    delete dataConnection;
    delete localConnection;
    return;
  }

  //connect to GPRS-gateway
  if (!dataConnection->Connect(iGateway, GATEWAY+1)){
    ERR(" CGPRSPlugin::AdvertThread :failed to connect");
    iAdvertFailed = true;
    delete dataConnection;
    delete localConnection;
    return;
  }
 
  CGPRSSignal tempDataSignal;
  tempDataSignal.iMessageType = ADVERT;
  tempDataSignal.iOriginalId = iId;
  tempDataSignal.iData = 0;
  tempDataSignal.iData2 = 0;

  if(!EncapsulateSignal(dataConnection->GetFd(), tempDataSignal))
    {
      ERR("CGPRSPlugin::AdvertThread() : Encapsulate failed");
      iAdvertFailed = true;
      dataConnection->Disconnect();
      delete dataConnection;
      delete localConnection;
      return;
    } 

  FD_ZERO(&masterSet);
  FD_ZERO(&tempSet);
  
  FD_SET(localConnection->GetFd(), &masterSet);
  FD_SET(dataConnection->GetFd(), &masterSet);
  
  if (dataConnection->GetFd() > (int)maxFd)
    maxFd = dataConnection->GetFd();
 
  if (localConnection->GetFd() > (int)maxFd)
    maxFd = localConnection->GetFd();
  
  while(iInSdp) {  
  
    tempSet = masterSet;

    timeVal.tv_sec = 10;
    
    if(select(maxFd + 1, &tempSet, NULL, NULL, &timeVal) ==-1){
      ERR("CGPRSPlugin::AdvertThread() : select failed");
      iAdvertFailed = true;
      dataConnection->Disconnect();
      delete dataConnection;
      delete localConnection;
      return;
    }
       
    for(unsigned int i=0;i<=maxFd;i++){
      if (FD_ISSET(i, &tempSet))
	{
	  //exiting
	  if(i==0)
	    {
	      iInSdp = false;
	      dataConnection->Disconnect();
	      delete dataConnection;
	      delete localConnection;
	      break;
	    }
	  
	  //new local connection
	  else if(i==(unsigned int)localConnection->GetFd())
	    {
	      if(!iInSdp)
		break;
	      
	      CGPRSConnection *newLocalConnection;
	      if(!(newLocalConnection = (CGPRSConnection *)localConnection->AcceptL()))
		{
		  ERR("CGPSPlugin::AdvertThread() : AcceptL failed");
		  iAdvertFailed = true;
		  dataConnection->Disconnect();
		  delete dataConnection;
		  delete localConnection;
		  return;
		}
	      else
		{
		  if(!StartClientDataThread(newLocalConnection))
		    {
		      ERR("CGPSPlugin::AdvertThread() : can't create a thread");
		      iAdvertFailed = true;
		      dataConnection->Disconnect();
		      delete dataConnection;
		      delete localConnection;
		      return;
		    }
		}
	    }
	  //old connections 
	  else
	    {
	      //incoming from GPRS-gateway
	      if(i == (unsigned int)dataConnection->GetFd())
		{
		  CGPRSSignal tempDataSignal;
		  if(!DecapsulateSignal(i, tempDataSignal))
		    {
		      ERR("CGPRSPlugin::AdvertThread() : Decapsulate failed");
		      iAdvertFailed = true;
		      dataConnection->Disconnect();
		      delete dataConnection;
		      delete localConnection;
		      return;
		    }
		  
		  switch(tempDataSignal.iMessageType)
		    {
		    case CONNECT:
		      {
			if(!StartServerDataThread(tempDataSignal.iOriginalId, tempDataSignal.iServiceName, tempDataSignal.iData2))
			  {
			    ERR("CGPSPlugin::AdvertThread() : can't create a thread");
			    iAdvertFailed = true;
			    dataConnection->Disconnect();
			     delete dataConnection;
			     delete localConnection;
			    return;
			  }
			break;
		      }

		    case PROBE:
		      {
			 CGPRSSignal sendSignal;
			 sendSignal.iMessageType = REPPROBE;
			 sendSignal.iOriginalId = iId;
			 sendSignal.iData = 0;
			 sendSignal.iData2 = 0;
			 sendSignal.iServiceName = "";
			 
			 if(!EncapsulateSignal(dataConnection->GetFd(), sendSignal))
			   {
			     ERR("CGPRSPlugin::AdvertThread() : REPPROBE Encapsulate failed");
			     iAdvertFailed = true;
			     dataConnection->Disconnect();
			     delete dataConnection;
			     delete localConnection;
			     return;
			   }
		
			 break;
		      }
		      
		    }
				  
		}// if(i == (unsigned int)dataConnection.GetFd())
	    }//else (old connections)
	}//if (FD_ISSET(i, &tempSet))
    }//for(unsigned int i=0;i<=maxFd;i++)
  }// while(iInSdp)
  dataConnection->Disconnect();
  delete dataConnection;
  delete localConnection;
}

/**
 * @memo Receives and decapsulates metadata
 * @doc Receives and decapsulates metadata.
 * First data length is received
 * and after that, the whole data.
 *
 * @param aSourceFd Source socket filedescriptor
 * @param aSignal Container for received message
 *
 * @return true if succesful
 */
bool CGPRSPlugin::DecapsulateSignal(int aSourceFd, CGPRSSignal& aSignal)
{
  int bufferSize = 0;
  char *buffer;
  fd_set testSet;
  struct timeval timeVal;

  timeVal.tv_sec = 3;
  timeVal.tv_usec = 0;
  
  FD_ZERO(&testSet);
  FD_SET(aSourceFd, &testSet);

  if(select(aSourceFd + 1, &testSet, NULL, NULL, &timeVal) ==-1){
    ERR("CGPRSPlugin::DecapsulateSignal : select failed");
    return false;
  }

  if(FD_ISSET(aSourceFd, &testSet))
    {
      if(!TestConnection(aSourceFd, sizeof(bufferSize)))
	{
	  ERR("CGPRSPlugin::DecapsulateSignal : TestConnection failed");
	  return false;
	}
      
      //receive data length
      if(recv(aSourceFd, &bufferSize, sizeof(bufferSize), MSG_WAITALL)<=0)
	{
	  DBG("CGPRSPlugin::DecapsulateSignal : receiving data length failed");
	  return false;
	}
      
      bufferSize = ntohl(bufferSize);
      
      if(FD_ISSET(aSourceFd, &testSet))
	{  
	  if(!TestConnection(aSourceFd, bufferSize))
	    {
	      ERR("CGPRSPlugin::DecapsulateSignal : TestConnection failed");
	      return false;
	    }
	  
	  buffer = new char[bufferSize];
	  memset(buffer, 0, bufferSize);
	  
	  //receive data
	  if(recv(aSourceFd, buffer, bufferSize, MSG_WAITALL)<=0)
	    {
	      DBG("CGPRSPlugin::DecapsulateSignal : receiving data failed");
	      delete[] buffer;
	      return false;
	    }
	  
	  memcpy(&aSignal.iMessageType, &buffer[0], sizeof(int));
	  memcpy(&aSignal.iOriginalId, &buffer[sizeof(int)], sizeof(int));
	  memcpy(&aSignal.iData, &buffer[2*sizeof(int)], sizeof(int));
	  memcpy(&aSignal.iData2, &buffer[3*sizeof(int)], sizeof(int));
	  aSignal.iServiceName = std::string(&buffer[4*sizeof(int)]);
	  delete[] buffer;
	  return true;
	}
    }

  return false;
}

/**
 * @memo Encapsulates and sends metadata
 * @doc Encapsulates and sends metadata.
 * First data length is sent
 * and after that, the whole data.
 *
 * @param aTargetFd target filedescpriptor
 * @param aSignal Signal which is wanted to be sent
 * @return true if succesful
 */
bool CGPRSPlugin::EncapsulateSignal(int aTargetFd, CGPRSSignal& aSignal)
{
  int bufferSize = 0;
  char *buffer;
  bufferSize = 4*sizeof(int);
  bufferSize += aSignal.iServiceName.length()+1;
  
  buffer = new char[bufferSize];
  memset(buffer, 0, bufferSize);
  memcpy(&buffer[0], &aSignal.iMessageType, sizeof(int));
  memcpy(&buffer[sizeof(int)], &aSignal.iOriginalId, sizeof(int));
  memcpy(&buffer[2*sizeof(int)], &aSignal.iData, sizeof(int));
  memcpy(&buffer[3*sizeof(int)], &aSignal.iData2, sizeof(int));
  strcpy(&buffer[4*sizeof(int)], aSignal.iServiceName.c_str());
  strcat(buffer, "\0");
  
  //send length
  bufferSize = htonl(bufferSize);
  if(write(aTargetFd, &bufferSize, sizeof(bufferSize)) <= 0){
    ERR("CGPRSPlugin::EncapsulateSignal : failed to send length");
    delete[] buffer;
    return false;
  }
  bufferSize = ntohl(bufferSize);

  DBG("CGPRSPlugin::EncapsulateSignal : sending %d bytes", sizeof(bufferSize));

  //send data
  if(write(aTargetFd, buffer, bufferSize) <= 0){
    ERR("CGPRSPlugin::EncapsulateSignal : failed to send");
    delete[] buffer;
    return false;
  }

  delete[] buffer;
  return true;
}

/**
 * @memo Fetches local services from daemon
 * @doc Fetches local services from daemon. Function 
 * queries all found local services and searches service
 * by given aSeviceName. User is responsible to free
 * allocated memory for aService class.
 *
 * @param aService class for receicing found service entry
 * @param aServiceName servicaneme to search for
 *
 * @return true if succesful
 */
bool CGPRSPlugin::GetLocalServicesL(CService *&aService, std::string& aServiceName)
{
  char* buffer = NULL;
  int numberOfServices = 0, length = 0;
  TCommand pdu;
  bool found = false;

  CLocalConnection *tempConnection = new CLocalConnection();
  
  //create a new connection to daemon
  if (!tempConnection->Connect(LOCAL_SOCKET_NAME, 0)){
    ERR(" CGPRSPlugin::GetLocalServices() : Connect failed");
    return false;
  }

  pdu.iCommand = PH_INIT;
  pdu.iPid = 0;
  
  if (tempConnection->Write(&pdu, sizeof(pdu)) <=0) {
    ERR("CGPRSPlugin::GetLocalServices() : writing the PH_INIT command");
    delete tempConnection;
    return false;
  }
  
  pdu.iCommand = PH_GET_LOCAL_SERVICELIST;
  pdu.iPid = 0;
   
  if(tempConnection->Write(&pdu, sizeof(pdu)) <=0)
    {
      ERR(" CGPRSPlugin::GetLocalServices() : pdu write failed");
      delete tempConnection;
      return false;
    }
  
  if (tempConnection->Read(&numberOfServices, sizeof(numberOfServices)) <=0) {
    ERR(" CGPRSPlugin::GetLocalServices() : numberOfServices read failed");
    delete tempConnection;
    return false;
  }
  

  // read all devices
  for (int i = 0;i < numberOfServices;i++) {
    
    if (tempConnection->Read(&length, sizeof(length)) <=0) {
      ERR(" CGPRSPlugin::GetLocalServices() : service length read failed");
      delete tempConnection;
       return false;
    }
            
    buffer = new char[length];
    
    if (tempConnection->Read(buffer, length) <=0) {
      ERR(" CGPRSPlugin::GetLocalServices() : service read failed");
      delete[] buffer;
      delete tempConnection;
      return false;
    }
    
    
    if (!found)
      {
	aService = new CService(buffer, "local");
	if(aServiceName == aService->GetName())
	  found = true;
      }
    delete[] buffer;
   
  }
  tempConnection->Disconnect();
  delete tempConnection;
  
  if (found)
    return true;
  
  ERR(" CGPRSPlugin::GetLocalServices() : Can't find pid for service name=%s", aServiceName.c_str());
  return false;
}

/**
 * @memo Starts ClientData -thread.
 * @doc Starts ClientData -thread. This ClientData -thread creates a new Data Connection
 * to the GPRS Gateway. ClientData -thread negoatiates a new Data Connection between 
 * GPRS Gateway and target device and starts to route data between localsocket -connection
 * from application layer and TCP connection to the GPRS Gateway.
 *
 * @param aLocalConnection LocalConnection from the application layer.
 *
 * @return true if succesful
 */
bool CGPRSPlugin::StartClientDataThread(CGPRSConnection* aLocalConnection)
{
  pthread_t tempThread;
  CGPRSThread *tempThreadInfo = new CGPRSThread;
  
  tempThreadInfo->iInstance = this;
  tempThreadInfo->iThread = tempThread;
  tempThreadInfo->iTargetId = 0;
  tempThreadInfo->iCommand = 0;
  tempThreadInfo->iServiceName = "";
  tempThreadInfo->iLocalConnection = aLocalConnection;

  if (pthread_create(&tempThread, NULL, *CGPRSPlugin::ClientDataThreadStarter, (CGPRSThread *)tempThreadInfo) != 0) {
    ERR("CGPRSPlugin::StartDataThread : failed to create the advertising thread");
    delete tempThreadInfo;
    return false;
  }
  pthread_detach(tempThread);
  return true;
}
/**
 * @memo ClientData -thread starter
 * @doc ClientData -thread starter
 *
 * @param aArguments Arguments for the thread
 *
 * @return true if succesful
 */
void *CGPRSPlugin::ClientDataThreadStarter(void *aArguments)
{
  CGPRSThread *tempThreadInfo = (CGPRSThread *)aArguments;
  CGPRSPlugin *fake = (CGPRSPlugin *)tempThreadInfo->iInstance;
  fake->ClientDataThread((*tempThreadInfo));
  delete tempThreadInfo;
  return NULL;
}

/**
 * @memo Implementation of ClientData -thread
 * @doc Implementation of ClientData -thread. Negoatiates a new Data Connection
 * between GPRS Gateway and target device. ClientData -thread routes data between
 * local connection and established Data Connection.
 *
 * @param aThreadInfo Reference to ThreadInfo class
 *
 * @return true if succesful
 */
void CGPRSPlugin::ClientDataThread(CGPRSThread& aThreadInfo)
{
  fd_set dataSet;
  struct timeval timeVal;
  unsigned int maxFd = 0, state = 0, targetId = 0;;
  u_int8_t command = 0;
  char nameBuf[256];

  CTCPConnection *dataConnection = new CTCPConnection("NULL", "GPRS");
  
  while(iInSdp)
    {
      switch(state)
	{
	case 0:
	  {
	    //receive target id from local service
	    if(read(aThreadInfo.iLocalConnection->GetFd(), &targetId, sizeof(targetId)) == -1){
	      ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving TargetId");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	    }

	    //receive command PH_DIRECT_DATA from local service
	    int realLength = 0;
	    if(read(aThreadInfo.iLocalConnection->GetFd(), &realLength, sizeof(realLength)) ==-1){
	      ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving command");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	    }
	    if(read(aThreadInfo.iLocalConnection->GetFd(), &command, realLength) ==-1){
	      ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving command");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	    }
	    
	    //receive service name
	    if(read(aThreadInfo.iLocalConnection->GetFd(), &realLength, sizeof(realLength)) ==-1){
	      ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving command");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	    }
	    
	    if(read(aThreadInfo.iLocalConnection->GetFd(), nameBuf, realLength) <=0){
	      ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving service name");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	     }

	    state = 1;
	    break;
	  }
	  
	case 1:
	  {
	    //connect to GPRS-gateway
	    if(!dataConnection->Connect(iGateway, GATEWAY+1)){
	      ERR("CGPRSPlugin::ClientDataThread : failed to connect");
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete dataConnection;
	      delete aThreadInfo.iLocalConnection;
	      return;
	    }
	    
	    CGPRSSignal tempDataSignal;
	    tempDataSignal.iMessageType = CONNECT;
	    tempDataSignal.iOriginalId = iId;
	    tempDataSignal.iData = targetId;
	    tempDataSignal.iData2 = command;
	    tempDataSignal.iServiceName = std::string(nameBuf);
	    
	    if(!EncapsulateSignal(dataConnection->GetFd(), tempDataSignal))
	      {
		ERR("CGPRSPlugin::ClientDataThread() : Encapsulate failed");
		dataConnection->Disconnect();
		aThreadInfo.iLocalConnection->Disconnect();
		delete dataConnection;
		delete aThreadInfo.iLocalConnection;
		return;
	      }

	    FD_ZERO(&dataSet);
	    FD_SET(dataConnection->GetFd(), &dataSet);
	    
	    timeVal.tv_sec = 10;
	    timeVal.tv_usec = 0;
		
	    if(select(dataConnection->GetFd() + 1, &dataSet, NULL, NULL, &timeVal) ==-1){
	      ERR("CGPRSPlugin::ClientDataThread : select failed");
	      dataConnection->Disconnect();
	      aThreadInfo.iLocalConnection->Disconnect();
	      delete aThreadInfo.iLocalConnection;
	      delete dataConnection;
	      return;
	    }
	    
	    //receive PROBE
	     if(FD_ISSET(dataConnection->GetFd(), &dataSet))
	      {
		CGPRSSignal tempDataSignal;
		if(!DecapsulateSignal(dataConnection->GetFd(), tempDataSignal))
		  {
		    ERR("CGPRSPlugin::ClientDataThread() : receiving PROBE failed");
		    dataConnection->Disconnect();
		    aThreadInfo.iLocalConnection->Disconnect();
		    delete aThreadInfo.iLocalConnection;
		    delete dataConnection;
		    return;
		  }

		if(tempDataSignal.iMessageType == PROBE)
		  {
		    tempDataSignal.iMessageType = REPPROBE;
		    tempDataSignal.iOriginalId = iId;
		    tempDataSignal.iData = 0;
		    tempDataSignal.iData2 = 0;
		    tempDataSignal.iServiceName = "";
		    
		    if(!EncapsulateSignal(dataConnection->GetFd(), tempDataSignal))
		      {
			ERR("CGPRSPlugin::ClientDataThread() : REPPROBE Encapsulate failed");
			dataConnection->Disconnect();
			aThreadInfo.iLocalConnection->Disconnect();
			delete dataConnection;
			delete aThreadInfo.iLocalConnection;
			return;	
		      }
		  }
		else
		  {
		    ERR("CGPRSPlugin::ClientDataThread() : expected PROBE, received %d", tempDataSignal.iMessageType);
		    dataConnection->Disconnect();
		    aThreadInfo.iLocalConnection->Disconnect();
		    delete dataConnection;
		    delete aThreadInfo.iLocalConnection;
		    return;	
		  }
	      }
	    else
	      {
		ERR("CGPRSPlugin::ClientDataThread() : timeout while waiting PROBE");
		dataConnection->Disconnect();
		aThreadInfo.iLocalConnection->Disconnect();
		delete dataConnection;
		delete aThreadInfo.iLocalConnection;
		return;
	      }

	     //receive REPCONNECT
	    if(FD_ISSET(dataConnection->GetFd(), &dataSet))
	      {
		CGPRSSignal tempDataSignal;
		if(!DecapsulateSignal(dataConnection->GetFd(), tempDataSignal))
		  {
		    ERR("CGPRSPlugin::ClientDataThread() : receiving REPCONNECT failed");
		    dataConnection->Disconnect();
		    aThreadInfo.iLocalConnection->Disconnect();
		    delete aThreadInfo.iLocalConnection;
		    delete dataConnection;
		    return;
		  }

		if(tempDataSignal.iMessageType == REPCONNECT)
		  {
		    command = PH_OK;
		    if(write(aThreadInfo.iLocalConnection->GetFd(), &command, sizeof(command)) == -1){
		      ERR("CGPRSPlugin::ClientDataThread() : sending nameBuf failed");
		      dataConnection->Disconnect();
		      aThreadInfo.iLocalConnection->Disconnect();
		      delete dataConnection;
		      delete aThreadInfo.iLocalConnection;
		      return;
		    }
		    state = 2;
		  }
		else
		  {
		    ERR("CGPRSPlugin::ClientDataThread() : expected REPCONNECT, received %d", tempDataSignal.iMessageType);
		    dataConnection->Disconnect();
		    aThreadInfo.iLocalConnection->Disconnect();
		    delete dataConnection;
		    delete aThreadInfo.iLocalConnection;
		    return;	
		  }
	      }
	    else
	      {
		ERR("CGPRSPlugin::ClientDataThread() : timeout while waiting REPCONNECT");
		dataConnection->Disconnect();
		aThreadInfo.iLocalConnection->Disconnect();
		delete dataConnection;
		delete aThreadInfo.iLocalConnection;
		return;
	      }
	    break;
	  }
	  
	case 2:
	  {
	   	    
	    while(iInSdp)
	      {
		FD_ZERO(&dataSet);
		FD_SET(aThreadInfo.iLocalConnection->GetFd(), &dataSet);
		FD_SET(dataConnection->GetFd(), &dataSet);
		
		if ( aThreadInfo.iLocalConnection->GetFd() > (int)maxFd)
		  maxFd = aThreadInfo.iLocalConnection->GetFd();
		
		if ( dataConnection->GetFd() > (int)maxFd)
		  maxFd = dataConnection->GetFd();	
		
		if(select(maxFd + 1, &dataSet, NULL, NULL, NULL) ==-1){
		  ERR("CGPRSPlugin::ClientDataThread : select failed");
		  dataConnection->Disconnect();
		  aThreadInfo.iLocalConnection->Disconnect();
		  delete aThreadInfo.iLocalConnection;
		  delete dataConnection;
		  return;
		}
		
		for(unsigned int i=0;i<=maxFd;i++){
		  if (FD_ISSET(i, &dataSet))
		    {
		      if(i==(unsigned int)dataConnection->GetFd())
			{
			  unsigned int length = 0;
			  
			  if(!TestConnection(dataConnection->GetFd(), sizeof(length)))
			    {
			      ERR("CGPRSPlugin::ClientDataThread : TestConnection failed");
			      dataConnection->Disconnect();
			      aThreadInfo.iLocalConnection->Disconnect();
			      delete aThreadInfo.iLocalConnection;
			      delete dataConnection;
			      return;
			    }
			  
			  if(recv(dataConnection->GetFd(), &length, sizeof(length), MSG_WAITALL) <=0) {
			    ERR("CGPRSPlugin::ClientDataThread : DataConnection Socket was closed while receiving data length");
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete aThreadInfo.iLocalConnection;
			    delete dataConnection;
			    return;
			  }
			  
			  length = ntohl(length);
			  
			  if(!TestConnection(dataConnection->GetFd(), length))
			    {
			      ERR("CGPRSPlugin::ClientDataThread : TestConnection failed");
			      dataConnection->Disconnect();
			      aThreadInfo.iLocalConnection->Disconnect();
			      delete aThreadInfo.iLocalConnection;
			      delete dataConnection;
			      return;
			    }
			  
			  char *buffer = new char[length];
			  memset(buffer, 0, length);
			  
			  if(recv(dataConnection->GetFd(), buffer, length, MSG_WAITALL) <= 0){
			    ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving data");
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete aThreadInfo.iLocalConnection;
			    delete dataConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  if(write(aThreadInfo.iLocalConnection->GetFd(), buffer, length) <= 0){
			    ERR("CGPRSPlugin::ClientDataThread() : sending nameBuf failed");
			    delete[] buffer;
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete dataConnection;
			    delete aThreadInfo.iLocalConnection;
			    return;
			  }
			  
			  delete[] buffer;
			}
		      else
			{
			  unsigned int length = 0;
			  if(read(aThreadInfo.iLocalConnection->GetFd(), &length, sizeof(length)) <= 0){
			    ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving data length");
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete dataConnection;
			    delete aThreadInfo.iLocalConnection;
			    return;
			  }

			  char *buffer = new char[length];
			  memset(buffer, 0, length);
			  if(read(aThreadInfo.iLocalConnection->GetFd(), buffer, length) <= 0){
			    ERR("CGPRSPlugin::ClientDataThread : LocalSocket was closed while receiving data");
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete dataConnection;
			    delete aThreadInfo.iLocalConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  length = htonl(length);
			  
			  if(write(dataConnection->GetFd(), &length, sizeof(length)) <= 0){
			    ERR("CGPRSPlugin::ClientDataThread() : sending length failed");
			    delete[] buffer;
			    dataConnection->Disconnect();
			    aThreadInfo.iLocalConnection->Disconnect();
			    delete aThreadInfo.iLocalConnection;
			    delete dataConnection;
			    return;
			  } 
			  
			  length = ntohl(length);
			  
			  if(write(dataConnection->GetFd(), buffer, length) <= 0) {
			    ERR("CGPRSPlugin::ClientDataThread() : sending data failed");
			    delete[] buffer;
			    aThreadInfo.iLocalConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete aThreadInfo.iLocalConnection;
			    return;
			  }
			
			  delete[] buffer;
			}//else
		    } //if (FD_ISSET(i, &dataSet))
		}//for(unsigned int i=0;i<=maxFd;i++){
	      }// while(iInSdp)
	    break;
	  }//case(2)
	}// switch(state)
    }//while(iInSdp)
  dataConnection->Disconnect();
  aThreadInfo.iLocalConnection->Disconnect();
  delete dataConnection;
  delete aThreadInfo.iLocalConnection;
}

/**
 * @memo Starts ServerData -thread.
 * @doc Starts ServerData -thread. This ServerData -thread creates a new Data Connection
 * to the GPRS Gateway. ClientData -thread negoatiates a new Data Connection between 
 * GPRS Gateway and target device and starts to route data between Data Connection
 * from the GPRS Gateway to the local service using local socket connection.
 *
 * @param aOriginalId Id of the remote device which initiated connection
 * @param aServiceName Name of the local service which is wanted to be connected by a remote device
 * @param aCommand Command from remote device (for example PH_DIRECT_DATA)
 * @return true if succesful
 */
bool CGPRSPlugin::StartServerDataThread(unsigned int aOriginalId, const std::string &aServiceName, unsigned int aCommand)
{
  pthread_t tempThread;
  CGPRSThread *tempThreadInfo = new CGPRSThread;
  tempThreadInfo->iInstance = this;
  tempThreadInfo->iThread = tempThread;
  tempThreadInfo->iCommand = aCommand;
  tempThreadInfo->iServiceName = aServiceName;
  tempThreadInfo->iTargetId = aOriginalId;
  tempThreadInfo->iLocalConnection = NULL;

  if (pthread_create(&tempThread, NULL, *CGPRSPlugin::ServerDataThreadStarter, (CGPRSThread *)tempThreadInfo) != 0) {
    ERR("CGPRSPlugin::StartServerDataThread() : failed to create the advertising thread");
    delete tempThreadInfo;
    return false;
  }
  pthread_detach(tempThread);
   
  return true;
}

/**
 * @memo Starts a new ServerData -thread.
 * @doc Starts a new ServerData -thread.
 *
 * @param aArguments Arguments for thread
 * @return true if succesful
 */
void *CGPRSPlugin::ServerDataThreadStarter(void *aArguments)
{
  CGPRSThread *tempThreadInfo = (CGPRSThread *)aArguments;
  CGPRSPlugin *fake = (CGPRSPlugin *)tempThreadInfo->iInstance;
  fake->ServerDataThread((*tempThreadInfo));
  delete tempThreadInfo;
  return NULL;
}

/**
 * @memo Implementation of ServerData -thread.
 * @doc Implementation of ServerData -thread. This thread
 * creates a new Data Connection to GPRS gateway and to local connection
 * to local service. After this, thread starts to route data betweeen
 * these connections.
 *
 * @param aThreadInfo Reference to thread information
 * @return true if succesful
 */
void CGPRSPlugin::ServerDataThread(CGPRSThread& aThreadInfo)
{
  fd_set dataSet;
  unsigned int maxFd = 0, state = 0,  targetPid = 0;
  u_int8_t command = 0;

  CTCPConnection *dataConnection = new CTCPConnection("NULL", "GPRS");
  CGPRSConnection *localServiceConnection = new CGPRSConnection;

  while(iInSdp) {  

    switch(state)
      {
      case 0:
	{
	  //connect to GPRS-gateway
	  if(!dataConnection->Connect(iGateway, GATEWAY+1)){
	    ERR("CGPRSPlugin::DataThread : failed to connect");
	    delete dataConnection;
	    delete localServiceConnection;
	    return;
	  }
	  
	  CGPRSSignal tempDataSignal;
	  tempDataSignal.iMessageType = REPCONNECT;
	  tempDataSignal.iOriginalId = iId;
	  tempDataSignal.iData = aThreadInfo.iTargetId;
	  tempDataSignal.iData2 = 0;
	  tempDataSignal.iServiceName = aThreadInfo.iServiceName;
	  
	  if(!EncapsulateSignal(dataConnection->GetFd(), tempDataSignal))
	    {
	      ERR("CGPRSPlugin::ServerDataThread() : Encapsulate REPCONNECT failed");
	      dataConnection->Disconnect();
	      delete dataConnection;
	      delete localServiceConnection;
	      return;
	    }
	  state = 2;
	  break;
	}

      case 2:
	{
	  CService* tempService;
	  
	  if(!GetLocalServicesL(tempService, aThreadInfo.iServiceName))
	    {
	      ERR("CGPRSPlugin::ServerDataThread() : GetLocalServices failed");
	      dataConnection->Disconnect();
	      delete dataConnection;
	      delete localServiceConnection;
	      return;
	    }

	  targetPid = tempService->GetPort();
	  delete tempService;

	  //open connection to local service
	  std::ostringstream addr;
	  addr << "/tmp/phGPRS." << targetPid;
	  
	  if(!localServiceConnection->Connect("toService", targetPid))
 	    {
	      ERR("CGPRSPlugin::ServerDataThread() : Connection to localServiceConnection failed");
	      dataConnection->Disconnect();
	      delete dataConnection;
	      delete localServiceConnection;
	      return;
	    }
			    
	  //send command 
	  command = aThreadInfo.iCommand;
	  if(write(localServiceConnection->GetFd(), &command, sizeof(command)) <= 0){
	    ERR("CGPRSPlugin::ServerDataThread() : sending command failed");
	    localServiceConnection->Disconnect();
	    dataConnection->Disconnect();
	    delete dataConnection;
	    delete localServiceConnection;
	    return;
	  }
			    
	  //send nameBuf to engine
	  if(write(localServiceConnection->GetFd(), aThreadInfo.iServiceName.c_str(), aThreadInfo.iServiceName.length()) <= 0){
	    ERR("CGPRSPlugin::ServerDataThread() : sending nameBuf failed");
	    localServiceConnection->Disconnect();
	    dataConnection->Disconnect();
	    delete dataConnection;
	    delete localServiceConnection;
	    return;
	  }
			   		   
	  //receive reply from engine
	  int length = 0;
	  if(read(localServiceConnection->GetFd(), &length, sizeof(length)) <= 0){
	    ERR("CGPRSPlugin::ServerDataThread() : receiving command failed");
	    localServiceConnection->Disconnect();
	    dataConnection->Disconnect();
	    delete dataConnection;
	    delete localServiceConnection;
	    return;
	  }

	  // if(localServiceConnection->Read(&command, sizeof(command)) <= 0){
	  if(read(localServiceConnection->GetFd(), &command, sizeof(command)) <= 0){
	    ERR("CGPRSPlugin::ServerDataThread() : receiving command failed");
	    localServiceConnection->Disconnect();
	    dataConnection->Disconnect();
	    delete dataConnection;
	    delete localServiceConnection;
	    return;
	  }
			    
	  if(command == PH_OK)
	    DBG("CGPRSPlugin::ServerDataThread() : PH_OK");
	  
	  if(command == PH_NO_CALLBACK)
	    {
	      ERR("CGPRSPlugin::ServerDataThread() : PH_NO_CALLBACK");
	       localServiceConnection->Disconnect();
	       dataConnection->Disconnect();
	       delete dataConnection;
	       delete localServiceConnection;
	       return;
	    }
	  
	  state  = 3;
	  break;
	}

      case 3:
	{
	  while(iInSdp)
	      {
		FD_ZERO(&dataSet);
		FD_SET(dataConnection->GetFd(), &dataSet);
		FD_SET(localServiceConnection->GetFd(), &dataSet);
		
		if (localServiceConnection->GetFd() > (int)maxFd)
		  maxFd = localServiceConnection->GetFd();
		
		if (dataConnection->GetFd() > (int)maxFd)
		  maxFd = dataConnection->GetFd();
		
		if(select(maxFd + 1, &dataSet, NULL, NULL, NULL) ==-1){
		  ERR("CGPRSPlugin::ServerDataThread : select failed");
		  localServiceConnection->Disconnect();
		  dataConnection->Disconnect();
		  delete dataConnection;
		  delete localServiceConnection;
		  return;
		}
		
		for(unsigned int i=0;i<=maxFd;i++){
		  if (FD_ISSET(i, &dataSet))
		    {
		      if(i==(unsigned int)dataConnection->GetFd())
			{
			  unsigned int length = 0;
			  /*
			  if(!TestConnection(dataConnection->GetFd(), sizeof(length)))
			    {
			      ERR("CGPRSPlugin::ServerDataThread : TestConnection failed");
			      localServiceConnection->Disconnect();
			      dataConnection->Disconnect();
			      delete dataConnection;
			      delete localServiceConnection;
			      return;
			    }
			  */
			  if(recv(dataConnection->GetFd(), &length, sizeof(length), MSG_WAITALL) <= 0) {
			    ERR("CGPRSPlugin::ServerDataThread : DataConnection Socket was closed while receiving data length");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    return;
			  }
			  
			  length = ntohl(length);

			  /*
			  if(!TestConnection(dataConnection->GetFd(), length))
			    {
			      ERR("CGPRSPlugin::ServerDataThread : TestConnection failed");
			      localServiceConnection->Disconnect();
			      dataConnection->Disconnect();
			      delete dataConnection;
			      delete localServiceConnection;
			      return;
			    }
			  */

			  char *buffer = new char[length];
			  memset(buffer, 0, length);
			  
			  if(recv(dataConnection->GetFd(), buffer, length, MSG_WAITALL) <= 0) {
			    ERR("CGPRSPlugin::ServerDataThread : DataConnection Socket was closed while receiving data");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  if(write(localServiceConnection->GetFd(), buffer, length) <= 0){
			    ERR("CGPRSPlugin::ServerDataThread() : sending nameBuf failed");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  delete[] buffer;
			}
		      else
			{
			  unsigned int length = 0;
			  if(read(localServiceConnection->GetFd(), &length, sizeof(length)) <= 0 ){
			    ERR("CGPRSPlugin::ServerDataThread : LocalSocket was closed while receiving data length");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    return;
			  }
			  
			  char *buffer = new char[length];
			  memset(buffer, 0, length);
			  
			  if(read(localServiceConnection->GetFd(), buffer, length) <= 0 ){
			    ERR("CGPRSPlugin::ServerDataThread : LocalSocket was closed while receiving data");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  length = htonl(length);
			  
			  if(write(dataConnection->GetFd(), &length, sizeof(length)) <= 0 ){
			    ERR("CGPRSPlugin::ServerDataThread() : sending length failed");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    delete[] buffer;
			    return;
			  } 
			  
			  length = ntohl(length);
			  
			  if(write(dataConnection->GetFd(), buffer, length) <= 0) {
			    ERR("CGPRSPlugin::ServerDataThread() : sending data failed");
			    localServiceConnection->Disconnect();
			    dataConnection->Disconnect();
			    delete dataConnection;
			    delete localServiceConnection;
			    delete[] buffer;
			    return;
			  }
			  
			  delete[] buffer;
			}//else
		    } //if (FD_ISSET(i, &dataSet))
		}//for(unsigned int i=0;i<=maxFd;i++){
	      }// while(iInSdp)
	  break;
	}//case(3)
      }// switch(state)
  }// while(iInSdp)
  localServiceConnection->Disconnect();
  dataConnection->Disconnect();
  delete dataConnection;
  delete localServiceConnection; 
}

void CGPRSPlugin::RemoveListeners()
{
	std::list<MAbstractListener*>::iterator iter;
	// Go through all listeners and disconnect, delete object and erase from list
	for(iter = iListenerList.begin(); iter != iListenerList.end();)
	{
		(*iter)->Disconnect();
		delete *iter;
		iter = iListenerList.erase(iter);
	}
}

void CGPRSPlugin::SetAdapter(const std::string& aInterface, int aId)
{
	DBG("CGPRSPlugin::SetAdapter: called with parameters: interface = %s, id = %d",aInterface.c_str(), aId);
}

const std::string& CGPRSPlugin::GetAdapter()
{
	return iNDevice;
}

static CGPRSPlugin gprsplugin;






  



