/** This file is part of PeerHood.
*
*   PeerHood is free software: you can redistribute it and/or modify
*   it under the terms of the GNU Lesser General Public License 
*   version 2 as published by the Free Software Foundation.
*
*   PeerHood is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU Lesser General Public License for more details.
*
*   You should have received a copy of the GNU Lesser General Public
*   License along with PeerHood. If not, see <http://www.gnu.org/licenses/>.
*/

// Emacs, hail the mighty -*-C++-*-

/**
 * Copyright 2003 LUT. .
 *
 * @name AbstractPlugin.h
 * @memo Common abstract interface for all PeerHood plugins. Every plugin must
 * implement this interface.
 *
 * @version 0.1
 * date     08.04.2003
 * change   08.04.2003
 */

#ifndef __ABSTRACTPLUGIN_H__
#define __ABSTRACTPLUGIN_H__

#include <string>
#include <map>
#include <syslog.h>
#include "AbstractStateConverter.h"

#define ERR(format, msg...) syslog(LOG_ERR, "ERROR: " format "\n" , ## msg)

#ifdef PH_DEBUG
#define DBG(format, msg...) syslog(LOG_DEBUG, format "\n" , ## msg)
#else
#define DBG(A...)
#endif

/**
 * @memo Common abstract interface for all PeerHood plugins. Every plugin must
 * implement this interface.
 * @doc Common abstract interface for all PeerHood plugins. Every real plugin
 * must implement this interface so that the PeerHood Daemon can use it without
 * any further knowledge of plugin's special characteristics. All plugins are 
 * compiled as dynamically linkable libraries and loaded as a part of the
 * daemon's startup procedure from the directory pointed by the PH_PLUGIN_DIR 
 * environment variable. Every plugin must meet the following requirements:<ol>
 * <li>Plugin's name must end with the 'plugin.so' suffix.</li>
 * <li>Every plugin must have a unique prototype id (a string).</li>
 * <li>A static instance of the plugin must be declared at the end of its
 * implementation.</li>
 * <li>Plugin's constructor must register it to daemon's map: 
 * <code>pluginMap["plugin_id"] = this</code>.</li>
 * <li>Plugin must be able to deliver a list of available services to another
 * PeerHood device. This is done by calling the <code>
 * CDaemon::SendServiceList(MAbstractPlugin*)</code> function</li></ol>
 *
 * Every plugin implements MAbstractStateConverter interface. This allows the
 * usage of this class by system state listeners via simple interface.
 * 
 * @see CDaemon
 * @see MAbstractStateConverter
 */
class MAbstractPlugin : public MAbstractStateConverter
{
 public:
  /**
   * @memo Default destructor.
   * @doc Default destructor. Currently this does nothing but is here because
   * base classes without a destructor are <b><i>evil</i></b>.
   *
   * @return none
   */
  virtual ~MAbstractPlugin() {};

  /** 
   * @memo Starts to advert the PeerHood service by using plugin's networking
   * technology.
   * @doc Starts to advert the PeerHood service by using plugin's networking
   * technology. There's no standard advertising method. Instead, every plugin
   * should use the means that suit for the networking technology in question.
   *
   * @return true if the adverting could be started
   */
  virtual bool Advert() = 0;

  /** 
   * @memo Ends the advertising of the PeerHood service.
   * @doc Ends the advertising of the PeerHood service. Although this function
   * can return also <i>false</i> the plugin should take care that the PeerHood
   * service is not advertised after this function call. In other words, the
   * advertising routine must be ended by force if otherwise impossible. The
   * return value is there to mainly inform the user.
   *
   * @return true if the operation was successfull
   */
  virtual bool Unadvert() = 0;

  /** 
   * @memo Starts the plugin's inquiry routine.
   * @doc Starts the plugin's inquiry routine. After this method is called the
   * plugin starts to monitor its environment and tries to locate nearby 
   * PeerHood-capable devices. Whenever a change in the neighborhood status is
   * detected the plugin will update the database located in the daemon.
   *
   * @return boolean telling if the plugin could be started
   */
  virtual bool Start() = 0;

  /** 
   * @memo Stops the plugin's inquiry routine.
   * @doc Stops the plugin's inquiry routine. After this call is handled the
   * plugin will stop monitoring the environment. This means that daemon's
   * device & service database will be deprecated as the time passes.
   *
   * @return none
   */
  virtual void Stop() = 0;

  /** 
   * @memo Returns plugin's unique id string.
   * @doc Returns plugin's unique id string. Every plugin must have a <b>unique
   * </b> id or disasters will follow.
   *
   * @return plugin's unique id string
   */
  virtual const std::string& GetName() = 0;
	
	/**
	 * @memo Updates the internal state of the plugin
	 * @doc Checks the state of the device used by the plugin via listener which is made
	 * specially for that plugin. 
	 */
	virtual void UpdateState() = 0;
	
	/**
	 * @memo Loads all listeners for plugin
	 * @doc Calls ListenerFactory to create listeners for this plugin
	 */
	virtual void LoadListeners() = 0;
};

#ifdef __PH_OLD_PLUGIN_STYLE__
extern std::map<const char*, MAbstractPlugin*> pluginMap;
#endif

#endif // __ABSTRACTPLUGIN_H__
