/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDJSON_H
#define MICROFEEDJSON_H

#include <sys/types.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedJson
 *
 * JavaScript Object Notation parser.
 *
 * @{ 
 */

typedef struct _MicrofeedJson MicrofeedJson;

typedef enum {
	MICROFEED_JSON_TYPE_NULL = 0,
	MICROFEED_JSON_TYPE_OBJECT,
	MICROFEED_JSON_TYPE_ARRAY,
	MICROFEED_JSON_TYPE_INTEGER,
	MICROFEED_JSON_TYPE_DECIMAL,
	MICROFEED_JSON_TYPE_STRING,
	MICROFEED_JSON_TYPE_BOOLEAN
} MicrofeedJsonType;

typedef int (*MicrofeedJsonCompareMembersFunction)(MicrofeedJson* json, unsigned int index1, unsigned int index2, void* user_data);

MicrofeedJson* microfeed_json_new_object(void);
MicrofeedJson* microfeed_json_new_array(void);
MicrofeedJson* microfeed_json_new_from_data(const char* data, size_t length);
void microfeed_json_free(MicrofeedJson* json);

int microfeed_json_is_array(MicrofeedJson* json);
MicrofeedJson* microfeed_json_get_parent(MicrofeedJson* json);
unsigned int microfeed_json_get_size(MicrofeedJson* json);
MicrofeedJsonType microfeed_json_get_type(MicrofeedJson* json, const char* name);
MicrofeedJsonType microfeed_json_get_type_by_index(MicrofeedJson* json, unsigned int index);
MicrofeedJsonType microfeed_json_get_type_by_path(MicrofeedJson* json, const char* name, ...);
const char* microfeed_json_get_name_by_index(MicrofeedJson* json, unsigned int index);
int microfeed_json_is_null(MicrofeedJson* json, const char* name);
int microfeed_json_is_null_by_index(MicrofeedJson* json, unsigned int index);
int microfeed_json_is_null_by_path(MicrofeedJson* json,const char* name1, ...);
MicrofeedJson* microfeed_json_get_object(MicrofeedJson* json, const char* name);
MicrofeedJson* microfeed_json_get_object_by_index(MicrofeedJson* json, unsigned int index);
MicrofeedJson* microfeed_json_get_object_by_path(MicrofeedJson* json, const char* name, ...);
MicrofeedJson* microfeed_json_get_array(MicrofeedJson* json, const char* name);
MicrofeedJson* microfeed_json_get_array_by_index(MicrofeedJson* json, unsigned int index);
MicrofeedJson* microfeed_json_get_array_by_path(MicrofeedJson* json, const char* name, ...);
long long int microfeed_json_get_integer(MicrofeedJson* json, const char* name);
long long int microfeed_json_get_integer_by_index(MicrofeedJson* json, unsigned int index);
long long int microfeed_json_get_integer_by_path(MicrofeedJson* json, const char* name, ...);
double microfeed_json_get_decimal(MicrofeedJson* json, const char* name);
double microfeed_json_get_decimal_by_index(MicrofeedJson* json, unsigned int index);
double microfeed_json_get_decimal_by_path(MicrofeedJson* json, const char* name, ...);
const char* microfeed_json_get_string(MicrofeedJson* json, const char* name);
const char* microfeed_json_get_string_by_index(MicrofeedJson* json, unsigned int index);
const char* microfeed_json_get_string_by_path(MicrofeedJson* json, const char* name, ...);
int microfeed_json_get_boolean(MicrofeedJson* json, const char* name);
int microfeed_json_get_boolean_by_index(MicrofeedJson* json, unsigned int index);
int microfeed_json_get_boolean_by_path(MicrofeedJson* json, const char* name, ...);
const char* microfeed_json_get_as_string(MicrofeedJson* json, const char* name);
const char* microfeed_json_get_as_string_by_index(MicrofeedJson* json, unsigned int index);
const char* microfeed_json_get_as_string_by_path(MicrofeedJson* json, const char* name, ...);
char* microfeed_json_to_string(MicrofeedJson* json, const char* name);
char* microfeed_json_to_string_by_index(MicrofeedJson* json, unsigned int index);
char* microfeed_json_to_string_by_path(MicrofeedJson* json, const char* name, ...);
void microfeed_json_set_null(MicrofeedJson* json, const char* name);
void microfeed_json_append_null(MicrofeedJson* json, MicrofeedJson* object);
void microfeed_json_set_object(MicrofeedJson* json, const char* name, MicrofeedJson* object);
void microfeed_json_append_object(MicrofeedJson* json, MicrofeedJson* object);
void microfeed_json_set_integer(MicrofeedJson* json, const char* name, long long int integer);
void microfeed_json_append_integer(MicrofeedJson* json, long long int integer);
void microfeed_json_set_decimal(MicrofeedJson* json, const char* name, double decimal);
void microfeed_json_append_decimal(MicrofeedJson* json, double decimal);
void microfeed_json_set_string(MicrofeedJson* json, const char* name, const char* string);
void microfeed_json_append_string(MicrofeedJson* json, const char* string);
void microfeed_json_set_boolean(MicrofeedJson* json, const char* name, int boolean);
void microfeed_json_append_boolean(MicrofeedJson* json, int boolean);
void microfeed_json_sort_array(MicrofeedJson* json, MicrofeedJsonCompareMembersFunction compare_members, void* user_data);
int microfeed_json_compare_members(MicrofeedJson* json, unsigned int index1, unsigned int index2);

/**
 * @}
 * @}
 */

#endif
